#version 150

uniform sampler2D NoiseSampler;
uniform sampler2D DepthSampler;
in vec2 texCoord;
in vec4 vPosition;
uniform vec2 OutSize;
uniform float STime;
uniform float BreakingProgress;
uniform vec3 CameraPosition;
uniform vec3 CameraFront;
uniform vec3 CameraLeft;
uniform vec3 CameraUp;
uniform float BlackHoleScale;
uniform mat4 InverseTransformMatrix;
uniform ivec4 ViewPort;
out vec4 fragColor;

const float PI = 3.1415927;
const float MAX_RAD = 300.0;
const float MAX_DIST = 600.0;

vec3 getWorldPos(float z) {
    vec3 ndc = vec3(((2.0 * gl_FragCoord.xy) - (2.0 * ViewPort.xy)) / (ViewPort.zw) - 1.0,
    (2.0 * z - gl_DepthRange.near - gl_DepthRange.far) / (gl_DepthRange.far - gl_DepthRange.near));
    vec4 clip = InverseTransformMatrix * vec4(ndc, 1.0);
    return clip.xyz / clip.w;
}

float intersectSphere(vec3 origin, vec3 dir, float r) {
    float b = dot(origin, dir);
    float c = dot(origin, origin) - r * r;
    float h = b * b - c;
    return (h < 0.0) ? -1.0 : -b - sqrt(h);
}

void main() {
    vec2 screenPos = gl_FragCoord.xy / OutSize.xy;
    float rawDepth = texture(DepthSampler, screenPos).x;
    vec3 worldPos = getWorldPos(rawDepth);
    float targetDist = length(worldPos - CameraPosition);

    if (rawDepth >= 1.0) targetDist = MAX_DIST * BlackHoleScale;

    vec2 uv = screenPos * 2.0 - 1.0;
    vec3 rayOrg = CameraPosition / BlackHoleScale;
    vec3 rayDir = normalize(CameraFront - CameraLeft * uv.x + CameraUp * uv.y);

    float travel = 0.0;
    float bound = intersectSphere(rayOrg, rayDir, MAX_RAD);

    if (length(rayOrg) > MAX_RAD) {
        if (bound < 0.0) discard;
        travel = bound;
    }

    vec3 pos = rayOrg + rayDir * travel;
    vec3 vel = rayDir;
    vec3 finalColor = vec3(0.0);
    float exists = 1.0;

    vec3 colCore = mix(vec3(0.44, 0.15, 0.49), vec3(1.0, 0.1, 0.0), BreakingProgress);
    vec3 colRim = mix(vec3(0.88, 0.815, 0.925), vec3(1.0, 0.4, 0.0), BreakingProgress);
    vec3 colGlow = mix(vec3(0.2, 0.05, 0.3), vec3(0.7, 0.1, 0.0), BreakingProgress);

    float totalDist = travel;
    for(int i = 0; i < 90; i++) {
        float centerDist = length(pos);
        float sqDist = dot(pos, pos);

        if (totalDist * BlackHoleScale > targetDist) break;
        if (totalDist > MAX_DIST) break;

        float gravityZone = smoothstep(0.0, 3.0, centerDist);
        float stepSize = 0.02 + gravityZone * 0.08;

        pos += vel * stepSize;
        totalDist += stepSize;

        vec3 toCenterDir = -pos * inversesqrt(sqDist + 0.00001);
        vel += toCenterDir * (0.09 / (sqDist + 0.01)) * stepSize;
        vel = normalize(vel);

        if (sqDist < 0.08) {
            exists = 0.0;
            break;
        }

        if (abs(pos.y) < 2.0) {
            float rad = length(pos.xz);

            float torusSDF = length(vec2(rad - 0.8, pos.y * 18.0)) - 0.98;
            float diskMask = smoothstep(0.0, 0.5, -torusSDF);

            if (diskMask > 0.01) {
                float ang = atan(pos.x, pos.z);
                vec2 map = vec2(rad, ang * (0.01 + (rad - 0.12) * 0.002) + STime * 0.005) * vec2(10.0, 20.0);

                float noise = texture(NoiseSampler, map * 0.1).r;
                float shine = (4.5 / (0.001 + (rad - 0.12) * 40.0));

                vec3 diskCol = mix(colRim, colCore, clamp(rad - 0.12, 0.0, 1.0)) * (noise + 0.3) * shine;
                finalColor += diskCol * diskMask * (stepSize * 65.0);
            }
        }

        finalColor += colGlow * (0.0035 / (sqDist + 0.015)) * (stepSize * 55.0);
    }

    float bright = max(finalColor.r, max(finalColor.g, finalColor.b));
    float alpha = clamp(bright * 1.5 + (1.0 - exists), 0.0, 1.0);

    fragColor = vec4(finalColor, alpha);
}