/*
 * Decompiled with CFR 0.152.
 */
package net.levelscraft7.cobblecapsule.util;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import net.levelscraft7.cobblecapsule.CobbleCapsule;
import net.minecraft.core.Holder;
import net.minecraft.core.component.DataComponents;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.packs.resources.ResourceManager;
import net.minecraft.server.packs.resources.SimpleJsonResourceReloadListener;
import net.minecraft.util.GsonHelper;
import net.minecraft.util.RandomSource;
import net.minecraft.util.profiling.ProfilerFiller;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.alchemy.Potion;
import net.minecraft.world.item.alchemy.PotionContents;
import net.minecraft.world.level.ItemLike;
import net.neoforged.fml.ModList;

public class CapsuleLootManager
extends SimpleJsonResourceReloadListener {
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String LOOT_DIRECTORY = "loot_tables";
    private static final String BONUS_DIRECTORY = "capsule_bonus";
    public static final CapsuleLootManager INSTANCE = new CapsuleLootManager();
    private final Map<ResourceLocation, CapsuleLootTable> lootTables = new HashMap<ResourceLocation, CapsuleLootTable>();

    private CapsuleLootManager() {
        super(GSON, LOOT_DIRECTORY);
    }

    protected void apply(Map<ResourceLocation, JsonElement> object, ResourceManager resourceManager, ProfilerFiller profiler) {
        this.lootTables.clear();
        HashMap<String, CapsuleLootTableBuilder> builders = new HashMap<String, CapsuleLootTableBuilder>();
        object.forEach((id, element) -> {
            if (!id.getPath().startsWith("capsules/")) {
                return;
            }
            List<CapsuleLootContribution> contributions = this.parseTable((ResourceLocation)id, (JsonElement)element);
            if (!contributions.isEmpty()) {
                CapsuleLootTableBuilder builder = builders.computeIfAbsent(id.getPath(), ignored -> new CapsuleLootTableBuilder());
                for (CapsuleLootContribution contribution : contributions) {
                    builder.addContribution(id.getPath(), (ResourceLocation)id, contribution);
                }
            }
        });
        this.loadBonusTables(resourceManager, builders);
        builders.forEach((path, builder) -> {
            CapsuleLootTable table = builder.build((String)path);
            if (table != null) {
                ResourceLocation tableId = ResourceLocation.fromNamespaceAndPath((String)"cobblecapsule", (String)path);
                this.lootTables.put(tableId, table);
                builder.logSummary(tableId, table.entries().size());
            }
        });
        CobbleCapsule.LOGGER.info("Loaded {} capsule loot tables", (Object)this.lootTables.size());
    }

    private void loadBonusTables(ResourceManager resourceManager, Map<String, CapsuleLootTableBuilder> builders) {
        String prefix = "capsule_bonus/";
        resourceManager.listResources(BONUS_DIRECTORY, location -> location.getPath().endsWith(".json")).forEach((location, resource) -> {
            String path = location.getPath();
            if (!path.startsWith(prefix)) {
                return;
            }
            String relativePath = path.substring(prefix.length());
            if (relativePath.endsWith(".json")) {
                relativePath = relativePath.substring(0, relativePath.length() - 5);
            }
            if (!relativePath.startsWith("capsules/")) {
                return;
            }
            ResourceLocation tableId = ResourceLocation.fromNamespaceAndPath((String)location.getNamespace(), (String)relativePath);
            try (BufferedReader reader = resource.openAsReader();){
                JsonElement element = JsonParser.parseReader((Reader)reader);
                List<CapsuleLootContribution> contributions = this.parseTable(tableId, element);
                if (!contributions.isEmpty()) {
                    CapsuleLootTableBuilder builder = builders.computeIfAbsent(relativePath, ignored -> new CapsuleLootTableBuilder());
                    for (CapsuleLootContribution contribution : contributions) {
                        builder.addContribution(relativePath, tableId, contribution);
                    }
                }
            }
            catch (JsonParseException exception) {
                CobbleCapsule.LOGGER.error("Failed to parse capsule bonus loot table {}", (Object)tableId, (Object)exception);
            }
            catch (IOException exception) {
                CobbleCapsule.LOGGER.error("I/O error while reading capsule bonus loot table {}", (Object)tableId, (Object)exception);
            }
        });
    }

    private List<CapsuleLootContribution> parseTable(ResourceLocation id, JsonElement element) {
        JsonObject obj = GsonHelper.convertToJsonObject((JsonElement)element, (String)"table");
        if (obj.has("entries") && obj.get("entries").isJsonArray()) {
            JsonArray array = obj.getAsJsonArray("entries");
            ArrayList<CapsuleLootContribution> contributions = new ArrayList<CapsuleLootContribution>();
            for (int i = 0; i < array.size(); ++i) {
                JsonElement entryElement = array.get(i);
                if (!entryElement.isJsonObject()) {
                    CobbleCapsule.LOGGER.warn("Skipping malformed entry {} in capsule loot table {}", (Object)i, (Object)id);
                    continue;
                }
                CapsuleLootContribution contribution = this.parseContribution(id, "entry " + (i + 1), entryElement.getAsJsonObject());
                if (contribution == null) continue;
                contributions.add(contribution);
            }
            return contributions;
        }
        CapsuleLootContribution contribution = this.parseContribution(id, null, obj);
        return contribution != null ? List.of(contribution) : List.of();
    }

    private CapsuleLootContribution parseContribution(ResourceLocation tableId, String entryLabel, JsonObject obj) {
        String context = entryLabel == null ? tableId.toString() : String.valueOf(tableId) + " (" + entryLabel + ")";
        ModRequirementResult entryRequirement = CapsuleLootManager.evaluateModRequirement(obj);
        if (!entryRequirement.met()) {
            CobbleCapsule.LOGGER.info("Skipping capsule loot contribution {} because required mod(s) {} are not loaded", (Object)context, (Object)String.join((CharSequence)", ", entryRequirement.missingMods()));
            return null;
        }
        Integer minRolls = null;
        Integer maxRolls = null;
        Boolean allowDuplicates = null;
        int definedEntries = 0;
        int addedEntries = 0;
        HashMap<String, Integer> missingMods = new HashMap<String, Integer>();
        if (obj.has("rolls") && obj.get("rolls").isJsonObject()) {
            JsonObject rollsObject = obj.getAsJsonObject("rolls");
            minRolls = GsonHelper.getAsInt((JsonObject)rollsObject, (String)"min", (int)1);
            maxRolls = GsonHelper.getAsInt((JsonObject)rollsObject, (String)"max", (int)minRolls);
        }
        if (obj.has("allow_duplicates")) {
            allowDuplicates = GsonHelper.getAsBoolean((JsonObject)obj, (String)"allow_duplicates");
        }
        if (!obj.has("items") || !obj.get("items").isJsonArray()) {
            CobbleCapsule.LOGGER.warn("Capsule loot table {} is missing an items array", (Object)context);
            return null;
        }
        JsonArray itemsArray = obj.getAsJsonArray("items");
        ArrayList<CapsuleLootEntry> entries = new ArrayList<CapsuleLootEntry>();
        for (JsonElement itemElement : itemsArray) {
            JsonObject itemObject = GsonHelper.convertToJsonObject((JsonElement)itemElement, (String)"item");
            ++definedEntries;
            if (!itemObject.has("id")) {
                CobbleCapsule.LOGGER.warn("Skipping capsule loot entry without an id in table {}", (Object)context);
                continue;
            }
            String name = GsonHelper.getAsString((JsonObject)itemObject, (String)"id");
            ResourceLocation itemId = ResourceLocation.tryParse((String)name);
            if (itemId == null) {
                CobbleCapsule.LOGGER.warn("Invalid item id '{}' in capsule loot table {}", (Object)name, (Object)context);
                continue;
            }
            ModRequirementResult modRequirement = CapsuleLootManager.evaluateModRequirement(itemObject);
            if (!modRequirement.met()) {
                modRequirement.missingMods().forEach(mod -> missingMods.merge((String)mod, 1, Integer::sum));
                CobbleCapsule.LOGGER.info("Skipping capsule loot entry {} from table {} because required mod(s) {} are not loaded", new Object[]{itemId, context, String.join((CharSequence)", ", modRequirement.missingMods())});
                continue;
            }
            Optional itemOptional = BuiltInRegistries.ITEM.getOptional(itemId);
            if (itemOptional.isEmpty()) {
                CobbleCapsule.LOGGER.warn("Skipping unknown item {} in capsule loot table {}", (Object)itemId, (Object)context);
                continue;
            }
            int weight = Math.max(0, GsonHelper.getAsInt((JsonObject)itemObject, (String)"weight", (int)1));
            StackRange stackRange = CapsuleLootManager.parseStackRange(itemObject);
            Holder<Potion> potion = CapsuleLootManager.parsePotion(itemObject, itemId);
            if (weight <= 0 || stackRange.max() <= 0) continue;
            entries.add(new CapsuleLootEntry((Item)itemOptional.get(), weight, stackRange, potion));
            ++addedEntries;
        }
        if (entries.isEmpty()) {
            CobbleCapsule.LOGGER.warn("Capsule loot table {} does not contain any valid entries", (Object)context);
            return null;
        }
        return new CapsuleLootContribution(minRolls, maxRolls, allowDuplicates, entries, new ContributionMetadata(definedEntries, addedEntries, missingMods));
    }

    private static ModRequirementResult evaluateModRequirement(JsonObject itemObject) {
        JsonElement element;
        if (itemObject.has("required_mods") && (element = itemObject.get("required_mods")).isJsonArray()) {
            HashSet<String> mods = new HashSet<String>();
            for (JsonElement modElement : element.getAsJsonArray()) {
                String modId;
                if (!modElement.isJsonPrimitive() || !modElement.getAsJsonPrimitive().isString() || (modId = modElement.getAsString().trim()).isEmpty()) continue;
                mods.add(modId);
            }
            if (mods.isEmpty()) {
                return ModRequirementResult.fulfilled();
            }
            List<String> missing = mods.stream().filter(mod -> !ModList.get().isLoaded(mod)).sorted().toList();
            return missing.isEmpty() ? ModRequirementResult.fulfilled() : new ModRequirementResult(false, missing);
        }
        if (!itemObject.has("required_mod")) {
            return ModRequirementResult.fulfilled();
        }
        String modId = GsonHelper.getAsString((JsonObject)itemObject, (String)"required_mod").trim();
        if (modId.isEmpty()) {
            return ModRequirementResult.fulfilled();
        }
        return ModList.get().isLoaded(modId) ? ModRequirementResult.fulfilled() : new ModRequirementResult(false, List.of(modId));
    }

    private static StackRange parseStackRange(JsonObject itemObject) {
        if (itemObject.has("stack")) {
            JsonElement stackElement = itemObject.get("stack");
            if (stackElement.isJsonArray()) {
                JsonArray stackArray = stackElement.getAsJsonArray();
                if (stackArray.size() >= 2) {
                    int min = stackArray.get(0).getAsInt();
                    int max = stackArray.get(1).getAsInt();
                    return new StackRange(min, max);
                }
            } else if (stackElement.isJsonObject()) {
                JsonObject stackObject = stackElement.getAsJsonObject();
                int min = GsonHelper.getAsInt((JsonObject)stackObject, (String)"min", (int)1);
                int max = GsonHelper.getAsInt((JsonObject)stackObject, (String)"max", (int)min);
                return new StackRange(min, max);
            }
        }
        return new StackRange(1, 1);
    }

    public static List<ItemStack> generateLoot(ServerLevel level, ResourceLocation tableId) {
        CapsuleLootTable table = CapsuleLootManager.INSTANCE.lootTables.get(tableId);
        if (table == null) {
            CobbleCapsule.LOGGER.warn("Unknown capsule loot table {}", (Object)tableId);
            return List.of();
        }
        return table.generate(level.random);
    }

    public static Optional<CapsuleLootDisplay> getDisplay(ResourceLocation tableId) {
        CapsuleLootTable table = CapsuleLootManager.INSTANCE.lootTables.get(tableId);
        if (table == null) {
            return Optional.empty();
        }
        ArrayList<CapsuleLootDisplayEntry> entries = new ArrayList<CapsuleLootDisplayEntry>();
        for (CapsuleLootEntry entry : table.entries) {
            int min = Math.max(1, entry.stackRange.min());
            int max = Math.max(min, entry.stackRange.max());
            ItemStack stack = new ItemStack((ItemLike)entry.item());
            stack.setCount(Math.min(64, min));
            if (entry.potion() != null) {
                stack.set(DataComponents.POTION_CONTENTS, (Object)PotionContents.EMPTY.withPotion(entry.potion()));
            }
            entries.add(new CapsuleLootDisplayEntry(stack, entry.weight(), min, max));
        }
        return Optional.of(new CapsuleLootDisplay(tableId, table.minRolls, table.maxRolls, table.allowDuplicates, List.copyOf(entries)));
    }

    private static Holder<Potion> parsePotion(JsonObject itemObject, ResourceLocation itemId) {
        if (!itemObject.has("potion")) {
            return null;
        }
        String potionName = GsonHelper.getAsString((JsonObject)itemObject, (String)"potion");
        ResourceLocation potionId = ResourceLocation.tryParse((String)potionName);
        if (potionId == null) {
            CobbleCapsule.LOGGER.warn("Invalid potion id '{}' in capsule loot table entry for {}", (Object)potionName, (Object)itemId);
            return null;
        }
        Optional potionOptional = BuiltInRegistries.POTION.getHolder(potionId);
        if (potionOptional.isEmpty()) {
            CobbleCapsule.LOGGER.warn("Unknown potion {} referenced in capsule loot table entry for {}", (Object)potionId, (Object)itemId);
            return null;
        }
        return (Holder)potionOptional.get();
    }

    private record CapsuleLootContribution(Integer minRolls, Integer maxRolls, Boolean allowDuplicates, List<CapsuleLootEntry> entries, ContributionMetadata metadata) {
    }

    private record ModRequirementResult(boolean met, List<String> missingMods) {
        private static ModRequirementResult fulfilled() {
            return new ModRequirementResult(true, List.of());
        }
    }

    private record StackRange(int min, int max) {
    }

    private record CapsuleLootEntry(Item item, int weight, StackRange stackRange, Holder<Potion> potion) {
        public ItemStack createStack(RandomSource randomSource) {
            int max;
            int min = Math.max(1, this.stackRange.min());
            int count = min >= (max = Math.max(min, this.stackRange.max())) ? min : randomSource.nextInt(max - min + 1) + min;
            ItemStack stack = new ItemStack((ItemLike)this.item, count);
            if (this.potion != null) {
                stack.set(DataComponents.POTION_CONTENTS, (Object)PotionContents.EMPTY.withPotion(this.potion));
            }
            return stack;
        }
    }

    private record ContributionMetadata(int definedEntries, int addedEntries, Map<String, Integer> missingMods) {
    }

    private record CapsuleLootTable(int minRolls, int maxRolls, boolean allowDuplicates, List<CapsuleLootEntry> entries) {
        public List<ItemStack> generate(RandomSource randomSource) {
            CapsuleLootEntry entry;
            ArrayList<ItemStack> result = new ArrayList<ItemStack>();
            ArrayList<CapsuleLootEntry> available = new ArrayList<CapsuleLootEntry>(this.entries);
            int rolls = this.minRolls >= this.maxRolls ? Math.max(0, this.minRolls) : randomSource.nextInt(Math.max(1, this.maxRolls - this.minRolls + 1)) + Math.max(0, this.minRolls);
            for (int i = 0; i < rolls && !available.isEmpty() && (entry = this.pickRandom(available, randomSource)) != null; ++i) {
                result.add(entry.createStack(randomSource));
                if (this.allowDuplicates) continue;
                available.remove(entry);
            }
            return result;
        }

        private CapsuleLootEntry pickRandom(List<CapsuleLootEntry> pool, RandomSource randomSource) {
            int totalWeight = pool.stream().mapToInt(CapsuleLootEntry::weight).sum();
            if (totalWeight <= 0) {
                return null;
            }
            int target = randomSource.nextInt(totalWeight);
            int cumulative = 0;
            for (CapsuleLootEntry entry : pool) {
                if (target >= (cumulative += entry.weight())) continue;
                return entry;
            }
            return pool.getLast();
        }
    }

    public record CapsuleLootDisplayEntry(ItemStack stack, int weight, int minCount, int maxCount) {
    }

    public record CapsuleLootDisplay(ResourceLocation tableId, int minRolls, int maxRolls, boolean allowDuplicates, List<CapsuleLootDisplayEntry> entries) {
    }

    private static final class CapsuleLootTableBuilder {
        private Integer minRolls;
        private Integer maxRolls;
        private Boolean allowDuplicates;
        private final List<CapsuleLootEntry> entries = new ArrayList<CapsuleLootEntry>();
        private final Map<ResourceLocation, ContributionMetadata> contributionMetadata = new LinkedHashMap<ResourceLocation, ContributionMetadata>();

        private CapsuleLootTableBuilder() {
        }

        void addContribution(String path, ResourceLocation sourceId, CapsuleLootContribution contribution) {
            if (contribution.minRolls != null) {
                if (this.minRolls == null) {
                    this.minRolls = contribution.minRolls;
                } else if (!this.minRolls.equals(contribution.minRolls)) {
                    this.minRolls = Math.max(this.minRolls, contribution.minRolls);
                }
            }
            if (contribution.maxRolls != null) {
                if (this.maxRolls == null) {
                    this.maxRolls = contribution.maxRolls;
                } else if (!this.maxRolls.equals(contribution.maxRolls)) {
                    this.maxRolls = Math.max(this.maxRolls, contribution.maxRolls);
                }
            }
            if (contribution.allowDuplicates != null) {
                if (this.allowDuplicates == null) {
                    this.allowDuplicates = contribution.allowDuplicates;
                } else if (!this.allowDuplicates.equals(contribution.allowDuplicates)) {
                    this.allowDuplicates = this.allowDuplicates != false && contribution.allowDuplicates != false;
                }
            }
            this.entries.addAll(contribution.entries);
            this.contributionMetadata.merge(sourceId, contribution.metadata(), CapsuleLootTableBuilder::combineMetadata);
        }

        CapsuleLootTable build(String path) {
            if (this.entries.isEmpty()) {
                return null;
            }
            int resolvedMin = this.minRolls != null ? this.minRolls : 1;
            int resolvedMax = this.maxRolls != null ? Math.max(resolvedMin, this.maxRolls) : resolvedMin;
            boolean resolvedAllowDuplicates = this.allowDuplicates != null ? this.allowDuplicates : true;
            return new CapsuleLootTable(resolvedMin, resolvedMax, resolvedAllowDuplicates, List.copyOf(this.entries));
        }

        void logSummary(ResourceLocation tableId, int entryCount) {
            if (this.contributionMetadata.isEmpty()) {
                return;
            }
            ArrayList fragments = new ArrayList(this.contributionMetadata.size());
            this.contributionMetadata.forEach((sourceId, metadata) -> {
                StringBuilder builder = new StringBuilder();
                builder.append(sourceId.getNamespace()).append(": ");
                builder.append(metadata.addedEntries()).append("/").append(metadata.definedEntries()).append(" entries");
                if (!metadata.missingMods().isEmpty()) {
                    builder.append(" (missing mods: ");
                    builder.append(metadata.missingMods().entrySet().stream().map(entry -> (String)entry.getKey() + "=" + String.valueOf(entry.getValue())).sorted().reduce((left, right) -> left + ", " + right).orElse(""));
                    builder.append(")");
                }
                fragments.add(builder.toString());
            });
            CobbleCapsule.LOGGER.info("Capsule loot table {} built with {} entries from {} source(s): {}", new Object[]{tableId, entryCount, fragments.size(), String.join((CharSequence)", ", fragments)});
        }

        private static ContributionMetadata combineMetadata(ContributionMetadata first, ContributionMetadata second) {
            HashMap<String, Integer> combinedMissingMods = new HashMap<String, Integer>();
            combinedMissingMods.putAll(first.missingMods());
            second.missingMods().forEach((mod, count) -> combinedMissingMods.merge((String)mod, (Integer)count, Integer::sum));
            return new ContributionMetadata(first.definedEntries() + second.definedEntries(), first.addedEntries() + second.addedEntries(), combinedMissingMods);
        }
    }
}

