/*
 * Decompiled with CFR 0.152.
 */
package dev.gegy.roles.store.db;

import it.unimi.dsi.fastutil.objects.Object2LongMap;
import it.unimi.dsi.fastutil.objects.Object2LongMaps;
import it.unimi.dsi.fastutil.objects.Object2LongOpenHashMap;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.UUID;
import org.jetbrains.annotations.Nullable;

public final class Uuid2BinaryDatabase
implements Closeable {
    private static final int MAX_VALUE_SIZE = 0x400000;
    private static final int UUID_BYTES = 16;
    private static final int SIZE_BYTES = 4;
    private static final int HEADER_BYTES = 20;
    private static final long NULL_POINTER = -1L;
    private static final ByteOrder BYTE_ORDER = ByteOrder.BIG_ENDIAN;
    private final FileChannel file;
    private final Object2LongMap<UUID> pointers;
    private final ByteBuffer uuidBytes = ByteBuffer.allocate(16).order(BYTE_ORDER);
    private final ByteBuffer sizeBytes = ByteBuffer.allocate(4).order(BYTE_ORDER);
    private final LongBuffer uuidBuffer = this.uuidBytes.asLongBuffer();
    private final IntBuffer sizeBuffer = this.sizeBytes.asIntBuffer();
    private final ByteBuffer[] headerBytes = new ByteBuffer[]{this.uuidBytes, this.sizeBytes};

    private Uuid2BinaryDatabase(FileChannel file, Object2LongMap<UUID> pointers) {
        this.file = file;
        this.pointers = pointers;
        this.pointers.defaultReturnValue(-1L);
    }

    public static Uuid2BinaryDatabase open(Path path) throws IOException {
        FileChannel channel = FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        Object2LongMap<UUID> pointers = Uuid2BinaryDatabase.buildPointerIndex(channel);
        return new Uuid2BinaryDatabase(channel, pointers);
    }

    private static Object2LongMap<UUID> buildPointerIndex(FileChannel channel) throws IOException {
        Object2LongOpenHashMap pointers = new Object2LongOpenHashMap();
        ByteBuffer uuidBytes = ByteBuffer.allocate(16).order(BYTE_ORDER);
        ByteBuffer sizeBytes = ByteBuffer.allocate(4).order(BYTE_ORDER);
        LongBuffer uuidBuffer = uuidBytes.asLongBuffer();
        IntBuffer sizeBuffer = sizeBytes.asIntBuffer();
        int pointer = 0;
        long fileSize = channel.size();
        while ((long)pointer < fileSize) {
            channel.position(pointer);
            uuidBytes.clear();
            sizeBytes.clear();
            channel.read(uuidBytes);
            channel.read(sizeBytes);
            UUID uuid = new UUID(uuidBuffer.get(0), uuidBuffer.get(1));
            int size = Uuid2BinaryDatabase.validateSize(sizeBuffer.get(0));
            pointers.put((Object)uuid, (long)pointer);
            pointer += 20 + size;
        }
        return pointers;
    }

    @Nullable
    public synchronized ByteBuffer get(UUID key) throws IOException {
        long pointer = this.pointers.getLong((Object)key);
        if (pointer == -1L) {
            return null;
        }
        this.file.position(pointer + 16L);
        this.sizeBytes.clear();
        this.readToEnd(this.sizeBytes);
        int size = this.sizeBuffer.get(0);
        ByteBuffer buffer = ByteBuffer.allocate(size).order(BYTE_ORDER);
        this.readToEnd(buffer);
        return buffer;
    }

    public synchronized void put(UUID key, ByteBuffer bytes) throws IOException {
        Uuid2BinaryDatabase.validateSize(bytes.capacity());
        long pointer = this.pointers.getLong((Object)key);
        if (pointer == -1L) {
            this.push(key, bytes);
        } else {
            this.update(pointer, bytes);
        }
    }

    public synchronized boolean remove(UUID key) throws IOException {
        long pointer = this.pointers.removeLong((Object)key);
        if (pointer == -1L) {
            return false;
        }
        this.file.position(pointer + 16L);
        this.sizeBytes.clear();
        this.readToEnd(this.sizeBytes);
        int size = Uuid2BinaryDatabase.validateSize(this.sizeBuffer.get(0));
        long endPointer = pointer + 20L + (long)size;
        this.shiftAfter(endPointer, -(size + 20));
        return true;
    }

    private void push(UUID key, ByteBuffer bytes) throws IOException {
        long pointer = this.file.size();
        this.file.position(pointer);
        this.writeToEnd(this.writeHeader(key, bytes.capacity()));
        this.writeToEnd(bytes);
        this.pointers.put((Object)key, pointer);
    }

    private ByteBuffer[] writeHeader(UUID key, int size) {
        this.uuidBytes.clear();
        this.uuidBuffer.clear();
        this.uuidBuffer.put(key.getMostSignificantBits()).put(key.getLeastSignificantBits());
        this.sizeBytes.clear();
        this.sizeBuffer.clear();
        this.sizeBuffer.put(size);
        return this.headerBytes;
    }

    private void update(long pointer, ByteBuffer bytes) throws IOException {
        this.file.position(pointer + 16L);
        this.sizeBytes.clear();
        this.readToEnd(this.sizeBytes);
        int lastSize = Uuid2BinaryDatabase.validateSize(this.sizeBuffer.get(0));
        int newSize = Uuid2BinaryDatabase.validateSize(bytes.capacity());
        if (lastSize != newSize) {
            long endPointer = pointer + 20L + (long)lastSize;
            this.shiftAfter(endPointer, newSize - lastSize);
        }
        this.sizeBytes.clear();
        this.sizeBuffer.clear();
        this.sizeBuffer.put(newSize);
        this.file.position(pointer + 16L);
        this.writeToEnd(this.sizeBytes);
        this.writeToEnd(bytes);
    }

    private void shiftAfter(long source, int amount) throws IOException {
        long destination = source + (long)amount;
        long length = this.file.size() - source;
        if (amount > 0) {
            this.file.position(this.file.size());
            this.writeToEnd(ByteBuffer.allocate(amount).order(BYTE_ORDER));
        }
        if (length > 0L) {
            Uuid2BinaryDatabase.moveBytes(this.file, source, destination, length);
        }
        if (amount < 0) {
            this.file.truncate(this.file.size() + (long)amount);
        }
        for (Object2LongMap.Entry entry : Object2LongMaps.fastIterable(this.pointers)) {
            long pointer = entry.getLongValue();
            if (pointer < source) continue;
            entry.setValue(pointer + (long)amount);
        }
    }

    private void writeToEnd(ByteBuffer ... buffers) throws IOException {
        for (ByteBuffer buffer : buffers) {
            this.writeToEnd(buffer);
        }
    }

    private void writeToEnd(ByteBuffer buffer) throws IOException {
        for (long remaining = (long)buffer.remaining(); remaining > 0L; remaining -= (long)this.file.write(buffer)) {
        }
    }

    private void readToEnd(ByteBuffer buffer) throws IOException {
        for (long remaining = (long)buffer.remaining(); remaining > 0L; remaining -= (long)this.file.read(buffer)) {
        }
    }

    private static void moveBytes(FileChannel file, long source, long destination, long length) throws IOException {
        if (source < destination) {
            Uuid2BinaryDatabase.moveBytesForwards(file, source, destination, length);
        } else {
            Uuid2BinaryDatabase.moveBytesBackwards(file, source, destination, length);
        }
    }

    private static void moveBytesForwards(FileChannel file, long source, long destination, long length) throws IOException {
        int bufferSize = Math.min(1024, (int)length);
        ByteBuffer buffer = ByteBuffer.allocate(bufferSize).order(BYTE_ORDER);
        long backPointer = source + length;
        long offset = destination - source;
        long remaining = length;
        while (remaining > 0L) {
            int copySize = bufferSize;
            if (remaining < (long)bufferSize) {
                copySize = (int)remaining;
                buffer = ByteBuffer.allocate(copySize).order(BYTE_ORDER);
            }
            long frontPointer = backPointer - (long)copySize;
            int read = Uuid2BinaryDatabase.copyBytes(file, buffer, frontPointer, frontPointer + offset);
            remaining -= (long)read;
            backPointer -= (long)read;
        }
    }

    private static void moveBytesBackwards(FileChannel file, long source, long destination, long length) throws IOException {
        int bufferSize = Math.min(1024, (int)length);
        ByteBuffer buffer = ByteBuffer.allocate(bufferSize).order(BYTE_ORDER);
        long frontPointer = source;
        long offset = destination - source;
        long remaining = length;
        while (remaining > 0L) {
            int read = Uuid2BinaryDatabase.copyBytes(file, buffer, frontPointer, frontPointer + offset);
            remaining -= (long)read;
            frontPointer += (long)read;
        }
    }

    private static int copyBytes(FileChannel file, ByteBuffer buffer, long source, long destination) throws IOException {
        file.position(source);
        buffer.clear();
        int read = file.read(buffer);
        buffer.flip();
        file.position(destination);
        file.write(buffer);
        return read;
    }

    private static int validateSize(int size) throws IOException {
        if (size > 0x400000) {
            throw new IOException("size greater than maximum (" + size + ">4194304)");
        }
        if (size < 0) {
            throw new IOException("size is negative (" + size + "<0)");
        }
        return size;
    }

    @Override
    public synchronized void close() throws IOException {
        this.file.close();
    }
}

