/*
 * Decompiled with CFR 0.152.
 */
package de.waterdu.atlantis.file.storage;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import de.waterdu.atlantis.Atlantis;
import de.waterdu.atlantis.AtlantisLogger;
import de.waterdu.atlantis.Settings;
import de.waterdu.atlantis.file.ClassHolder;
import de.waterdu.atlantis.file.Config;
import de.waterdu.atlantis.file.datatypes.Configuration;
import de.waterdu.atlantis.file.datatypes.Data;
import de.waterdu.atlantis.file.datatypes.NamedData;
import de.waterdu.atlantis.file.storage.Storage;
import de.waterdu.atlantis.util.java.UUIDUtils;
import de.waterdu.atlantis.util.java.interfaces.QuadConsumer;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisURI;
import io.lettuce.core.pubsub.RedisPubSubAdapter;
import io.lettuce.core.pubsub.RedisPubSubListener;
import io.lettuce.core.pubsub.StatefulRedisPubSubConnection;
import java.io.File;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class Redis
implements Storage {
    private final UUID uuid = UUID.randomUUID();
    private final Set<UUID> connections = Sets.newConcurrentHashSet();
    private final AtomicBoolean acking = new AtomicBoolean(false);
    private final RedisURI uri;
    private final RedisClient client;
    private final StatefulRedisPubSubConnection<String, String> sub;
    private final StatefulRedisPubSubConnection<String, String> pub;
    private final Map<UUID, Long> timestamps = Maps.newConcurrentMap();
    private final AtomicBoolean updateTimestamps = new AtomicBoolean(true);
    private Config<? extends Configuration> config;
    private String channel;

    public Redis(String url) throws InvalidRedisUrlException {
        this(Redis.toURIArray(url));
    }

    private Redis(Object[] uriArray) {
        this((String)uriArray[0], (Integer)uriArray[1], (String)uriArray[2], (String)uriArray[3]);
    }

    public Redis(String ip, int port, String username, String password) {
        RedisURI.Builder builder = RedisURI.builder().withHost(ip).withPort(port);
        if (!username.isEmpty()) {
            builder.withAuthentication(username, password);
        }
        if (!password.isEmpty()) {
            builder.withPassword(password);
        }
        builder.withSsl(Settings.getSettings().isRedisSSL());
        this.uri = builder.build();
        this.client = RedisClient.create(this.uri);
        this.sub = this.client.connectPubSub();
        this.pub = this.client.connectPubSub();
    }

    @Override
    public <T extends Configuration> void init(Config<T> config, String modID, String key, String path, Gson gson, T configuration, ClassHolder<T> classes) {
        this.config = config;
        this.channel = modID + ":" + path;
        this.sub.sync().subscribe(this.channel);
        this.sub.addListener((RedisPubSubListener<String, String>)new RedisPubSubAdapter<String, String>(){

            @Override
            public void message(String channel, String message) {
                Redis.this.receive(message);
            }
        });
        this.publishAsync(Message.HI.prepare(this));
    }

    @Override
    public <T extends Configuration> void destruct(Config<T> config) {
        this.sub.close();
        this.pub.close();
        this.client.shutdown();
    }

    @Override
    public <T extends Configuration> CompletableFuture<Boolean> write(Config<T> config) {
        CompletableFuture<Boolean> result = new CompletableFuture<Boolean>();
        if (!this.isServerAlive()) {
            result.complete(false);
            return result;
        }
        boolean updateTimestamps = this.updateTimestamps.get();
        if (config.getConfigurationContainer().isData()) {
            Atlantis.THREAD_POOL.submit(() -> {
                for (Configuration out : config.getUUIDDataMap().values()) {
                    if (out == null) continue;
                    try {
                        this.writeSpecific(config, out, updateTimestamps);
                    }
                    catch (Exception e) {
                        AtlantisLogger.error("Failed to publish instance of {} called {} with UUID {}!", config.getName(), out.getUniqueName(), out.getUUID());
                    }
                }
                result.complete(true);
                AtlantisLogger.info("Successfully published {}.", config.getName());
            });
        } else {
            Atlantis.THREAD_POOL.submit(() -> {
                try {
                    if (updateTimestamps) {
                        this.timestamps.put(UUIDUtils.ZEROED_UUID, System.currentTimeMillis());
                    }
                    this.publishJson(UUIDUtils.ZEROED_UUID, config.getGson().toJson(config.getConfigurationContainer().get()));
                    AtlantisLogger.info("Successfully published {}.", config.getName());
                    config.remapLang();
                    result.complete(true);
                }
                catch (Exception e) {
                    AtlantisLogger.error("Failed to publish {}!", config.getName());
                    e.printStackTrace();
                    result.complete(false);
                }
            });
        }
        return result;
    }

    @Override
    public <T extends Configuration> CompletableFuture<Boolean> writeSpecific(Config<T> config, T data) {
        return this.writeSpecific(config, data, this.updateTimestamps.get());
    }

    private <T extends Configuration> CompletableFuture<Boolean> writeSpecific(Config<T> config, T data, boolean updateTimestamps) {
        CompletableFuture<Boolean> result = new CompletableFuture<Boolean>();
        if (!this.isServerAlive()) {
            result.complete(false);
            return result;
        }
        Atlantis.THREAD_POOL.submit(() -> {
            try {
                if (updateTimestamps) {
                    this.timestamps.put(data.getUUID(), System.currentTimeMillis());
                }
                this.publishJson(data.getUUID(), config.getGson().toJson((Object)data));
                result.complete(true);
            }
            catch (Exception e) {
                AtlantisLogger.error("Failed to publish instance of {} for file {}.", config.getName(), data.getUniqueName());
                e.printStackTrace();
                result.complete(false);
            }
        });
        return result;
    }

    @Override
    public <T extends Configuration> CompletableFuture<Boolean> read(Config<T> config, boolean skipNulls) {
        return CompletableFuture.completedFuture(false);
    }

    @Override
    public <T extends Configuration> CompletableFuture<Boolean> readAll(Config<T> config, File dir, Set<String> names) {
        return CompletableFuture.completedFuture(false);
    }

    @Override
    public <T extends Configuration> CompletableFuture<T> readFromUUID(Config<T> config, UUID uuid) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public <T extends Configuration> CompletableFuture<T> readFromName(Config<T> config, String name) {
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public <T extends Configuration> CompletableFuture<T> delete(Config<T> config, T data) {
        return this.delete(config, data, true);
    }

    private <T extends Configuration> CompletableFuture<T> delete(Config<T> config, T data, boolean sendMessage) {
        CompletableFuture<Object> result = new CompletableFuture<Object>();
        if (!this.isServerAlive()) {
            result.complete(null);
            return result;
        }
        Atlantis.THREAD_POOL.submit(() -> {
            try {
                Configuration removed = (Configuration)config.getUUIDDataMap().remove(data.getUUID());
                config.getFilenameDataMap().remove(data.getUniqueName() + config.getDataFileExtension());
                if (data instanceof NamedData) {
                    NamedData namedData = (NamedData)data;
                    config.getNameDataMap().remove(namedData.getName());
                }
                if (sendMessage) {
                    this.publish(Message.DEL.prepare(this, removed.getUUID()));
                }
                result.complete(removed);
            }
            catch (Exception e) {
                AtlantisLogger.error("Failed to remove {} for file {}!", config.getName(), data.getUniqueName());
                e.printStackTrace();
                result.complete(null);
            }
        });
        return result;
    }

    @Override
    public <T extends Configuration> CompletableFuture<T> deleteFromUUID(Config<T> config, UUID uuid) {
        return this.deleteFromUUID(config, uuid, true);
    }

    private <T extends Configuration> CompletableFuture<T> deleteFromUUID(Config<T> config, UUID uuid, boolean sendMessage) {
        CompletableFuture<Object> result = new CompletableFuture<Object>();
        if (!this.isServerAlive()) {
            result.complete(null);
            return result;
        }
        CompletableFuture<T> data = config.get(uuid);
        if (data != null && data.isDone()) {
            try {
                return this.delete(config, (Configuration)data.get(), sendMessage);
            }
            catch (InterruptedException | ExecutionException exception) {
                // empty catch block
            }
        }
        result.complete(null);
        return result;
    }

    @Override
    public <T extends Configuration> CompletableFuture<T> deleteFromName(Config<T> config, String name) {
        T data = config.get(name);
        if (data == null) {
            return CompletableFuture.completedFuture(null);
        }
        return this.delete(config, data);
    }

    private void publishJson(UUID uuid, String json) {
        this.publish((Object)((Object)Message.JSON) + " " + this.timestamps.get(uuid) + " " + json);
    }

    public void publish(String message) {
        this.pub.sync().publish(this.channel, message);
    }

    public void publishAsync(String message) {
        this.pub.async().publish(this.channel, message);
    }

    private void receive(String message) {
        Message.process(this, message).ifPresent(msg -> AtlantisLogger.debug("Received Redis message of type {} on channel {}.", msg.toString(), this.channel));
    }

    private <T extends Configuration> void put(long timestamp, String json) {
        Config<? extends Configuration> config = this.config;
        Configuration instance = (Configuration)config.getGson().fromJson(json, config.getClasses().getConfigurationClass());
        if (instance != null && this.timestamps.getOrDefault(instance.getUUID(), -1L) <= timestamp) {
            this.timestamps.put(instance.getUUID(), timestamp);
            if (instance instanceof Data) {
                config.put(instance);
            } else {
                config.getConfigurationContainer().set(config, instance);
            }
        }
    }

    private static Object[] toURIArray(String url) throws InvalidRedisUrlException {
        String[] account;
        if ((url = url.replace("redis://", "")).endsWith("/")) {
            url = url.substring(0, url.length() - 1);
        }
        if ((account = url.split("@")).length > 2) {
            throw new InvalidRedisUrlException(url);
        }
        String[] addr = account[account.length - 1].split(":");
        if (addr.length != 2) {
            throw new InvalidRedisUrlException(url);
        }
        String[] userpass = account[0].split(":");
        return new Object[]{addr[0], Integer.parseInt(addr[1]), userpass[0], userpass[1]};
    }

    private static enum Message {
        SYNC((redis, uuid, ack, msg) -> {
            if (ack.equals(((Redis)redis).uuid)) {
                ((Redis)redis).updateTimestamps.set(false);
                redis.write(((Redis)redis).config);
                ((Redis)redis).updateTimestamps.set(true);
            }
        }),
        DEL((redis, uuid, ack, msg) -> {
            if (!uuid.equals(((Redis)redis).uuid)) {
                ((Redis)redis).deleteFromUUID(((Redis)redis).config, ack, false);
            }
        }),
        ACK((redis, uuid, ack, msg) -> {
            if (ack.equals(((Redis)redis).uuid)) {
                ((Redis)redis).connections.add(uuid);
                if (((Redis)redis).acking.compareAndSet(false, true)) {
                    Atlantis.THREAD_POOL.schedule(() -> {
                        long delay = Settings.getSettings().getRedisSyncMilliseconds();
                        long totalDelay = 0L;
                        int i = 0;
                        for (UUID other : ((Redis)redis).connections) {
                            boolean last = i + 1 == ((Redis)redis).connections.size();
                            Atlantis.THREAD_POOL.schedule(() -> {
                                redis.publishAsync(SYNC.prepare((Redis)redis, other));
                                if (last) {
                                    ((Redis)redis).acking.set(false);
                                }
                            }, totalDelay, TimeUnit.MILLISECONDS);
                            totalDelay += delay;
                            ++i;
                        }
                    }, Settings.getSettings().getRedisAckMilliseconds(), TimeUnit.MILLISECONDS);
                }
            }
        }),
        HI((redis, uuid, ack, msg) -> {
            if (!uuid.equals(((Redis)redis).uuid)) {
                ((Redis)redis).connections.add(uuid);
                redis.publishAsync(ACK.prepare((Redis)redis, (UUID)uuid));
            }
        }),
        BYE((redis, uuid, ack, msg) -> {
            if (!uuid.equals(((Redis)redis).uuid)) {
                ((Redis)redis).connections.remove(uuid);
            }
        }),
        JSON((redis, uuid, ack, msg) -> {});

        private static final Map<String, Message> MESSAGES;
        private final QuadConsumer<Redis, UUID, UUID, String> consumer;

        private Message(QuadConsumer<Redis, UUID, UUID, String> consumer) {
            this.consumer = consumer;
        }

        private static Optional<Message> process(Redis redis, String string) {
            if (string.isEmpty()) {
                return Optional.empty();
            }
            if (string.startsWith(JSON.name())) {
                string = string.substring(5);
                long timestamp = Long.parseLong(string.substring(0, string.indexOf(" ")));
                string = string.substring(string.indexOf(" ") + 1);
                redis.put(timestamp, string);
                return Optional.of(JSON);
            }
            String[] args = string.split(" ");
            if (args.length < 2) {
                return Optional.empty();
            }
            Message message = MESSAGES.getOrDefault(args[0], null);
            if (message == null) {
                return Optional.empty();
            }
            try {
                message.consumer.accept(redis, UUID.fromString(args[1]), args.length > 2 ? UUID.fromString(args[2]) : UUIDUtils.ZEROED_UUID, string);
            }
            catch (IllegalArgumentException e) {
                return Optional.empty();
            }
            return Optional.of(message);
        }

        private String prepare(Redis redis) {
            return this.name() + " " + redis.uuid;
        }

        private String prepare(Redis redis, UUID ack) {
            return this.prepare(redis) + " " + ack.toString();
        }

        public String toString() {
            return this.name();
        }

        static {
            MESSAGES = Maps.newConcurrentMap();
            for (Message message : Message.values()) {
                MESSAGES.put(message.name().toLowerCase(Locale.ROOT), message);
            }
        }
    }

    public static class InvalidRedisUrlException
    extends RuntimeException {
        private final String url;

        public InvalidRedisUrlException(String url) {
            super(url);
            this.url = url;
        }

        @Override
        public String getMessage() {
            return "URL must be in the format \"redis://password@ip:port/\". Your URL is \"" + this.url + "\".";
        }
    }
}

