// Server-side, on-demand ArmorPicker (standalone).
// No caching: computes eligible armor list from tags each call.
//
//priority: 55
//
// Behavior notes:
// - Reads items from #forge:armors/* tags per slot
// - Computes required "power" tier based on armor and positive extra attributes
// - Picks a random eligible piece (optionally biased) for the requested power

(function () {
  let EquipmentSlot = Java.loadClass('net.minecraft.world.entity.EquipmentSlot');
  let Attributes = Java.loadClass('net.minecraft.world.entity.ai.attributes.Attributes');
  let HashSet = Java.loadClass('java.util.HashSet');

  global.ArmorPicker = global.ArmorPicker || {};
  global.ArmorPicker.version = global.ArmorPicker.version || 'server-ondemand-1.0.0';

  let MAX_ARMOR_BY_SLOT = {
    helmet: 10,
    chestplate: 12,
    leggings: 10,
    boots: 8,
  };

  let SLOT_INFO = {
    helmet: { slot: EquipmentSlot.HEAD, tag: '#forge:armors/helmets' },
    chestplate: { slot: EquipmentSlot.CHEST, tag: '#forge:armors/chestplates' },
    leggings: { slot: EquipmentSlot.LEGS, tag: '#forge:armors/leggings' },
    boots: { slot: EquipmentSlot.FEET, tag: '#forge:armors/boots' },
  };

  let SLOT_ALIASES = {
    head: 'helmet',
    chest: 'chestplate',
    legs: 'leggings',
    feet: 'boots',
  };

  function clampPower(p) {
    p = Math.floor(Number(p) || 0);
    if (p < 0) return 0;
    if (p > 10) return 10;
    return p;
  }

  function canonicalSlot(slot) {
    let key = String(slot || '').toLowerCase();
    if (SLOT_INFO[key]) return key;
    if (SLOT_ALIASES[key]) return SLOT_ALIASES[key];
    return null;
  }

  function itemIdsFromTag(tag) {
    try {
      let ing = Ingredient.of(tag);
      if (!ing) return [];
      if (ing.getItemIds) return ing.getItemIds().toArray();
      if (ing.itemIds) return ing.itemIds.toArray();
      if (ing.getStacks) {
        let stacks = ing.getStacks().toArray();
        let out = [];
        for (let i = 0; i < stacks.length; i++) {
          let s = stacks[i];
          let sid = null;
          try {
            if (s && s.id) sid = String(s.id);
            else if (s && s.getId) sid = String(s.getId());
            else sid = String(s);
          } catch (_e) {
            sid = String(s);
          }
          out.push(sid);
        }
        return out;
      }
    } catch (e) {
      console.log('[ArmorPicker] Failed to read tag', tag, e);
    }
    return [];
  }

  function extractAttributesForSlot(item, slotEnum) {
    let armor = 0;
    let tough = 0;
    let kbr = 0;
    let extras = new HashSet();

    try {
      let multi = item.getDefaultAttributeModifiers(slotEnum);
      if (multi) {
        let it = multi.entries().iterator();
        while (it.hasNext()) {
          let entry = it.next();
          let attr = entry.getKey();
          let mod = entry.getValue();
          let amt = Number(mod.getAmount());
          if (attr.equals(Attributes.ARMOR)) {
            armor += amt;
          } else if (attr.equals(Attributes.ARMOR_TOUGHNESS)) {
            tough += amt;
          } else if (attr.equals(Attributes.KNOCKBACK_RESISTANCE)) {
            kbr += amt;
          } else if (amt > 0) {
            extras.add(attr);
          }
        }
      }
    } catch (_e) {}

    return { armor: armor, tough: tough, kbr: kbr, extrasCount: extras.size() };
  }

  function computeRequiredPower(slotKey, armorValue, extrasCount) {
    let cap = MAX_ARMOR_BY_SLOT[slotKey] || 0;
    if (cap <= 0) return 10;
    let base = Math.ceil((armorValue * 10) / cap);
    if (base < 0) base = 0;
    if (base > 10) base = 10;
    let extraTier = Math.floor(extrasCount / 2);
    let required = base + extraTier;
    if (required > 10) return 11;
    if (required < 0) required = 0;
    return required;
  }

  function buildEntriesForSlot(slotKey) {
    let info = SLOT_INFO[slotKey];
    if (!info) return [];

    let ids = itemIdsFromTag(info.tag);
    if (!ids || ids.length === 0) return [];

    let entries = [];
    for (let i = 0; i < ids.length; i++) {
      let id = String(ids[i]);
      if (!id || id.indexOf('miapi:') === 0) continue;
      if (id.toString().includes('dragonsteel')) continue;
      if (
        id === 'gobber2:gobber2_helmet_dragon' ||
        id === 'gobber2:gobber2_chestplate_dragon' ||
        id === 'gobber2:gobber2_leggings_dragon' ||
        id === 'gobber2:gobber2_boots_dragon'
      ) {
        continue;
      }

      try {
        let jsStack = Item.of(id);
        let mcItem = jsStack && jsStack.item ? jsStack.item : null;
        if (!mcItem) continue;
        let attrs = extractAttributesForSlot(mcItem, info.slot);
        if (attrs.armor <= 0) continue;
        let armorCap = MAX_ARMOR_BY_SLOT[slotKey] || 0;
        if (armorCap > 0 && attrs.armor > armorCap) continue;
        let req = computeRequiredPower(slotKey, attrs.armor, attrs.extrasCount);
        if (req > 10) continue;
        entries.push({ id: id, armor: attrs.armor, extras: attrs.extrasCount, required: req });
      } catch (_e) {}
    }

    entries.sort(function (a, b) {
      if (a.required !== b.required) return a.required - b.required;
      if (a.armor !== b.armor) return a.armor - b.armor;
      return a.extras - b.extras;
    });

    return entries;
  }

  function pickImpl(slot, power, opts) {
    let key = canonicalSlot(slot);
    if (!key) return null;
    let p = clampPower(power);

    let entries = buildEntriesForSlot(key);
    if (!entries || entries.length === 0) return null;

    let ub = -1;
    for (let i = 0; i < entries.length; i++) {
      if (entries[i].required <= p) ub = i; else break;
    }
    if (ub < 0) return null;

    let bias = opts && typeof opts.bias !== 'undefined' ? Number(opts.bias) : 1.0;
    let u = Math.random();
    if (!isFinite(bias) || bias <= 0) {
      u = Math.random();
    } else if (bias !== 1.0) {
      let gamma = 1 / bias;
      u = Math.pow(u, gamma);
    }

    let idx = Math.min(ub, Math.floor(u * (ub + 1)));
    let id = entries[idx] && entries[idx].id ? entries[idx].id : null;
    if (!id) return null;
    if (opts && opts.returnStack) return Item.of(id);
    return id;
  }

  function rebuildImpl() {
    try {
      console.log('[ArmorPicker] On-demand implementation: rebuild() is a no-op (no cache).');
    } catch (_e) {}
    return false;
  }

  // Standalone API (so startup wrapper files can be deleted)
  global.ArmorPicker.pick = function (slot, power, opts) {
    try {
      return pickImpl(slot, power, opts || {});
    } catch (e) {
      console.log('[ArmorPicker] pick() error:', e);
      return null;
    }
  };

  global.ArmorPicker.rebuild = function () {
    try {
      return rebuildImpl();
    } catch (e) {
      console.log('[ArmorPicker] rebuild() error:', e);
      return false;
    }
  };

  try {
    console.log('[ArmorPicker] Loaded server-side on-demand implementation (no caching).');
  } catch (_e) {}
})();
