// Vampire subclass abilities
// Core mechanic: applying and consuming §c[Blood Marks]§7 on enemies,
// plus maintaining combat momentum via Thirst stacks.

/**
 * Crimson Brand
 * - Melee attacks have a chance to apply §c[Blood Mark]§7 to non‑ally, non‑boss monsters.
 * - Marks stack via potion amplifier on `kubejs:blood_mark`.
 * - Higher levels increase chance and duration.
 *
 * Suggested scaling:
 * §aLevel 1: ~10% chance, short duration
 * §aLevel 2: ~15% chance
 * §aLevel 3: ~20% chance
 * §aLevel 4: ~25% chance
 * §aLevel 5: ~30% chance, longer duration

EntityEvents.hurt(event => {
    if (!event.source.player) return
    let player = event.source.player
    if (!player.persistentData.get('crimson_brand')) return
    if (!event.entity.isMonster()) return
    if (typeof isAlly === 'function' && isAlly(player, event.entity)) return
    if (event.entity.tags.contains('boss')) return

    // Only trigger on melee hits
    if (typeof determineMethod === 'function') {
        let method = determineMethod(event.source)
        if (method !== 'melee') return
    }

    let level = player.persistentData.crimson_brand
    if (!level || level <= 0) return

    // 10% base +5% per level (≈10–30%)
    let chance = 0.10 + 0.05 * (level - 1)
    if (Math.random() > chance) return

    let duration = 60 + (20 * level) // 3–7 seconds
    let currentMarks = 0

    if (event.entity.potionEffects.isActive('kubejs:blood_mark')) {
        currentMarks = event.entity.potionEffects.getActive('kubejs:blood_mark').amplifier + 1
    }

    // Add one mark, clamp at 10
    let newMarks = currentMarks + 1
    if (newMarks > 10) newMarks = 10

    applyEffect(event.entity, 'kubejs:blood_mark', duration, newMarks - 1, 'crimson_brand', player)
})


/**
 * Blood Feast
 * - Killing a §c[Blood Marked]§7 enemy heals you and, at higher levels, nearby allies.
 * - Healing and buffs scale with both level and number of marks consumed.
 
EntityEvents.death(event => {
    if (!event.source.player) return
    if (!event.entity.isMonster()) return
    let player = event.source.player
    if (!player.persistentData.get('blood_feast')) return
    if (!event.entity.potionEffects.isActive('kubejs:blood_mark')) return

    let level = player.persistentData.blood_feast
    if (!level || level <= 0) return

    let markEffect = event.entity.potionEffects.getActive('kubejs:blood_mark')
    let marks = markEffect ? (markEffect.amplifier + 1) : 1

    // Heal % of max HP, scaled by level
    let healPercByLevel = {
        1: 0.03,
        2: 0.05,
        3: 0.07,
        4: 0.09,
        5: 0.12
    }
    let baseHealPerc = healPercByLevel[level] || 0.03
    let healAmount = player.maxHealth * baseHealPerc

    player.health = Math.min(player.maxHealth, player.health + healAmount)

    // Ally healing + buffs
    let radius = 4 + 2 * level
    let box = AABB.of(
        player.x + radius, player.y + radius, player.z + radius,
        player.x - radius, player.y - radius, player.z - radius
    )
    let dim = event.level
    let allies = dim.getEntitiesWithin(box).filter(ent => ent.alive && (!isAlly || isAlly(player, ent)))

    if (level >= 3 && allies.length > 0) {
        let allyHeal = player.maxHealth * 0.03 // ~3% flat ally heal
        allies.forEach(ent => {
            if (!ent.player) return
            ent.health = Math.min(ent.maxHealth, ent.health + allyHeal)
        })
    }

    if (level >= 4) {
        applyEffect(player, 'minecraft:strength', 100 + 20 * level, level >= 5 ? 1 : 0, 'blood_feast', player)
        if (level >= 5) {
            applyEffect(player, 'minecraft:regeneration', 100, 0, 'blood_feast', player)
        }
    }

    // Consume the marks on the killed enemy
    Utils.server.runCommandSilent(`/effect clear ${event.entity.uuid} kubejs:blood_mark`)
})


// Night Veil – state flags
let nightVeilActive = {}
let nightVeilOpeningHit = {}

/**
 * Night Veil – Active ability
 * - Activated via ability hotkey (night_veil_ability).
 * - Grants short invisibility + speed, enabling a strong opening hit against §c[Blood Marked]§7 foes.
 *
 * Cooldown and effect duration scale with level.
 *
function night_veil_ability(player) {
    if (!player.persistentData.get('night_veil')) return
    if (typeof isSkillCoolingDown === 'function' && isSkillCoolingDown(player, 'night_veil')) return

    let level = player.persistentData.night_veil
    if (!level || level <= 0) return

    // Duration ~3–7 seconds
    let duration = 60 + 20 * level

    // Cooldown: 20s → 12s
    let cdByLevel = {
        1: 400,
        2: 360,
        3: 320,
        4: 280,
        5: 240
    }
    let cooldown = cdByLevel[level] || 400
    if (typeof addSkillCooldown === 'function') {
        addSkillCooldown(player, 'night_veil', cooldown)
    }

    player.potionEffects.add('minecraft:invisibility', duration, 0, true, true)
    player.potionEffects.add('minecraft:speed', duration, level >= 4 ? 1 : 0, true, true)

    nightVeilActive[player.username] = true
    nightVeilOpeningHit[player.username] = true

    Utils.server.runCommandSilent(
        `/execute in ${player.level.dimension} run playsound irons_spellbooks:cast.shadow_step ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.5`
    )

    Utils.server.scheduleInTicks(duration, () => {
        nightVeilActive[player.username] = false
        nightVeilOpeningHit[player.username] = false
    })
}


// Night Veil – bonus damage vs Blood‑Marked enemies while active
EntityEvents.hurt(event => {
    if (!event.source.player) return
    let player = event.source.player
    if (!player.persistentData.get('night_veil')) return
    if (!nightVeilActive[player.username]) return
    if (!event.entity.isMonster()) return
    if (!event.entity.potionEffects.isActive('kubejs:blood_mark')) return

    let level = player.persistentData.night_veil
    if (!level || level <= 0) return

    // Bonus damage vs marked enemies (≈10–30%)
    let bonusMult = 0.10 + 0.05 * (level - 1)
    let extraDamage = event.damage * bonusMult
    if (extraDamage > 0) {
        event.entity.attack(extraDamage)
    }

    // Opening hit: add/refresh a Blood Mark
    if (nightVeilOpeningHit[player.username]) {
        nightVeilOpeningHit[player.username] = false

        let currentMarks = 0
        let currentDuration = 80
        if (event.entity.potionEffects.isActive('kubejs:blood_mark')) {
            let eff = event.entity.potionEffects.getActive('kubejs:blood_mark')
            currentMarks = eff.amplifier + 1
            currentDuration = eff.duration
        }
        let newMarks = currentMarks + 1
        if (newMarks > 10) newMarks = 10
        let newDuration = currentDuration + (20 * level)
        applyEffect(event.entity, 'kubejs:blood_mark', newDuration, newMarks - 1, 'night_veil', player)
    }
})


/**
 * Sanguine Frenzy – Active ability
 * - Consumes §c[Blood Marks]§7 within a radius, dealing burst damage and granting buffs per mark.
 * - Stronger when many enemies are marked.
 *
 * Activated via ability hotkey (sanguine_frenzy_ability).
 *
function sanguine_frenzy_ability(player) {
    if (!player.persistentData.get('sanguine_frenzy')) return
    if (typeof isSkillCoolingDown === 'function' && isSkillCoolingDown(player, 'sanguine_frenzy')) return

    let level = player.persistentData.sanguine_frenzy
    if (!level || level <= 0) return

    let radius = 4 + 2 * level
    let box = AABB.of(
        player.x + radius, player.y + 2, player.z + radius,
        player.x - radius, player.y - 2, player.z - radius
    )
    let dim = player.level
    let enemies = dim.getEntitiesWithin(box).filter(ent => ent.isMonster() && !ent.tags.contains('boss') && ent.potionEffects.isActive('kubejs:blood_mark'))
    if (enemies.length === 0) return

    let totalMarks = 0

    enemies.forEach(ent => {
        let eff = ent.potionEffects.getActive('kubejs:blood_mark')
        let marks = eff ? (eff.amplifier + 1) : 1
        totalMarks += marks

        // Burst damage per marked enemy
        let baseDamage = 1 + level
        let extraFromMarks = 0.5 * marks
        let damage = baseDamage + extraFromMarks
        ent.attack(damage)

        Utils.server.runCommandSilent(`/effect clear ${ent.uuid} kubejs:blood_mark`)
        Utils.server.runCommandSilent(
            `/execute in ${dim.dimension} run particle irons_spellbooks:blood ${ent.x} ${ent.y + 1} ${ent.z} 0.3 0.3 0.3 0.1 20 force`
        )
    })

    if (totalMarks <= 0) return

    // Heal 0.5% max HP per mark
    let healPercentPerMark = 0.005
    let healAmount = player.maxHealth * healPercentPerMark * totalMarks
    player.health = Math.min(player.maxHealth, player.health + healAmount)

    // Strength + speed buffs scale with marks and level
    let buffDuration = 60 + 4 * totalMarks // up to a few seconds
    let strengthAmp = Math.min(3, Math.floor(totalMarks / (level >= 3 ? 3 : 5)))
    if (strengthAmp > 0) {
        applyEffect(player, 'minecraft:strength', buffDuration, strengthAmp - 1, 'sanguine_frenzy', player)
    }
    if (level >= 3) {
        let speedAmp = Math.min(2, Math.floor(totalMarks / 6))
        if (speedAmp > 0) {
            applyEffect(player, 'minecraft:speed', buffDuration, speedAmp - 1, 'sanguine_frenzy', player)
        }
    }

    // Big payoff at high levels with many marks
    if (level >= 4 && totalMarks >= 10) {
        applyEffect(player, 'minecraft:regeneration', 160, 0, 'sanguine_frenzy', player)
    }
    if (level >= 5 && totalMarks >= 15) {
        let allyRadius = radius + 2
        let boxAllies = AABB.of(
            player.x + allyRadius, player.y + allyRadius, player.z + allyRadius,
            player.x - allyRadius, player.y - allyRadius, player.z - allyRadius
        )
        let allies = dim.getEntitiesWithin(boxAllies).filter(ent => ent.alive && (!isAlly || isAlly(player, ent)))
        allies.forEach(ent => {
            if (!ent.player) return
            applyEffect(ent, 'minecraft:resistance', 160, 0, 'sanguine_frenzy', player)
        })
    }

    // Cooldown: ~30s → 15s
    let cdMap = {
        1: 600,
        2: 500,
        3: 400,
        4: 360,
        5: 300
    }
    let cooldown = cdMap[level] || 600
    if (typeof addSkillCooldown === 'function') {
        addSkillCooldown(player, 'sanguine_frenzy', cooldown)
    }

    Utils.server.runCommandSilent(
        `/execute in ${player.level.dimension} run playsound minecraft:entity.evoker.prepare_wololo ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 0.8`
    )
}


// Thirst Unending – track stacks per‑player
let thirstStacks = {}
let thirstTimers = {}

/**
 * Thirst Unending
 * - Hitting §c[Blood Marked]§7 enemies builds stacks while in combat.
 * - Stacks grant increasing Strength/Speed and minor sustain at higher levels.
 * - Stacks decay over time when not attacking.
 *
EntityEvents.hurt(event => {
    if (!event.source.player) return
    let player = event.source.player
    if (!player.persistentData.get('thirst_unending')) return
    if (!event.entity.isMonster()) return
    if (!event.entity.potionEffects.isActive('kubejs:blood_mark')) return

    let level = player.persistentData.thirst_unending
    if (!level || level <= 0) return

    let name = player.username
    let maxStacks = 3 + level // 4–8 stacks
    let currentStacks = thirstStacks[name] || 0
    currentStacks += 1
    if (currentStacks > maxStacks) currentStacks = maxStacks
    thirstStacks[name] = currentStacks
    thirstTimers[name] = 6 // seconds until decay tick

    // Convert stacks into buffs
    let strengthAmp = Math.max(0, Math.floor((currentStacks - 1) / 2))
    let speedAmp = level >= 3 ? Math.max(0, Math.floor((currentStacks - 1) / 3)) : 0

    applyEffect(player, 'minecraft:strength', 140, strengthAmp, 'thirst_unending', player)
    if (speedAmp > 0) {
        applyEffect(player, 'minecraft:speed', 140, speedAmp, 'thirst_unending', player)
    }
    if (level >= 4 && currentStacks >= maxStacks) {
        applyEffect(player, 'minecraft:regeneration', 80, 0, 'thirst_unending', player)
    }
    if (level >= 5 && currentStacks >= maxStacks) {
        applyEffect(player, 'minecraft:absorption', 160, 0, 'thirst_unending', player)
    }
})


// Thirst Unending – decay logic
PlayerEvents.tick(event => {
    if (event.server.tickCount % 20 !== 0) return
    let player = event.player
    if (!player.persistentData.get('thirst_unending')) return

    let name = player.username
    if (thirstTimers[name] === undefined) return

    thirstTimers[name] -= 1
    if (thirstTimers[name] > 0) return

    thirstTimers[name] = 0
    let stacks = thirstStacks[name] || 0
    if (stacks <= 0) return

    stacks -= 1
    thirstStacks[name] = stacks
    if (stacks === 0) {
        // Clear buffs when fully decayed
        Utils.server.runCommandSilent(`/effect clear ${player.uuid} minecraft:strength`)
        Utils.server.runCommandSilent(`/effect clear ${player.uuid} minecraft:speed`)
        Utils.server.runCommandSilent(`/effect clear ${player.uuid} minecraft:regeneration`)
        Utils.server.runCommandSilent(`/effect clear ${player.uuid} minecraft:absorption`)
    }
})

*/