
/**
 * Mining Fever: 5 levels
 * - Mining blocks without stopping grants stacking effects that speed up mining.
 * - Thresholds per level: 50/100/150/200/250 blocks.
 * - Effects stack and scale with thresholds; capped by `mining_fever` level.
 */

// Per-player counters keyed by username (simple objects for clarity/consistency)
let fever_blocks_mined = {}
let fever_countdown = {}
let fever_decay_timer = {}

// Effects granted at each 50-block threshold
let fever_effects = {
  50: { effects: ['minecraft:haste'], amplifier: 0, ticks: 30 },
  100: { effects: ['minecraft:haste', 'minecraft:speed'], amplifier: 0, ticks: 30 },
  150: { effects: ['minecraft:haste', 'minecraft:speed'], amplifier: 1, ticks: 40 },
  200: { effects: ['minecraft:haste', 'minecraft:speed'], amplifier: 1, ticks: 50 },
  250: { effects: ['minecraft:haste', 'minecraft:speed'], amplifier: 2, ticks: 60 },
}

// Soft decay + kill-extend tuning
let DECAY_INTERVAL_TICKS = 2   // apply decay once per second
let DECAY_AMOUNT = 1           // lose 10 points per interval while idle
let KILL_EXTEND_TICKS = 20      // extend fever timer by 1s on monster kill
let KILL_EXTEND_CAP = 60        // fever timer extension cap (3s)

// Helpers
let getUser = p => p.username
let clamp = (val, min, max) => Math.max(min, Math.min(max, val))
let isPickaxe = id => id && id.includes('pickaxe')
let isOre = id => id && id.includes('ore')

function remainingToNextLevel(player, count) {
  let cap = (player && player.persistentData && player.persistentData.mining_fever) ? (player.persistentData.mining_fever * 50) : 0
  if (cap <= 0) return 0
  let base = Math.floor(count / 50) * 50
  let next = base + 50
  if (next > cap) next = cap
  return Math.max(0, next - count)
}

function smallCounterText(player, count) {
  let level = player?.persistentData?.mining_fever || 0
  let currentTier = Math.floor(count / 50)
  if (level >= 5 && currentTier >= 5) {
    return 'Max Level'
  }
  let remaining = remainingToNextLevel(player, count)
  return `Next: ${remaining}`
}


// Display Level Properties
let dl_props = {
    type: 'text',
    color: 'gold',
    shadow: true,
    scale: 0.5,
    x: 0,
    y: 15,
    alignX: 'center',
    alignY: 'center',
    visible: true,
}

// Stacks Properties
let s_props = {
    type: 'text',
    color: 'darkAqua',
    shadow: true,
    scale: 0.5,
    x: 1,
    y: 10,
    alignX: 'center',
    alignY: 'center',
    visible: true,
}


// Iron Pickaxe Properties
let p_props = {
    type: 'rectangle',
    x: 0,
    y: 21,
    w: 8,
    h: 8,
    alignX: 'center',
    alignY: 'center',
    texture: 'minecraft:textures/item/iron_pickaxe.png',
    visible: true,
}



BlockEvents.broken(event => {
  let player = event.player
  if (!player || !player.persistentData.get('mining_fever')) return
  if (!isPickaxe(player.mainHandItem.id)) return
  if (!player.persistentData.mining_fever_started) {
    player.persistentData.putBoolean('mining_fever_started', true)
  }
  let user = getUser(player)
  let blockId = event.block.id

  // Increment streak (ores +2; others +1). First mined block starts at 1.
  if (isOre(blockId)) {
    if (fever_blocks_mined[user] === undefined) {
      fever_blocks_mined[user] = 1
    } else {
      fever_blocks_mined[user] += 2
    }
  } else {
    if (fever_blocks_mined[user] === undefined) {
      fever_blocks_mined[user] = 1
    } else {
      fever_blocks_mined[user] += 1
    }
  }

  // Refresh countdown
  fever_countdown[user] = 35
  // Reset decay timer on activity
  fever_decay_timer[user] = 0

  // Cap at 300 and at player level cap (level * 50)
  fever_blocks_mined[user] = clamp(
    fever_blocks_mined[user],
    0,
    Math.min(300, player.persistentData.mining_fever * 50)
  )

  // Determine current 50-block tier and apply effects
  let count = fever_blocks_mined[user]
  if (count >= 1) {
    let feverLevel = Math.floor(count / 50) * 50
    if (feverLevel > 0) {
      let displayLevel = Math.floor(count / 50)
      let bundle = fever_effects[feverLevel]
      if (bundle) {
        bundle.effects.forEach(eff => {
          player.potionEffects.add(eff, bundle.ticks, bundle.amplifier, true, true)
        })
      }
      player.paint({
        display_level: {
            type: dl_props.type,                                
            text: `Level: ${displayLevel}`,
            color: dl_props.color,
            shadow: dl_props.shadow,
            scale: dl_props.scale,
            x: dl_props.x,
            y: dl_props.y,
            alignX: dl_props.alignX,
            alignY: dl_props.alignY,
            visible: dl_props.visible,
        },
        iron_pickaxe_symbol: {
            type: p_props.type,
            x: p_props.x,
            y: p_props.y,
            w: p_props.w,
            h: p_props.h,
            alignX: p_props.alignX,
            alignY: p_props.alignY,
            texture: p_props.texture,
            visible: p_props.visible,
        },
      })
    }
  }
  // Always show small counter with blocks needed to next level (or Max Level)
  if ((fever_blocks_mined[user] || 0) > 0) {
    let text = smallCounterText(player, fever_blocks_mined[user] || 0)
    player.paint({
      fever_stacks_small: {
        type: s_props.type,
        text: `${text}`,
        color: s_props.color,
        shadow: s_props.shadow,
        scale: s_props.scale,
        x: s_props.x,
        y: s_props.y,
        alignX: s_props.alignX,
        alignY: s_props.alignY,
        visible: s_props.visible,
      }
    })
  }
})

PlayerEvents.tick(event => {
  let player = event.player
  if (!player || !player.persistentData.get('mining_fever')) return
  if (player.persistentData.mining_fever_started != true) return
  let user = getUser(player)

  // Initialize countdown if needed; clamp negative counts to zero
  if (fever_blocks_mined[user] < 0) fever_blocks_mined[user] = 0

  if (fever_countdown[user] === undefined) {
    fever_countdown[user] = 35
  } else {
    fever_countdown[user]--
  }

  // Still within streak window
  if (fever_countdown[user] > 0) return

  // Soft decay when idle: reduce points periodically until zero
  fever_decay_timer[user] = (fever_decay_timer[user] || 0) + 1
  if (fever_decay_timer[user] >= DECAY_INTERVAL_TICKS) {
    fever_decay_timer[user] = 0
    let before = fever_blocks_mined[user] || 0
    let after = Math.max(0, before - DECAY_AMOUNT)
    fever_blocks_mined[user] = after

    if (after <= 0) {
      player.paint({ display_level: { visible: false } })
      player.paint({ iron_pickaxe_symbol: { visible: false } })
      player.paint({ fever_stacks_small: { visible: false } })
      player.persistentData.putBoolean('mining_fever_started', false)
    } else {
      let displayLevel = Math.floor(after / 50)
      if (displayLevel > 0) {
        player.paint({
            display_level: {
                type: dl_props.type,    
                text: `Level: ${displayLevel}`,
                color: dl_props.color,
                shadow: dl_props.shadow,
                scale: dl_props.scale,
                x: dl_props.x,
                y: dl_props.y,
                alignX: dl_props.alignX,
                alignY: dl_props.alignY,
                visible: dl_props.visible,
            },
            iron_pickaxe_symbol: {
                type: p_props.type,
                x: p_props.x,
                y: p_props.y,
                w: p_props.w,
                h: p_props.h,
                alignX: p_props.alignX,
                alignY: p_props.alignY,
                texture: p_props.texture,
                visible: p_props.visible,
            },
        })
      } else {
        // Hide main overlay if below first threshold
        player.paint({ display_level: { visible: false } })
        player.paint({ iron_pickaxe_symbol: { visible: false } })
      }
      // Always update small counter while decaying above 0 (or show Max Level)
      let text = smallCounterText(player, after)
      player.paint({
        fever_stacks_small: {
            type: s_props.type,
            text: `${text}`,
            color: s_props.color,
            shadow: s_props.shadow,
            scale: s_props.scale,
            x: s_props.x,
            y: s_props.y,
            alignX: s_props.alignX,
            alignY: s_props.alignY,
            visible: s_props.visible,
        }
      })
    }
  }
})

// Kill Extend: killing a monster extends fever timer slightly (if active)
EntityEvents.death(event => {
  if (!event.source || !event.source.player) return
  let player = event.source.player
  if (!player.persistentData.get('mining_fever')) return
  if (!event.entity || !event.entity.isMonster()) return

  let user = getUser(player)
  let count = fever_blocks_mined[user] || 0
  if (count <= 0) return

  let current = fever_countdown[user] || 0
  fever_countdown[user] = clamp(current + KILL_EXTEND_TICKS, 0, KILL_EXTEND_CAP)
})




//_______________________________________________________________________________________________________________________

/**
 * Avarice: 5 Levels
 *  - Mining ores will give you a very brief x ray effect
 *  - Increasing the level: Procs the ability when mining more common ores, gives higher level of the effect, increases the duration of the effect
 * Level 1: Ancient Debris
 * Level 2: Ancient Debris, Emerald
 * Level 3: Ancient Debris, Emerald, Diamond
 * Level 4: Ancient Debris, Emerald, Diamond, Gold
 * Level 5: Ancient Debris, Emerald, Diamond, Gold, Copper
 * Level 6: Ancient Debris, Emerald, Diamond, Gold, Copper, Iron
 * Level 7: Ancient Debris, Emerald, Diamond, Gold, Copper, Iron, Lapis
 * Level 8: Ancient Debris, Emerald, Diamond, Gold, Copper, Iron, Lapis, Redstone
 * Level 9: Ancient Debris, Emerald, Diamond, Gold, Copper, Iron, Lapis, Redstone, Coal
 * Level 10: Ancient Debris, All Ores
 * let avarice = {
    1: {
        ores: ['ancient_debris'],
        level: 0,
        ticks: 40
    },
    2: {
        ores: ['ancient_debris', 'emerald'],
        level: 0,
        ticks: 50
    },
    3: {
        ores: ['ancient_debris','emerald', 'diamond'],
        level: 1,
        ticks: 60
    },
    4: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold'],
        level: 1,
        ticks: 70
    },
    5: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold', 'copper'],
        level: 2,
        ticks: 80
    },
    6: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold', 'copper', 'iron'],
        level: 2,
        ticks: 90
    },
    7: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold', 'copper', 'iron', 'lapis'],
        level: 2,
        ticks: 100
    },
    8: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold', 'copper', 'iron', 'lapis', 'redstone'],
        level: 2,
        ticks: 100
    },
    9: {
        ores: ['ancient_debris','emerald', 'diamond', 'gold', 'copper', 'iron', 'lapis', 'redstone', 'coal'],
        level: 2,
        ticks: 100
    },
    10: {
        ores: ['ancient_debris','ore'],
        level: 2,
        ticks: 100
    },
}
 * 
 */


const avarice = {
    1: {
        chance: 0.05,
        currency: 1,
    },
    2: {
        chance: 0.10,
        currency: 1,
    },
    3: {
        chance: 0.15,
        currency: 2,
    },
    4: {
        chance: 0.20,
        currency: 2,
    },
    5: {
        chance: 0.25,
        currency: 3,
    },
    6: {
        chance: 0.30,
        currency: 3,
    },
    7: {
        chance: 0.35,
        currency: 3,
    },
    8: {
        chance: 0.40,
        currency: 3,
    },
    9: {
        chance: 0.45,
        currency: 3,
    },
    10: {
        chance: 0.50,
        currency: 5,
    },
}



BlockEvents.broken(event => {
    if (!event.entity.player) return
    if (!event.player.persistentData.get(`avarice`)) return
    if (!event.block.id.includes('_ore')) return
    let level = event.player.persistentData.avarice
    let chance = avarice[level].chance
    if (event.player.potionEffects.isActive('minecraft:luck')) chance += 0.10
    let currency = avarice[level].currency
    if (chance >= 1.00) currency += 1
    if (chance <= Math.random()) {
        addMoney(event.player, currency)
        let player = event.player
        Utils.server.runCommandSilent(`/execute in ${event.player.level.dimension} run playsound sdmshoprework:buy_sound player ${event.player.username} ${event.block.x} ${event.block.y} ${event.block.z} 0.3 1.7`)
        spawnSpark(event, {x: event.block.x, y: event.block.y+0.25, z: event.block.z}, 'gold', 15, 0.20, {x: 0, y: 0.2, z: 0}, true)
    }
})


//_______________________________________________________________________________________________________________________


/**
 * Emberheart: 5 Levels
 * - campfires give you effects. Regular campfire gives restorative effects, 
 * 1. Regeneration
 * 2. Bug Pheromones
 * 3. Oakskin
 * 4. Saturation
 * 5. Clarity
 */
/**
 * Ashenheart: 5 levels:
 * - soul campfires give offensive effects
 * 1. Berserk
 * 2. Haste
 * 3. Impact
 * 4. Echoing Strikes
 * 5. Sun's Blessing
 */





let emberheart_effects = {
    'minecraft:campfire': {
        1: {
            effects: ['alexsmobs:bug_pheromones'],
            level: 1
        },
        2: {

            effects: ['minecraft:regeneration'],
            level: 2
        },
        3: {
            effects: ['minecraft:saturation'],
            level: 3
        },
        4: {
            effects: ['cofh_core:clarity'],
            level: 4
        },
        5: {
            effects: ['irons_spellbooks:oakskin'],
            level: 5

        },
    },
}

let ashenheart_effects = {
    'minecraft:soul_campfire': {
        1: {
            effects: ['runiclib:berserk'],
            level: 1
        },
        2: {
            effects: ['minecraft:haste'],
            level: 2
        },
        3: {
            effects: ['projectile_damage:impact'],
            level: 3
        },
        4: {
            effects: ['irons_spellbooks:echoing_strikes'],
            level: 4
        },
        5: {
            effects: ['mowziesmobs:suns_blessing'],
            level: 5

        },
    },
}


// Campfire
BlockEvents.placed('minecraft:campfire', event => {
    if (!event.player.persistentData.get(`emberheart`)) return
    Utils.server.scheduleInTicks(10, e => {
        let level = event.player.persistentData.emberheart
        if (event.level.getBlock(event.block.x, event.block.y, event.block.z).id == event.block.id) {
            e.repeating = true
            let range = 2*level
            let box = AABB.of(event.block.x + range, event.block.y + range, event.block.z + range, event.block.x - range, event.block.y - range, event.block.z - range)
            // get entities within and filter for players
            let entities = event.level.getEntitiesWithin(box)
            entities.forEach(entity => {
                if (entity.player) {
                    // loop through levels at the player's emberheart level and below
                    for (let i = 1; i <= level; i++) {
                        if (emberheart_effects[event.block.id][i] == undefined) return
                        let effects = emberheart_effects[event.block.id][i].effects
                        effects.forEach(effect => {
                            let amplifier = parseInt(level - emberheart_effects[event.block.id][i].level)
                            if (amplifier > 1) {
                                amplifier = 1
                            }
                            applyEffect(entity, effect, 60, amplifier, true, true)
                        })
                    }
                }

            })
        } else {
            e.repeating = false
        }
    })
    
})

// Soul Campfire
BlockEvents.placed('minecraft:soul_campfire', event => {
    if (!event.player.persistentData.get(`ashenheart`)) return
    Utils.server.scheduleInTicks(10, e => {
        let level = event.player.persistentData.ashenheart
        if (event.level.getBlock(event.block.x, event.block.y, event.block.z).id == event.block.id) {
            e.repeating = true
            let range = 2*level
            let box = AABB.of(event.block.x + range, event.block.y + range, event.block.z + range, event.block.x - range, event.block.y - range, event.block.z - range)
            // get entities within and filter for players
            let entities = event.level.getEntitiesWithin(box)
            entities.forEach(entity => {
                if (entity.player) {
                    // loop through levels at the player's ashenheart level and below
                    for (let i = 1; i <= level; i++) {
                        if (ashenheart_effects[event.block.id][i] == undefined) return
                        let effects = ashenheart_effects[event.block.id][i].effects
                        effects.forEach(effect => {
                            let amplifier = parseInt(level - ashenheart_effects[event.block.id][i].level)
                            if (amplifier > 1) {
                                amplifier = 1
                            }
                            applyEffect(entity, effect, 60, amplifier, true, true)
                        })
                    }
                }

            })
        } else {
            e.repeating = false
        }
    })
    
})

//_______________________________________________________________________________________________________________________

/**
 * 
 * ItemEvents.rightClicked('minecraft:emerald', event => {
    event.player.persistentData.remove('inspire')
    event.player.persistentData.remove('mindshatter')
    event.player.persistentData.putInt('windborne_countdown', 0)
    if (!event.player.persistentData.get(`windborne`)) {
        event.player.persistentData.putInt(`windborne`, 1)
    } else {
        event.player.persistentData.windborne++
    }
    if (event.player.persistentData.windborne > 5) {
        event.player.persistentData.windborne = 2
    }
    event.player.tell(`Windborne: ${event.player.persistentData.windborne}`)
    event.player.paint({'windborne_symbol': {visible: true}})
})
 */

// Windborne: 5 Levels
// Gives the player an AOE angel wings effect for a short duration
// Each level increases the range of the effect and the duration of the effect
// Level 1: 3 blocks, 5 seconds
// Level 2: 6 blocks, 10 seconds
// Level 3: 9 blocks, 15 seconds
// Level 4: 12 blocks, 20 seconds
// Level 5: 15 blocks, 25 seconds



function windborne_ability(player) {
    if (!player.persistentData.get('windborne_active')) {
        
        player.persistentData.putInt('windborne_active', 1)
    } else {
        if (player.persistentData.windborne_active == 1) {
            player.persistentData.windborne_active = 0
        } else {
            player.persistentData.windborne_active = 1
        }
    }
    activateWindborne(player) 
}

function activateWindborne(player) {
    if (!player.persistentData.get('windborne')) return
    if (player.persistentData.windborne_active == 1) { 
        Utils.server.runCommandSilent(`/execute at ${player.username} run playsound irons_spellbooks:spell.gust.cast ambient ${player.username} ~ ~ ~ 1 1.7`)
        Utils.server.runCommandSilent(`/effect give ${player.username} irons_spellbooks:angel_wings infinite 0 false`);
        Utils.server.runCommandSilent(`/effect give ${player.username} minecraft:weakness infinite 0 false`);
    } else {
        Utils.server.runCommandSilent(`/execute at ${player.username} run playsound irons_spellbooks:spell.gust.cast ambient ${player.username} ~ ~ ~ 1 0.9`)
        Utils.server.runCommandSilent(`/effect clear ${player.username} irons_spellbooks:angel_wings`);
        Utils.server.runCommandSilent(`/effect clear ${player.username} minecraft:weakness`);
    }

}




/**
 * 
 * // Windborne
NetworkEvents.dataReceived("global.hKeySpecial.consumeClick", (e) => {
    let player = e.player;
    if (!player.persistentData.get(`windborne`)) return;

    toggleWindborne(player)
})
 * 
 * 
 * 
 */


let revivePlayer = global.revivePlayer
let needsRevive = global.needsRevive

EntityEvents.hurt('player', event => {
    let player = event.entity;
    if (!player.persistentData.get('sacrifice')) return
    if (isSkillCoolingDown(player, 'sacrifice')) return; // prevent spamming the ability
    if (!player.persistentData.getBoolean('sacrifice_active') != true) return; // only if sacrifice is active
    Utils.server.scheduleInTicks(5, () => {
        if (!needsRevive(player)) return;
        let level = player.persistentData.sacrifice
        let allPets = player.level.entities.filter(entity => entity.tags.contains('tamed_beast') && entity.tags.contains(`Owner:${player.username}`))
        if (allPets.length === 0) return; // no pets to revive
        allPets.forEach(pet => {
            let index = allPets.indexOf(pet);
            Utils.server.scheduleInTicks(index * 5, () => {
                bloodBurstKill(pet); // kill the pet with blood burst effect
            })
                    
            
        })
        revivePlayer(player, true); // revive without effects
        let num = 0.20 + ((level - 1) * 0.15)
        if (!num || num <= 0) num = 1; 
        player.setHealth(parseFloat(player.maxHealth/num).toFixed(2)) // restore health based on level
        Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run playsound minecraft:entity.wither.ambient ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 0.75`)
        let cooldown = 3600-(300*(level-1))
        addSkillCooldown(player, 'sacrifice', cooldown)
    })
})


function sacrifice_ability(player) {
    if (!player.persistentData.get('sacrifice')) return
    if (player.persistentData.getBoolean('sacrifice_active') != true) {
        player.persistentData.putBoolean('sacrifice_active', true)
        player.tell(Text.green('§4Sacrifice §aactivated'))
    } else {
        player.persistentData.putBoolean('sacrifice_active', false)
        player.tell(Text.red('§4Sacrifice §cdeactivated'))
    }
}


/*
let sacrifice_cooldown = new WeakMap()
PlayerEvents.tick(event => {
    let player = event.player
    if (!player.persistentData.get('sacrifice')) return
    if (!player.tags.contains('sacrifice_cooldown')) return
    let level = player.persistentData.sacrifice
    let cd = 3600-(300*(level-1))
    if (sacrifice_cooldown[player.username] == undefined) {
        sacrifice_cooldown[player.username] = cd
    } else {
        sacrifice_cooldown[player.username]--
    }
    if (sacrifice_cooldown[player.username] <= 0) {
        player.tags.remove('sacrifice_cooldown')
        sacrifice_cooldown[player.username] = cd
    }
    
})
*/



function essence_of_giants_ability(player) {
    if (!player.persistentData.get('essence_of_giants')) return
    if (!player.tags.contains('essence_of_giants_cooldown')) {
        let level = player.persistentData.essence_of_giants
        Utils.server.runCommandSilent(`/scale delay set pehkui:height 20 ${player.username}`)
        Utils.server.runCommandSilent(`/scale delay set pehkui:width 20 ${player.username}`)
        Utils.server.runCommandSilent(`/scale delay set pehkui:base 3 ${player.username}`)
        Utils.server.runCommandSilent(`/scale set pehkui:base 2 ${player.username}`)
        let amp = level - 1
        let dur = 200 + ((level-1) * 100); // 200 ticks = 10 seconds, each level increases by 20 ticks
        player.potionEffects.add('minecraft:strength', dur, amp)
        player.potionEffects.add('minecraft:resistance', dur, amp)
        Utils.server.runCommandSilent(`/execute at ${player.username} run playsound irons_spellbooks:cast.generic.nature master ${player.username} ~ ~ ~ 100`)
        player.tags.add('essence_of_giants_cooldown')
        Utils.server.scheduleInTicks(dur, () => {
            Utils.server.runCommandSilent(`/scale reset ${player.username}`)
        })
    } else {
        Utils.server.runCommandSilent(`/scale reset ${player.username}`)

    }
}


let essence_of_giants_cooldown = new WeakMap()
PlayerEvents.tick(event => {
    let player = event.player
    if (!player.persistentData.get('essence_of_giants')) return
    if (!player.tags.contains('essence_of_giants_cooldown')) return
    let level = player.persistentData.essence_of_giants
    let cd = 2400-(300*(level-1))
    if (essence_of_giants_cooldown[player.username] == undefined) {
        essence_of_giants_cooldown[player.username] = cd
    } else {
        essence_of_giants_cooldown[player.username]--
    }
    if (essence_of_giants_cooldown[player.username] <= 0) {
        player.tags.remove('essence_of_giants_cooldown')
        essence_of_giants_cooldown[player.username] = cd
    }
    
})



function essence_of_eternity_ability(player) {
    if (!player.persistentData.get('essence_of_eternity')) return;
    if (isSkillCoolingDown(player, 'essence_of_eternity')) return;
    let level = player.persistentData.essence_of_eternity;
    let box = AABB.of(
        player.x + 10,
        player.y + 10,
        player.z + 10,
        player.x - 10,
        player.y - 10,
        player.z - 10
    );
    let entitiesWithin = player.level.getEntitiesWithin(box).filter(ent => ent.alive && !ent.tags.contains('boss') && !isAlly(player, ent) && ent.pickable);
    if (entitiesWithin.length === 0) return;
    let ticks = 200 + ((level-1) * 100); // 200 ticks = 10 seconds, each level increases by 100 ticks
    let tickRate = 5 - level
    entitiesWithin.forEach(ent => {
        //let uuid = ent.uuid;
        applyEffect(ent, 'minecraft:slowness', ticks, level-1, 'essence_of_eternity', player)
        applyEffect(ent, 'minecraft:weakness', ticks, level-1, 'essence_of_eternity', player)
        //Utils.server.runCommandSilent(`/setTickrate entity ${uuid} ${tickRate}`);
        Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run particle alexscaves:hazmat_breathe ${ent.x} ${ent.y+1} ${ent.z} 0.2 0.2 0.2 1 200`);
        /**
         *         Utils.server.scheduleInTicks(ticks, () => {
            Utils.server.runCommandSilent(`/setTickrate entity ${uuid} 20`);
        });
         * 
         * 
         */

    });
    applyEffect(player, 'irons_spellbooks:hastened', ticks, level-1, 'essence_of_eternity', player)
    let cooldown = 2400 - (300 * (level - 1)) // Cooldown is 3 minutes - 15 seconds per level
    addSkillCooldown(player, 'essence_of_eternity', cooldown);
}


/*
let essence_of_eternity_cooldown =new WeakMap()
PlayerEvents.tick(event => {
    if (!event.player.tags.contains('essence_of_eternity_cooldown')) return
    let level = event.player.persistentData.essence_of_eternity
    let cd = 2400 - (300 * (level - 1)) // Cooldown is 3 minutes - 15 seconds per level
    if (essence_of_eternity_cooldown[event.player.username] == undefined) {
        essence_of_eternity_cooldown[event.player.username] = cd
    } else {
        essence_of_eternity_cooldown[event.player.username] -= 1
    }
    if (essence_of_eternity_cooldown[event.player.username] <= 0) {
        event.player.tags.remove('essence_of_eternity_cooldown')
        essence_of_eternity_cooldown[event.player.username] = cd
    }
})
*/

