// Monk subclass abilities
// Core mechanic: building and spending Chi (player.persistentData.monk_chi)
// for offense, defense, and mobility.

// ---------------------------------------------------------------------------
// Shared Chi helpers
// ---------------------------------------------------------------------------


/**
 * function getMonkChi(player) {
    if (!player || !player.persistentData) return 0
    let value = player.persistentData.monk_chi
    if (!value || value < 0) value = 0
    return value
}

function setMonkChi(player, value) {
    if (!player || !player.persistentData) return
    let newVal = value
    if (newVal < 0) newVal = 0
    player.persistentData.putInt('monk_chi', newVal)
}

// Called any time we *lose* Chi so Inner Calm can react.
function monkHandleChiLoss(player, lostAmount) {
    if (!player || !player.persistentData) return
    if (!player.persistentData.get('inner_calm')) return
    if (lostAmount <= 0) return

    let level = player.persistentData.inner_calm
    if (!level || level <= 0) return

    // Base: small regen when you lose at least 2 Chi at once
    if (lostAmount >= 2) {
        let regenDuration = 40 + 20 * level // 2–6 seconds
        applyEffect(player, 'minecraft:regeneration', regenDuration, 0, 'inner_calm', player)

        // Extra instant heal if low HP
        if (level >= 2 && player.health <= player.maxHealth * 0.5) {
            let heal = player.maxHealth * 0.02 * lostAmount
            player.health = Math.min(player.maxHealth, player.health + heal)
        }
    }

    // Cleanse slowness / weakness at higher levels
    if (level >= 3) {
        let cleanseList = ['minecraft:slowness', 'minecraft:weakness']
        cleanseList.forEach(eff => {
            if (player.potionEffects.isActive(eff)) {
                Utils.server.runCommandSilent(`/effect clear ${player.uuid} ${eff}`)
            }
        })
    }

    // Share regen with nearby allies
    if (level >= 4) {
        let radius = 4 + level
        let box = AABB.of(
            player.x + radius, player.y + 2, player.z + radius,
            player.x - radius, player.y - 2, player.z - radius
        )
        let dim = player.level
        let allies = dim.getEntitiesWithin(box).filter(ent => ent.alive && (!isAlly || isAlly(player, ent)))
        allies.forEach(ent => {
            if (!ent.player) return
            applyEffect(ent, 'minecraft:regeneration', 60, 0, 'inner_calm', player)
        })
    }

    // At max rank: brief resistance to you + allies if you shed lots of Chi at once
    if (level >= 5 && lostAmount >= 4) {
        let radius = 5 + level
        let box = AABB.of(
            player.x + radius, player.y + 2, player.z + radius,
            player.x - radius, player.y - 2, player.z - radius
        )
        let dim = player.level
        let allies = dim.getEntitiesWithin(box).filter(ent => ent.alive && (!isAlly || isAlly(player, ent)))
        allies.forEach(ent => {
            if (!ent.player) return
            applyEffect(ent, 'minecraft:resistance', 100, 0, 'inner_calm', player)
        })
    }
}

// Generic helper to adjust Chi and route loss to Inner Calm.
function monkModifyChi(player, delta, source) {
    if (!player || !player.persistentData) return
    let current = getMonkChi(player)
    let next = current + delta
    if (next < 0) next = 0
    let lost = current - next
    setMonkChi(player, next)
    if (lost > 0) monkHandleChiLoss(player, lost)
}


// ---------------------------------------------------------------------------
// Flowing Fist – Chi generator
// ---------------------------------------------------------------------------
/**
 * Flowing Fist
 * - Melee hits on enemies grant Chi stacks (stored on player.persistentData.monk_chi).
 * - At max Chi, you gain short Haste (and later Speed).

EntityEvents.hurt(event => {
    if (!event.source.player) return
    let player = event.source.player
    if (!player.persistentData.get('flowing_fist')) return
    if (!event.entity.isMonster()) return
    if (typeof isAlly === 'function' && isAlly(player, event.entity)) return

    if (typeof determineMethod === 'function') {
        let method = determineMethod(event.source)
        if (method !== 'melee') return
    }

    let level = player.persistentData.flowing_fist
    if (!level || level <= 0) return

    let maxChiByLevel = {
        1: 4,
        2: 5,
        3: 6,
        4: 7,
        5: 8
    }
    let maxChi = maxChiByLevel[level] || 4
    let currentChi = getMonkChi(player)
    if (currentChi < maxChi) {
        monkModifyChi(player, 1, 'flowing_fist')
        currentChi = getMonkChi(player)
    }

    // At max Chi, grant short burst of Haste (+Speed at higher levels)
    if (currentChi >= maxChi) {
        let hasteAmp = level >= 4 ? 1 : 0
        applyEffect(player, 'minecraft:haste', 60, hasteAmp, 'flowing_fist', player)
        if (level >= 2) {
            applyEffect(player, 'minecraft:speed', 60, 0, 'flowing_fist', player)
        }
    }
})


// ---------------------------------------------------------------------------
// Iron Body – defensive passive + cheat‑death
// ---------------------------------------------------------------------------

/**
 * Iron Body – baseline resistance while you have Chi.
 * - Every second, if you have Chi, you gain short Resistance that scales loosely with Chi and level.

PlayerEvents.tick(event => {
    if (event.server.tickCount % 20 !== 0) return
    let player = event.player
    if (!player.persistentData.get('iron_body')) return

    let level = player.persistentData.iron_body
    if (!level || level <= 0) return

    let chi = getMonkChi(player)
    if (chi <= 0) return

    // Basic resistance when you have at least 2 Chi; stronger at high Chi + level
    let amp = 0
    if (chi >= 2) amp = 0
    if (chi >= 4 && level >= 3) amp = 1

    applyEffect(player, 'minecraft:resistance', 60, amp, 'iron_body', player)
})


/**
 * Iron Body – on‑hit logic:
 * - Heavy hits consume Chi and trigger Inner Calm.
 * - At higher levels, a lethal hit can be converted into a cheat‑death, consuming all Chi instead.
 
EntityEvents.hurt(event => {
    if (!event.entity.player) return
    let player = event.entity.player
    if (!player.persistentData.get('iron_body')) return

    let level = player.persistentData.iron_body
    if (!level || level <= 0) return

    let chi = getMonkChi(player)
    if (chi <= 0) return

    let damage = event.damage
    let maxHp = player.maxHealth

    // Cheat‑death: if this hit would kill you and you have enough Chi, consume all Chi and prevent death.
    if (level >= 4 && chi >= 3 && (player.health - damage) <= 0) {
        if (typeof isSkillCoolingDown !== 'function' || !isSkillCoolingDown(player, 'iron_body_cheat')) {
            event.cancel()
            setMonkChi(player, 0)
            player.health = 1
            if (typeof addSkillCooldown === 'function') {
                addSkillCooldown(player, 'iron_body_cheat', 3600) // ~90s
            }
            applyEffect(player, 'minecraft:resistance', 120, 1, 'iron_body', player)
            applyEffect(player, 'minecraft:regeneration', 120, 0, 'iron_body', player)
            return
        }
    }

    // Heavy hits (25%+ of max HP) consume 1 Chi and feed Inner Calm
    if (level >= 2 && damage >= maxHp * 0.25) {
        monkModifyChi(player, -1, 'iron_body_hit')
    }
})


// ---------------------------------------------------------------------------
// Tempest Step – active mobility / spender
// ---------------------------------------------------------------------------
/**
 * Tempest Step – Active ability
 * - Consumes Chi to damage and displace nearby enemies, granting mobility buffs.
 * - More Chi spent → larger radius and stronger debuffs.
 *
 * Triggered via ability hotkey: tempest_step_ability
 
function tempest_step_ability(player) {
    if (!player.persistentData.get('tempest_step')) return
    if (typeof isSkillCoolingDown === 'function' && isSkillCoolingDown(player, 'tempest_step')) return

    let level = player.persistentData.tempest_step
    if (!level || level <= 0) return

    let chi = getMonkChi(player)
    if (chi <= 0) return

    let maxSpend = level >= 3 ? 5 : 3
    let spendChi = Math.min(chi, maxSpend)
    if (spendChi <= 0) return

    monkModifyChi(player, -spendChi, 'tempest_step')

    let radius = 3 + spendChi
    let box = AABB.of(
        player.x + radius, player.y + 1.5, player.z + radius,
        player.x - radius, player.y - 1.5, player.z - radius
    )
    let dim = player.level
    let enemies = dim.getEntitiesWithin(box).filter(ent => ent.isMonster() && !ent.tags.contains('boss') && ent.isAlive && ent.pickable && (!isAlly || !isAlly(player, ent)))

    enemies.forEach(ent => {
        let damage = 1 + level + 0.5 * spendChi
        ent.attack(damage)

        // Light knockback away from the player using push loops (matches other class scripts)
        let pushes = 10 + 5 * spendChi
        for (let i = 0; i < pushes; i++) {
            player.push(ent)
        }

        if (level >= 3 && spendChi >= 4) {
            applyEffect(ent, 'minecraft:slowness', 60, 0, 'tempest_step', player)
            applyEffect(ent, 'minecraft:weakness', 60, 0, 'tempest_step', player)
        }

        Utils.server.runCommandSilent(
            `/execute in ${dim.dimension} run particle irons_spellbooks:shockwave ${ent.x} ${ent.y} ${ent.z} 0.3 0.3 0.3 0.1 10`
        )
    })

    // Self mobility buffs
    applyEffect(player, 'minecraft:speed', 60 + 20 * spendChi, level >= 4 ? 1 : 0, 'tempest_step', player)

    // Level 5: leave behind a brief "afterimage" that speeds allies and slows enemies
    if (level >= 5) {
        let afterRadius = 3
        let ax = player.x
        let ay = player.y
        let az = player.z
        Utils.server.scheduleInTicks(5, () => {
            let boxA = AABB.of(
                ax + afterRadius, ay + 1.5, az + afterRadius,
                ax - afterRadius, ay - 1.5, az - afterRadius
            )
            let ents = player.level.getEntitiesWithin(boxA)
            ents.forEach(ent => {
                if (ent.player && (!isAlly || isAlly(player, ent))) {
                    applyEffect(ent, 'minecraft:speed', 60, 0, 'tempest_step_afterimage', player)
                } else if (ent.isMonster()) {
                    applyEffect(ent, 'minecraft:slowness', 60, 0, 'tempest_step_afterimage', player)
                }
            })
        })
    }

    // Cooldown: ~8s → 5s
    let cdMap = {
        1: 160,
        2: 140,
        3: 120,
        4: 100,
        5: 100
    }
    if (typeof addSkillCooldown === 'function') {
        addSkillCooldown(player, 'tempest_step', cdMap[level] || 160)
    }

    Utils.server.runCommandSilent(
        `/execute in ${player.level.dimension} run playsound irons_spellbooks:cast.air_burst ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.8`
    )
}


// ---------------------------------------------------------------------------
// Harmony Strike – big Chi spender / finisher
// ---------------------------------------------------------------------------
/**
 * Harmony Strike – Active ability
 * - Consumes all current Chi for a powerful palm strike around you.
 * - Damage, healing, and buffs scale with Chi spent and level.
 *
 * Triggered via ability hotkey: harmony_strike_ability
 
function harmony_strike_ability(player) {
    if (!player.persistentData.get('harmony_strike')) return
    if (typeof isSkillCoolingDown === 'function' && isSkillCoolingDown(player, 'harmony_strike')) return

    let level = player.persistentData.harmony_strike
    if (!level || level <= 0) return

    let chi = getMonkChi(player)
    if (chi < 3) return // require at least a small build‑up

    let spendChi = chi
    monkModifyChi(player, -spendChi, 'harmony_strike')

    let radius = 3 + Math.floor(spendChi / 2)
    let box = AABB.of(
        player.x + radius, player.y + 2, player.z + radius,
        player.x - radius, player.y - 2, player.z - radius
    )
    let dim = player.level
    let enemies = dim.getEntitiesWithin(box).filter(ent => ent.isMonster() && !ent.tags.contains('boss') && ent.isAlive && ent.pickable && (!isAlly || !isAlly(player, ent)))

    enemies.forEach(ent => {
        let damage = 2 + level + spendChi * 0.7
        ent.attack(damage)
        Utils.server.runCommandSilent(
            `/execute in ${dim.dimension} run particle irons_spellbooks:shockwave ${ent.x} ${ent.y + 0.5} ${ent.z} 0.4 0.4 0.4 0.15 20`
        )

        if (level >= 2 && spendChi >= 5) {
            applyEffect(ent, 'minecraft:weakness', 80, 0, 'harmony_strike', player)
        }
    })

    // Self heal based on Chi spent
    let healPerChi = level >= 2 ? 0.03 : 0.02
    let healAmount = player.maxHealth * healPerChi * spendChi
    player.health = Math.min(player.maxHealth, player.health + healAmount)

    // Allies gain Strength at higher levels
    if (level >= 3) {
        let allyRadius = radius
        let boxAllies = AABB.of(
            player.x + allyRadius, player.y + 2, player.z + allyRadius,
            player.x - allyRadius, player.y - 2, player.z - allyRadius
        )
        let allies = dim.getEntitiesWithin(boxAllies).filter(ent => ent.player && (!isAlly || isAlly(player, ent)))
        allies.forEach(ent => {
            applyEffect(ent, 'minecraft:strength', 80, level >= 5 ? 1 : 0, 'harmony_strike', player)
        })
    }

    // Defensive boons for the Monk at higher levels
    if (level >= 4) {
        applyEffect(player, 'minecraft:resistance', 120, 0, 'harmony_strike', player)
        applyEffect(player, 'minecraft:regeneration', 120, 0, 'harmony_strike', player)
    }

    // Cooldown: ~20s scaling down slightly with level
    let cdMap = {
        1: 400,
        2: 360,
        3: 340,
        4: 320,
        5: 300
    }
    if (typeof addSkillCooldown === 'function') {
        addSkillCooldown(player, 'harmony_strike', cdMap[level] || 400)
    }

    Utils.server.runCommandSilent(
        `/execute in ${player.level.dimension} run playsound irons_spellbooks:cast.shockwave ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 0.9`
    )
}

 * 
 * 
 * 
 */
