/**
 * Abilities: 
 *  - Dragonlord: Summons a dragon that will fight for you. 3 levels
 *  §aLevel 1: Firebound.) Summons a Fire Dragon
 *  §aLevel 2: Icebound.) Summons an Ice Dragon
 *  §aLevel 3: Stormbound.) Summons a Lightning Dragon
 * 
 *  - Hardened Scales: Increases the health of your dragon. 20 levels
 *  §aLevel 1: 20 health. Increases by 5 each level
 *  §aLevel 9: 60 health
 *  §aLevel 10: 75 health. Increases by 5 each level
 * 
 * 
 * - Evolution: increases the dragons age: 20 levels
 * §aLevel 1: 0
 * §aLevel 2: 100000
 * §aLevel 3: 200000
 * §aLevel 4: 300000
 * §aLevel 5: 400000
 * §aLevel 6: 500000
 * §aLevel 7: 600000
 * §aLevel 8: 700000
 * §aLevel 9: 800000
 * §aLevel 10: 900000
 * §aLevel 11: 1000000
 * §aLevel 12: 1100000
 * §aLevel 13: 1200000
 * §aLevel 14: 1300000
 * §aLevel 15: 1400000
 * §aLevel 16: 1500000
 * §aLevel 17: 1600000
 * §aLevel 18: 1700000
 * §aLevel 19: 1800000
 * §aLevel 20: 1900000
 * 
 * 
 * Skyborn Arcane: Lets your dragon cast spells depending on its type. 5 levels
 * §aLevel 1: All dragons can cast stomp as a defensive spell
 * §aLevel 2: 
 *  - Ice Dragon: Ray of Frost as an offensive spell
 *  - Fire Dragon: Magma Bomb as an offensive spell
 *  - Lightning Dragon: Ball Lightning as an offensive spell
 * §aLevel 3:
 * - Ice Dragon: Ice Block as an offensive spell 
 * - Fire Dragon: Scorch as an offensive spell
 * - Lightning Dragon: Lightning Lance as an offensive spell
 * §aLevel 4:
 * - Ice Dragon: Frostwave as a defensive spell
 * - Fire Dragon: Heat Surge as a defensive spell
 * - Lightning Dragon: Shockwave as a defensive spell
 * §aLevel 5:
 * - Ice Dragon: Oakskin as a defensive spell
 * - Fire Dragon: Oakskin as a defensive spell
 * - Lightning Dragon: Oakskin as a defensive spell
 * 
 * 
 * 
 * 
 * Armor Abilities:
 *  *  §aLevel 1: Iron Armor
 *  §aLevel 2: Copper Armor
 *  §aLevel 3: Gold Armor
 *  §aLevel 4: Silver Armor
 * 
 * 
 * - Draconic Crown: Gives your dragon Head armor. 4 levels
 * 
 * - Draconic Amulet: Gives your dragon Neck armor. 4 levels
 * 
 * - Draconic Bulwark: Gives your dragon Body armor. 4 levels
 * 
 * - Draconic Tailguard: Gives your dragon Tail armor. 4 levels
 * 
 * 
 * 
 * 
 * Bound Might: Gives your dragon effects while you are riding it. 3 levels
 * §aLevel 1: Strength
 * §aLevel 2: Speed
 * §aLevel 3: Soulsteal
 * 
 * 
 * Molten Scales: Gives your dragon a permanent burning thorns effect. 3 levels.
 *  - Each level increases the amplifier of the effect by 1
 * 
 * 
 * 
 * Overlord: Increases the number of dragons you can summon. 1 Level. 
 *  - §aLevel 1: 2 dragons
 * 
 */

//____________________________________________________________________________________________________
let dragonlord_abilities = {
    dragontamer: {
        firebound: 'iceandfire:fire_dragon',
        icebound: 'iceandfire:ice_dragon',
        stormbound: 'iceandfire:lightning_dragon'
    },
    evolution: {
        1: 0,
        2: 100000,
        3: 200000,
        4: 300000,
        5: 400000,
        6: 500000,
        7: 600000,
        8: 700000,
        9: 800000,
        10: 900000,
        11: 1000000,
        12: 1100000,
        13: 1200000,
        14: 1300000,
        15: 1400000,
        16: 1500000,
        17: 1600000,
        18: 1700000,
        19: 1800000,
        20: 1900000,
    },
    skyborn_arcane: {
        1: {
            ice_dragon: {
                defense: ['stomp']
            },
            fire_dragon: {
                defense: ['stomp']
            },
            lightning_dragon: {
                defense: ['stomp']
            },
        },
        2: {
            ice_dragon: {
                defense: ['stomp'],
                offense: ['ray_of_frost']
            },
            fire_dragon: {
                defense: ['stomp'],
                offense: ['magma_bomb']
            },
            lightning_dragon: {
                defense: ['stomp'],
                offense: ['ball_lightning']
            },
        },
        3: {
            ice_dragon: {
                defense: ['stomp'],
                offense: ['ray_of_frost', 'ice_block']
            },
            fire_dragon: {
                defense: ['stomp'],
                offense: ['magma_bomb', 'scorch']
            },
            lightning_dragon: {
                defense: ['stomp'],
                offense: ['ball_lightning', 'lightning_lance']
            },
        },
        4: {
            ice_dragon: {
                defense: ['stomp', 'frostwave'],
                offense: ['ray_of_frost', 'ice_block']
            },
            fire_dragon: {
                defense: ['stomp', 'heat_surge'],
                offense: ['magma_bomb', 'scorch']
            },
            lightning_dragon: {
                defense: ['stomp', 'shockwave'],
                offense: ['ball_lightning', 'lightning_lance']
            },
        },
        5: {
            ice_dragon: {
                defense: ['stomp', 'frostwave', 'oakskin'],
                offense: ['ray_of_frost', 'ice_block']
            },
            fire_dragon: {
                defense: ['stomp', 'heat_surge', 'oakskin'],
                offense: ['magma_bomb', 'scorch']
            },
            lightning_dragon: {
                defense: ['stomp', 'shockwave', 'oakskin'],
                offense: ['ball_lightning', 'lightning_lance']
            },
        },
    },

    // Stomp: 0.5 cast time,
    // Ray of Frost: instant
    // MAgma Bomb: 1 second cast time
    // Ball Lightning: Instant
    // Ice Block: 1.5 seconds cast time
    // Scorch: 1 second cast time
    // Lightning Lance: 2 second cast time
    hardened_scales: { // increases health by 10 each level. 20 levels
        1: 15,
        2: 25,
        3: 35,
        4: 45,
        5: 55,
        6: 65,
        7: 75,
        8: 85,
        9: 95,
        10: 105,
        11: 115,
        12: 125,
        13: 135,
        14: 145,
        15: 155,
        16: 165,
        17: 175,
        18: 185,
        19: 195,
        20: 205,
    },
    draconic_crown: {
        1: 'iceandfire:dragonarmor_iron_head',
        2: 'iceandfire:dragonarmor_copper_head',
        3: 'iceandfire:dragonarmor_gold_head',
        4: 'iceandfire:dragonarmor_silver_head',
    },
    draconic_amulet: {
        1: 'iceandfire:dragonarmor_iron_neck',
        2: 'iceandfire:dragonarmor_copper_neck',
        3: 'iceandfire:dragonarmor_gold_neck',
        4: 'iceandfire:dragonarmor_silver_neck',
    },
    draconic_bulwark: {
        1: 'iceandfire:dragonarmor_iron_body',
        2: 'iceandfire:dragonarmor_copper_body',
        3: 'iceandfire:dragonarmor_gold_body',
        4: 'iceandfire:dragonarmor_silver_body',
    },
    draconic_tailguard: {
        1: 'iceandfire:dragonarmor_iron_tail',
        2: 'iceandfire:dragonarmor_copper_tail',
        3: 'iceandfire:dragonarmor_gold_tail',
        4: 'iceandfire:dragonarmor_silver_tail',
    },

    bound_might: {
        1: 'minecraft:strength',
        2: 'minecraft:speed',
        3: 'alexsmobs:soulsteal',
    },

    molten_scales: {
        1: 'runiclib:burning_thorns',
        2: 'runiclib:burning_thorns',
        3: 'runiclib:burning_thorns',
    },
}

//____________________________________________________________________________________________________
// ----------------------------------------------
// Shared helpers for Dragonlord ability handling
// ----------------------------------------------
const DRAGON_BOUND_TAG = 'bound_dragon'
const DRAGON_OWNER_PREFIX = 'Owner:'

/**
 * Returns true if the entity is one of our managed bound dragons.
 */
function dl_isBoundDragon(entity) {
    return entity && entity.tags && entity.tags.contains(DRAGON_BOUND_TAG)
}

/**
 * Resolves the owning player for a bound dragon, if they are online.
 */
function dl_getOwnerPlayer(dragon, server) {
    if (!dl_isBoundDragon(dragon)) return null
    try {
        let ownerTag = dragon.tags.find(tag => tag.startsWith(DRAGON_OWNER_PREFIX))
        if (!ownerTag) return null
        let ownerName = ownerTag.substring(DRAGON_OWNER_PREFIX.length)
        if (!ownerName) return null
        return server.getPlayer(ownerName)
    } catch (e) {
        return null
    }
}

/**
 * Retrieves the Skyborn Arcane config for the owner's level and dragon type.
 */
function dl_getSkybornLoadout(owner, dragonTypeKey) {
    if (!owner || !owner.persistentData || !owner.persistentData.get('skyborn_arcane')) return null
    let lvl = owner.persistentData.skyborn_arcane
    let tier = dragonlord_abilities.skyborn_arcane[lvl]
    if (!tier) return null
    return tier[dragonTypeKey] || null
}

/**
 * Lightweight utility to schedule and clear cooldown flags on the owner.
 */
function dl_startCooldown(owner, key, ticks) {
    if (!owner || !owner.persistentData) return
    owner.persistentData.putBoolean(key, true)
    Utils.server.scheduleInTicks(Math.max(1, ticks), () => {
        if (owner && owner.persistentData) owner.persistentData.remove(key)
    })
}

function dl_onCooldown(owner, key) {
    return owner && owner.persistentData && owner.persistentData.get(key)
}

/**
 * Convenience distance-squared helper that tolerates unloaded positions.
 */
function dl_distanceSq(a, b) {
    try {
        let dx = (a.x || 0) - (b.x || 0)
        let dy = (a.y || 0) - (b.y || 0)
        let dz = (a.z || 0) - (b.z || 0)
        return dx * dx + dy * dy + dz * dz
    } catch (e) {
        return Infinity
    }
}

function dl_uuidSelector(uuid) {
    return `@e[limit=1,uuid=${uuid}]`
}

function dl_runServerCommand(command) {
    try {
        Utils.server.runCommandSilent(command)
    } catch (e) {
        // swallow: commands fail silently in tight loops, which is acceptable here
    }
}

//____________________________________________________________________________________________________
/**
 * Returns the player's currently bound dragon entity if it is loaded nearby.
 */
function dl_getBoundDragon(player) {
    if (!player || !player.persistentData || !player.persistentData.get('dragon_id')) return null
    try {
        let entity = player.level.getEntity(player.persistentData.dragon_id)
        return dl_isBoundDragon(entity) ? entity : null
    } catch (e) {
        return null
    }
}

//____________________________________________________________________________________________________
// ----------------------------------------------
// Dragonlord: Dragon's Aegis and Draconic Cadence
// ----------------------------------------------

// Utility: map IAF dragon type id to shorthand used in skyborn_arcane
function dl_getDragonTypeKey(dragon) {
    try {
        let t = dragon.type || ''
        if (t.endsWith('fire_dragon')) return 'fire_dragon'
        if (t.endsWith('ice_dragon')) return 'ice_dragon'
        if (t.endsWith('lightning_dragon')) return 'lightning_dragon'
    } catch (e) {}
    return null
}

// Dragon's Aegis handler (see aggregated hurt event below)
function dl_handleDragonsAegis(event) {
    if (!dl_isBoundDragon(event.entity)) return
    let dragon = event.entity
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner || !owner.persistentData.get('dragons_aegis')) return

    let lvl = owner.persistentData.dragons_aegis
    if (!lvl || lvl < 1) return

    let aegisKey = `aegis_cd_${dragon.uuid}`
    if (dl_onCooldown(owner, aegisKey)) return

    let threshold = 0.5
    let cooldownTicks = 20 * 90
    if (lvl >= 2) { threshold = 0.35; cooldownTicks = 20 * 75 }
    if (lvl >= 3) { threshold = 0.20; cooldownTicks = 20 * 60 }

    let ratio
    try {
        ratio = (dragon.health || 0) / (dragon.maxHealth || 1)
    } catch (e) {
        ratio = 1
    }
    if (ratio > threshold) return

    dl_startCooldown(owner, aegisKey, cooldownTicks)

    let dragonKey = dl_getDragonTypeKey(dragon)
    try { castOnceLong(dragon, 'oakskin', dragon, Math.min(1 + lvl, 3), 2) } catch (e) {}
    let elemDef = null
    if (dragonKey == 'ice_dragon') elemDef = 'frostwave'
    if (dragonKey == 'fire_dragon') elemDef = 'heat_surge'
    if (dragonKey == 'lightning_dragon') elemDef = 'shockwave'
    if (elemDef) {
        try { castOnceLong(dragon, elemDef, dragon, Math.min(1 + lvl, 3), 2) } catch (e) {}
    }

    try {
        let bx = dragon.x, by = dragon.y, bz = dragon.z
        let box = AABB.of(bx + 6, by + 3, bz + 6, bx - 6, by - 3, bz - 6)
        let ents = dragon.level.getEntitiesWithin(box).filter(e => e.isMonster() && !e.tags.contains('boss'))
        ents.forEach(e => e.potionEffects.add('minecraft:slowness', 60, 0, true, true))
    } catch (e) {}
}

//____________________________________________________________________________________________________
// Draconic Cadence
// - Maintain proximity to your bound dragon to build stacks for both rider and dragon
// - Each stack grants small mobility/attack cadence; decays when separated
// Levels: max stacks 5/6/7/8; build every 10s in range (<=16 blocks); decay 1 stack per 3s out of range
// Tracks cadence build/decay state per-player UUID

/**
 * let dl_cadence_state = new Map()
PlayerEvents.tick(event => {
    let player = event.player
    if (!player || !player.persistentData || !player.persistentData.get('draconic_cadence')) return
    let lvl = player.persistentData.draconic_cadence
    if (!lvl || lvl < 1) return

    let dragon = dl_getBoundDragon(player)
    if (!dragon) return

    // Init state
    let cadenceKey = player.uuid
    if (!dl_cadence_state.has(cadenceKey)) {
        dl_cadence_state.set(cadenceKey, { build: 0, decay: 0, stacks: 0, lastStacks: 0 })
    }
    let st = dl_cadence_state.get(cadenceKey)

    let maxStacks = 5 + Math.min(lvl - 1, 3) // 5/6/7/8
    let inRange = false
    try {
        let dx = dragon.x - player.x
        let dy = dragon.y - player.y
        let dz = dragon.z - player.z
        inRange = (dx*dx + dy*dy + dz*dz) <= (16*16)
    } catch (e) { inRange = false }

    if (inRange) {
        st.build++
        st.decay = 0
        if (st.build >= 200) { // 10s
            st.build = 0
            if (st.stacks < maxStacks) st.stacks++
        }
    } else {
        st.build = 0
        st.decay++
        if (st.decay >= 60) { // lose 1 stack per 3s apart
            st.decay = 0
            if (st.stacks > 0) st.stacks--
        }
    }

    // Apply/refresh short buffs based on stacks (lightweight and stacks map -> small amplifiers)
    if (st.stacks > 0) {
        // Move speed: Speed effect (refreshed)
        let spdAmp = Math.min(Math.floor((st.stacks - 1) / 2), 2) // 0..2
        player.potionEffects.add('minecraft:speed', 60, spdAmp, true, true)
        dragon.potionEffects.add('minecraft:speed', 60, spdAmp, true, true)

        // Attack cadence: CoFH Celerity (attack speed) if available in pack
        try {
            let atkAmp = Math.min(Math.floor((st.stacks - 1) / 2), 2)
            player.potionEffects.add('cofh_core:celerity', 60, atkAmp, true, true)
            dragon.potionEffects.add('cofh_core:celerity', 60, atkAmp, true, true)
        } catch (e) {}

        // Visual hint every 5s on stack changes
        if (st.build === 0 && inRange && st.lastStacks != st.stacks) {
            st.lastStacks = st.stacks
            try {
                Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run particle irons_spellbooks:shockwave ${player.x} ${player.y + 1} ${player.z} 0 0 0 0 1`)
            } catch (e) {}
        }
    } else {
        st.lastStacks = 0
    }
})
 * 
 * 
 * 
 */


//____________________________________________________________________________________________________
// -------------------------------------------------
// Dragonlord: Searing Scale Veil (formerly: Charred Scale Sheen)
// Levels: 1-3
// - While the player is riding their bound dragon:
//   - Rider gains Fire Resistance (refreshed)
//   - Dragon gains Burning Thorns with amplifier = level-1 (refreshed)
// - Synergy: If an attacker hitting the dragon has Scorch, deal extra fire retaliation damage based on level
// persistentData key: 'searing_scale_veil'

// Maintain on-mount buffs by polling rider state


/**
 * 
 * PlayerEvents.tick(event => {
    if (event.server.tickCount % 2 == 0) return // 
    let player = event.player
    if (!player || !player.persistentData || !player.persistentData.get('searing_scale_veil')) return
    let lvl = player.persistentData.searing_scale_veil
    if (!lvl || lvl < 1) return

    // Check riding and mount is bound dragon
    let mount = player.vehicle.rootVehicle
    if (!mount || !mount.tags || !mount.tags.contains('bound_dragon')) return
    if (mount.type.includes('fire')) {
        player.potionEffects.add('minecraft:fire_resistance', 60, 0, true, true)
    } else if (mount.type.includes('ice')) {
        applyEffect(player, 'cofh_core:cold_resistance', 60, 0, true)
    } else {
        applyEffect(player, 'irons_spellbooks:charged', 60, 0, true)
    }

    // Apply temporary burning thorns to dragon, amplifier scales with level
    let amp = Math.max(0, Math.min(lvl - 1, 2))
    mount.potionEffects.add('runiclib:burning_thorns', 60, amp, true, true)
})
 * 
 */


function dl_handleSearingScaleVeil(event) {
    if (!dl_isBoundDragon(event.entity)) return
    let dragon = event.entity
    let attacker = event.source.actual
    if (!attacker) return
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner || !owner.persistentData.get('searing_scale_veil')) return
    let lvl = owner.persistentData.searing_scale_veil
    if (lvl < 1) return

    try {
        if (!attacker.potionEffects.isActive('irons_spellbooks:scorch')) return
    } catch (e) {
        return
    }

    let dmg = 2 + lvl
    dl_runServerCommand(`/execute in ${dragon.level.dimension} run damage ${dl_uuidSelector(attacker.uuid)} ${dmg} minecraft:fire by ${dl_uuidSelector(dragon.uuid)}`)
}

//____________________________________________________________________________________________________
// Spawns or refreshes the player's bonded dragon based on their unlocks and ability scaling
function dragonSummon(player) {
    // Dragon type is randomly selected from the list if you have the right level
    let summonable_dragons = []
    if (player.persistentData.get(`firebound`)) {
        summonable_dragons.push('iceandfire:fire_dragon')
    }
    if (player.persistentData.get(`icebound`)) {
        summonable_dragons.push('iceandfire:ice_dragon')
    }
    if (player.persistentData.get(`stormbound`)) {
        summonable_dragons.push('iceandfire:lightning_dragon')
    }
    if (summonable_dragons.length == 0) {
        // No unlocks yet; bail gracefully
        player.tell('Your draconic pact is still dormant. Unlock a dragon type first.')
        return
    }
    let dragonType = summonable_dragons[Math.floor(Math.random() * summonable_dragons.length)]

    let age = 1200000
    if (player.persistentData.get(`evolution`)) {
        let evolution = player.persistentData.evolution
        age = dragonlord_abilities.evolution[evolution] || 0
    }

    let health = 80
    if (player.persistentData.get(`hardened_scales`)) {
        let hardened_scales = player.persistentData.hardened_scales
        health = dragonlord_abilities.hardened_scales[hardened_scales] || health
    }

    let variant = Math.floor(Math.random() * 3) + 1
    // Spawn the dragon using the correct method
    let dragon = player.level.createEntity(dragonType)


    dragon.setPosition(player.x, player.y, player.z)
    let previousDragon = dl_getBoundDragon(player)
    if (previousDragon) {
        previousDragon.discard()
    }

    let head_armor = ''
    if (player.persistentData.get(`draconic_crown`)) {
        let draconic_crown = player.persistentData.draconic_crown
        head_armor = dragonlord_abilities.draconic_crown[draconic_crown]
        //Utils.server.tell(`Head Armor: ${head_armor}`)
    }

    let neck_armor = ''
    if (player.persistentData.get(`draconic_amulet`)) {
        let draconic_amulet = player.persistentData.draconic_amulet
        neck_armor = dragonlord_abilities.draconic_amulet[draconic_amulet]
    }

    let body_armor = ''
    if (player.persistentData.get(`draconic_bulwark`)) {
        let draconic_bulwark = player.persistentData.draconic_bulwark
        body_armor = dragonlord_abilities.draconic_bulwark[draconic_bulwark]
    }

    let tail_armor = ''
    if (player.persistentData.get(`draconic_tailguard`)) {
        let draconic_tailguard = player.persistentData.draconic_tailguard
        tail_armor = dragonlord_abilities.draconic_tailguard[draconic_tailguard]
    }

    if (player.persistentData.get('bound_might')) {
        let bound_might = player.persistentData.bound_might
        for (let i = bound_might; i > 0; i--) {
            let eff = dragonlord_abilities.bound_might[i]
            if (eff) {
                dragon.potionEffects.add(eff, 999999999, 0, true, true)
            }
        }
    }

    if (player.persistentData.get(`molten_scales`)) {
        let molten_scales = player.persistentData.molten_scales
        let amplifier = 0
        for (let i = molten_scales; i > 0; i--) {
            amplifier++
        }
        let moltenEffect = dragonlord_abilities.molten_scales[molten_scales]
        if (moltenEffect) {
            dragon.potionEffects.add(moltenEffect, 999999999, amplifier, true, true)
        }
    }
    let armorItems = [head_armor, neck_armor, body_armor, tail_armor]
        .map((armor, index) => armor && armor.length > 0 ? { Slot: index + 1, id: armor, Count: 1 } : null)
        .filter(item => item)

    dragon.mergeNbt({
        AgeTicks: age,
        Owner: player.username,
        Tags: ['tamed_beast', `Owner:${player.username}`, `bound_dragon`],
        Variant: variant,
        Command: 1,
        Items: armorItems
    })

    // Now spawn the dragon into the world
    dragon.spawn();
    dragon.maxHealth = health
    dragon.health = health
    player.persistentData.putString(`dragon_id`, dragon.uuid)
    let maxDragons = 1
    if (player.persistentData.get('overlord')) {
        maxDragons = Math.max(1, Math.min(1 + player.persistentData.overlord, 2))
    }
    player.persistentData.putInt(`max_dragons`, maxDragons)
    player.persistentData.putInt(`dragon_count`, 1)
    dragon.setCustomName(`${player.username}'s Dragon`)
    Utils.server.runCommandSilent(`/scale set pehkui:base 0.25 ${dragon.uuid}`)
}

//____________________________________________________________________________________________________
// -------------------------------------------------
// Drake's Mark (reversed): Dragon marks; you consume
// persistentData key: 'drakes_mark' (level >=1)
// - Dragon hit marks target with Glow for L-scaled duration if not lockout
// - Your next hit on that target consumes mark and applies a random negative effect
// - After applying, that target cannot be marked again for (30 - 2*L)s (min 10s)
// - Negative effect duration and amplifier scale with level

function dl_handleDrakesMarkDragonStrike(event) {
    if (!event.source || !event.source.actual) return
    let dragon = event.source.actual
    if (!dl_isBoundDragon(dragon)) return
    let target = event.entity
    if (!target || target.player || target.tags.contains('boss')) return
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner || !owner.persistentData.get('drakes_mark')) return
    let lvl = owner.persistentData.drakes_mark
    if (lvl < 1) return

    let lockTag = `dm_lock_${owner.uuid}`
    let markTag = `dm_mark_${owner.uuid}`
    if (target.tags.contains(lockTag) || target.tags.contains(markTag)) return

    let glowSeconds = 6 + 2 * lvl
    target.potionEffects.add('minecraft:glowing', 20 * glowSeconds, 0, true, true)
    target.tags.add(markTag)
    Utils.server.scheduleInTicks(20 * glowSeconds, () => {
        if (target && target.isAlive) target.tags.remove(markTag)
    })
}

function dl_handleDrakesMarkPlayerStrike(event) {
    if (!event.source || !event.source.player) return
    let player = event.source.player
    if (!player.persistentData || !player.persistentData.get('drakes_mark')) return
    let lvl = player.persistentData.drakes_mark
    if (lvl < 1) return
    let target = event.entity
    if (!target || target.player) return

    let markTag = `dm_mark_${player.uuid}`
    if (!target.tags.contains(markTag)) return

    target.tags.remove(markTag)
    try { target.potionEffects.remove('minecraft:glowing') } catch (e) {}

    let lockSeconds = Math.max(10, 30 - 2 * lvl)
    let lockTag = `dm_lock_${player.uuid}`
    target.tags.add(lockTag)
    Utils.server.scheduleInTicks(20 * lockSeconds, () => {
        if (target && target.isAlive) target.tags.remove(lockTag)
    })

    let debuffs = [
        { id: 'minecells:bleeding', fallback: 'minecraft:slowness' },
        { id: 'minecraft:poison', fallback: 'minecraft:slowness' },
        { id: 'irons_spellbooks:stun', fallback: 'minecraft:slowness' }
    ]
    let choice = debuffs[Math.floor(Math.random() * debuffs.length)]
    let dur = 20 * (4 + 2 * lvl)
    let amp = Math.min(lvl - 1, 3)
    try {
        target.potionEffects.add(choice.id, dur, amp, true, true)
    } catch (e) {
        target.potionEffects.add(choice.fallback, dur, amp, true, true)
    }
}

// Skyborn Arcane offensive / defensive spell triggers
function dl_handleDragonOffenseSpell(event) {
    let dragon = event.source && event.source.actual
    if (!dl_isBoundDragon(dragon)) return
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner) return
    let dragonKey = dl_getDragonTypeKey(dragon)
    let loadout = dl_getSkybornLoadout(owner, dragonKey)
    if (!loadout || !loadout.offense || loadout.offense.length == 0) return

    let level = owner.persistentData.skyborn_arcane || 0
    let procChance = Math.min(45, 10 + level * 5)
    if (Math.random() * 100 > procChance) return

    let spell = loadout.offense[Math.floor(Math.random() * loadout.offense.length)]
    if (!spell) return
    let spellPower = Math.floor(Math.random() * 3) + 1
    try { castOnceLong(dragon, spell, event.entity, spellPower, 2) } catch (e) {}
}

function dl_handleDragonDefenseSpell(event) {
    if (!dl_isBoundDragon(event.entity)) return
    let dragon = event.entity
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner) return
    let dragonKey = dl_getDragonTypeKey(dragon)
    let loadout = dl_getSkybornLoadout(owner, dragonKey)
    if (!loadout || !loadout.defense || loadout.defense.length == 0) return

    let level = owner.persistentData.skyborn_arcane || 0
    let procChance = Math.min(40, 15 + level * 5)
    if (Math.random() * 100 > procChance) return

    let spell = loadout.defense[Math.floor(Math.random() * loadout.defense.length)]
    if (!spell) return
    let target = event.source && event.source.actual ? event.source.actual : event.entity
    let spellPower = Math.floor(Math.random() * 3) + 1
    try { castOnceLong(dragon, spell, target, spellPower, 2) } catch (e) {}
}

//____________________________________________________________________________________________________
// -------------------------------------------------
// Elemental Tribute: mirror a subset of rider buffs onto dragon
// persistentData key: 'elemental_tribute'
// - Every second, copy up to N positive effects (by whitelist) with reduced amp to bound dragon in 8 block range or while mounted
// Simple per-player throttling so tribute runs once per second
let dl_et_tick = new Map()
const ET_WHITELIST = [
    'minecraft:regeneration',
    'minecraft:strength',
    'minecraft:speed',
    'minecraft:resistance',
    'cofh_core:celerity'
]

/**
 * PlayerEvents.tick(event => {
    let player = event.player
    if (!player) return
    let etKey = player.uuid
    if (!player.persistentData || !player.persistentData.get('elemental_tribute')) {
        dl_et_tick.delete(etKey)
        return
    }
    let lvl = player.persistentData.elemental_tribute
    if (lvl < 1) return

    // Throttle to once per second per player
    let tickCount = dl_et_tick.get(etKey) || 0
    tickCount++
    if (tickCount < 20) {
        dl_et_tick.set(etKey, tickCount)
        return
    }
    dl_et_tick.set(etKey, 0)

    let dragon = dl_getBoundDragon(player)
    if (!dragon) return

    // Must be mounted or close by
    let inRange = false
    try {
        inRange = (player.vehicle == dragon) || (dl_distanceSq(player, dragon) <= 64)
    } catch (e) { inRange = false }
    if (!inRange) return

    // Mirror up to N effects based on level
    let limit = Math.min(1 + Math.floor(lvl / 2), 4)
    let mirrored = 0
    try {
        let active = player.potionEffects.active || []
        for (let eff of active) {
            if (mirrored >= limit) break
            let id = eff.effect || eff.id || eff
            if (!id) continue
            if (!ET_WHITELIST.includes(id.toString())) continue
            let amp = Math.max(0, (eff.amplifier || 0) - 1)
            dragon.potionEffects.add(id, 40, amp, true, true)
            mirrored++
        }
    } catch (e) {}
})
 * 
 * 
 */


//____________________________________________________________________________________________________
// -------------------------------------------------
// Sovereign's Edict: cleanse allies near dragon, grant Resistance if Slowed/Chilled
// persistentData key: 'sovereigns_edict'
// Trigger: Right-click dragon stick (not sneaking)
ItemEvents.firstRightClicked('iceandfire:dragon_stick', event => {
    let player = event.player
    if (!player || player.shiftKeyDown) return
    if (!player.persistentData || !player.persistentData.get('sovereigns_edict')) return
    let cdKey = 'dl_cd_edict'
    if (dl_onCooldown(player, cdKey)) {
        player.tell('Sovereign\'s Edict is still recovering.')
        return
    }

    // Find bound dragon
    let dragon = dl_getBoundDragon(player)
    if (!dragon) return

    let lvl = player.persistentData.sovereigns_edict
    let radius = 6 + lvl // small scaling
    let bx = dragon.x, by = dragon.y, bz = dragon.z
    let box = AABB.of(bx + radius, by + radius, bz + radius, bx - radius, by - radius, bz - radius)
    let ents = dragon.level.getEntitiesWithin(box)

    // Cleanse first negative from allies; grant resistance if they had Slowness/Chilled
    const NEGATIVE = ['minecraft:slowness', 'minecraft:weakness', 'minecraft:poison', 'minecraft:wither', 'cofh_core:chilled', 'minecells:bleeding']
    ents.forEach(ent => {
        if (!ent.player) return
        if (!isAlly || !isAlly(player, ent)) return
        let hadSlow = false
        try {
            if (ent.potionEffects.isActive('minecraft:slowness') || ent.potionEffects.isActive('cofh_core:chilled')) hadSlow = true
            for (let neg of NEGATIVE) {
                if (ent.potionEffects.isActive(neg)) { ent.potionEffects.remove(neg); break }
            }
            if (hadSlow) ent.potionEffects.add('minecraft:resistance', 100 + 20 * lvl, Math.min(0 + Math.floor(lvl / 2), 2), true, true)
        } catch (e) {}
    })

    // Cooldown: 30s - 5s/level (min 10s)
    let cd = Math.max(10, 30 - 5 * lvl)
    dl_startCooldown(player, cdKey, cd * 20)
    try { Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run playsound irons_spellbooks:spell.healing_ward.cast ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.2`) } catch (e) {}
})

//____________________________________________________________________________________________________
// -------------------------------------------------
// Drakekeeper's Respite
// persistentData key: 'drakekeepers_respite'
// - When your bound dragon kills an enemy, heal the owner
// - If the victim was Burning/Chilled/Shocked, grant brief matching defensive boons
EntityEvents.death(event => {
    if (!event.entity || !event.entity.isMonster()) return
    if (!event.source || !event.source.actual) return
    let killer = event.source.actual
    if (!killer.tags || !killer.tags.contains('bound_dragon')) return
    let ownerTag = killer.tags.find(tag => tag.startsWith('Owner:'))
    if (!ownerTag) return
    let player = event.server.getPlayer(ownerTag.split(':')[1])
    if (!player || !player.persistentData || !player.persistentData.get('drakekeepers_respite')) return
    let lvl = player.persistentData.drakekeepers_respite

    // Heal owner (2 + lvl hp)
    try { player.heal(2 + lvl) } catch (e) {}

    // Check victim states and grant boons
    let dur = 20 * (5 + 2 * lvl)
    try {
        // Burning -> Fire Resistance
        if (event.entity.isOnFire()) player.potionEffects.add('minecraft:fire_resistance', dur, 0, true, true)
        // Chilled -> Resistance
        if (event.entity.potionEffects.isActive('cofh_core:chilled')) player.potionEffects.add('minecraft:resistance', dur, 0, true, true)
        // Shocked/Electrified -> Absorption I
        if (event.entity.potionEffects.isActive('irons_spellbooks:electrified')) player.potionEffects.add('minecraft:absorption', dur, 0, true, true)
    } catch (e) {}
})


 // Dragons Favor: When your dragon kills a mob, you get an effect based on the dragon type. 5 Levels
//* - §aLevel 1: 5% chance
 //* - §aLevel 2: 10% chance
// * - §aLevel 3: 15% chance
// * - §aLevel 4: 20% chance
 //* - §aLevel 5: 25% chance
 //* 
 
//____________________________________________________________________________________________________
let dragon_kill_effects = {
    fire_dragon: {
        dragon: 'iceandfire:fire_dragon',
        id: 'irons_spellbooks:echoing_strikes',
    },
    ice_dragon: {
        dragon: 'iceandfire:ice_dragon',
        id: 'kubejs:glacial_grasp',
    },
    lightning_dragon: {
        dragon: 'iceandfire:lightning_dragon',
        id: 'irons_spellbooks:thunderstorm',
    }
}

EntityEvents.death(event => {
    if (!event.entity.isMonster()) return
    let dragon = event.source && event.source.actual
    if (!dl_isBoundDragon(dragon)) return
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner || !owner.persistentData.get('dragons_favor')) return
    let level = owner.persistentData.dragons_favor
    // 5% chance per level
    if (Math.random() * 100 > 5 * level) return
    let type = dragon.type
    Object.keys(dragon_kill_effects).forEach(key => {
        if (type == dragon_kill_effects[key].dragon) {
            owner.potionEffects.add(dragon_kill_effects[key].id, 200, 1)
        }
    })
})


/**
 * Soulfire: When you kill an enemy, your dragon gets a special effect. 4 levels. Each level gives a new effect and increases the percent chance of it activating
 * 
 * - §aLevel 1: Speed
 * - §aLevel 2: Haste
 * - §aLevel 3: Strength
 * - §aLevel 4: Regeneration
 * 
 */

//____________________________________________________________________________________________________
let player_kill_effects = {
    1: ['minecraft:speed'],
    2: ['minecraft:haste'],
    3: ['minecraft:strength'],
    4: ['minecraft:regeneration']
}

EntityEvents.death(event => {
    if (!event.entity.isMonster()) return
    if (!event.source.player) return
    let player = event.source.player
    if (!player.persistentData.get('soulfire')) return
    // 5% chance per level
    let level = player.persistentData.soulfire
    let chance = Math.random() * 100
    if (chance > 5*level) return
    // loop down through the levels to get all effects
    let dragon = dl_getBoundDragon(player)
    if (!dragon) return
    for (let i = level; i > 0; i--) {
        let list = player_kill_effects[i] || []
        list.forEach(effectId => {
            dragon.potionEffects.add(effectId, 200, 0, true, true)
        })
    }
})
// Legacy dragonbinder progression table removed; progression now handled in dragonSummon.
//____________________________________________________________________________________________________
PlayerEvents.respawned(event => {
    if (!event.player.persistentData.get('firebound')) return
    let player = event.player
    dragonSummon(player)
})

//____________________________________________________________________________________________________
PlayerEvents.loggedIn(event => {
    let player = event.player
    if (!player.persistentData.get('firebound') && !player.persistentData.get('icebound') && !player.persistentData.get('stormbound')) return
    Utils.server.scheduleInTicks(20, () => {
        if (player && dl_getBoundDragon(player) == null) {
            dragonSummon(player)
        }
    })
})




EntityEvents.death(event => {
    if (!event.entity.tags.contains('tamed_beast')) return
    if (!event.entity.tags.contains('bound_dragon')) return
    let owner = dl_getOwnerPlayer(event.entity, event.server)
    if (!owner) return
    owner.persistentData.putInt(`dragon_count`, 0)
    owner.persistentData.remove(`dragon_id`)
})


EntityEvents.death(event => {
    if (!event.entity.player) return
    let player = event.entity
    if (!player.persistentData.get('firebound') && !player.persistentData.get('icebound') && !player.persistentData.get('stormbound')) return
    let dragon = dl_getBoundDragon(player)
    if (dragon) {
        dragon.discard()
    }
})



//____________________________________________________________________________________________________
// Aggregated combat reactions keep ordering consistent so abilities don't step on each other
EntityEvents.hurt(event => {
    dl_handleDragonsAegis(event)
    dl_handleSearingScaleVeil(event)
    dl_handleDrakesMarkDragonStrike(event)
    dl_handleDrakesMarkPlayerStrike(event)
    dl_handleDragonOffenseSpell(event)
    dl_handleDragonDefenseSpell(event)
})

//____________________________________________________________________________________________________
// Dragon Calling with the Dragon Flute
ItemEvents.firstLeftClicked('iceandfire:dragon_stick', event => {
    if (!event.player.shiftKeyDown) return
    let player = event.player
    let dragon = dl_getBoundDragon(player)
    if (!dragon) return
    dl_runServerCommand(`/teleport ${dl_uuidSelector(dragon.uuid)} ${player.x} ${player.y} ${player.z}`)
})


//____________________________________________________________________________________________________
ServerEvents.recipes(event => {
    event.recipes.summoningrituals.altar(Item.of('iceandfire:dragon_horn').enchant('ensorcellation:soulbound', 1).enchant('minecraft:mending', 1))
    .itemOutput(Item.of('iceandfire:dragon_horn').enchant('ensorcellation:soulbound', 1).enchant('minecraft:mending', 1))
    .id('kubejs:dragon_horn')
})


//____________________________________________________________________________________________________
SummoningRituals.start(event => {
    if (event.recipe.id != 'kubejs:dragon_horn') return
    let player = event.player
    if (!player.persistentData.get('kubejs_class:dragonlord')) {
        event.cancel()
        return
    }
})


//____________________________________________________________________________________________________
SummoningRituals.complete(event => {
    if (event.recipe.id != 'kubejs:dragon_horn') return
    if (!event.player.persistentData.get('kubejs_class:dragonlord')) return
    if (event.player.persistentData.get('firebound')) return
    if (event.player.persistentData.dragon_count != 0) return
    let player = event.player
    dragonSummon(player)
})

//____________________________________________________________________________________________________
const dragonHornClicks = new Map()
ItemEvents.firstLeftClicked('iceandfire:dragon_horn', event => {
    if (!event.player.tags.contains('shrunken')) return
    if (!event.player.persistentData.get('shrink_level')) return
    let key = event.player.uuid
    let clicks = (dragonHornClicks.get(key) || 0) + 1
    dragonHornClicks.set(key, clicks)
    if (clicks < 2) return
    let player = event.player
    let dragon = dl_getBoundDragon(player)
    if (!dragon) return
    dl_runServerCommand(`/scale set pehkui:base 1 ${dl_uuidSelector(dragon.uuid)}`)
    event.player.tags.remove('shrunken')
    shrinkCounter.set(key, 2400)
    event.player.persistentData.remove('shrink_level')
    dragonHornClicks.set(key, 0)
})







ItemEvents.firstRightClicked('iceandfire:dragon_stick', event => {
    if (!event.player.shiftKeyDown) return
    let player = event.player
    player.swing()
    let dragon = dl_getBoundDragon(player)
    if (!dragon || !dragon.tags.contains('tamed_beast')) return
    let owner = dl_getOwnerPlayer(dragon, event.server)
    if (!owner || owner.username != player.username) return
    if (!player.persistentData.get('shrink_level')) {
        player.persistentData.putInt(`shrink_level`, 1)
        player.tags.add('shrunken')
    } else if (player.persistentData.shrink_level >= 3) {
        return
    } else {
        player.persistentData.shrink_level+=1
        player.tags.add('shrunken')
    }
    dl_runServerCommand(`/scale divide pehkui:base 2 ${dl_uuidSelector(dragon.uuid)}`)
})

//____________________________________________________________________________________________________
const shrinkCounter = new Map()

/**
 * PlayerEvents.tick(event => {
    if (!event.player.tags.contains('shrunken')) return
    if (!event.player.persistentData.get('shrink_level')) return
    let ticks = 2400
    let key = event.player.uuid
    let remaining = shrinkCounter.has(key) ? shrinkCounter.get(key) - 1 : ticks
    shrinkCounter.set(key, remaining)
    if (remaining > 0) return
    let dragon = dl_getBoundDragon(event.player)
    if (dragon) {
        dl_runServerCommand(`/scale set pehkui:base 1 ${dl_uuidSelector(dragon.uuid)}`)
    }
    event.player.tags.remove('shrunken')
    shrinkCounter.set(key, ticks)
    event.player.persistentData.remove('shrink_level')
})
 * 
 * 
 * 
 */






/**
 * ItemEvents.rightClicked('iceandfire:dragon_skull_fire', event => {
    dragonSummon(event.player)
})
 * 
 */



 

