
/** castOnceShort:
 *  - Used for casting spells that have a short cast time
 *  - Example: firebolt, icicle, firecracker, etc.
 *  - Spell level CANNOT be 0
 * 
 * 
 */
function castOnceShort (caster, spell, target, spell_level) {
    caster.lookAt("eyes",new Vec3d(target.x, target.y+1, target.z))
    let dim = caster.level.dimension
    Utils.server.runCommandSilent(`/execute in ${dim} run cast ${caster.uuid} ${spell} ${spell_level}`)
}

/** castOnceLong:
    *  - Used for casting spells that have a long cast time. 
    *  - the cast time allows for spells that are continuous (firebreath) or just a long startup cast time (fireball).
    *  - Example: fire_breath, cone_of_cold, dragon_breath, etc.
    * - Spell level CANNOT be 0
    * 
    * 
    */

function castOnceLong (caster, spell, target, spell_level, cast_time) {
    caster.lookAt("eyes",new Vec3d(target.x, target.y+0.5, target.z))
    let dim = caster.level.dimension
    Utils.server.runCommandSilent(`/execute in ${dim} run cast ${caster.uuid} ${spell} ${spell_level}`)
    let iterations = cast_time/40
    // loop through the iterations
    for (let i = 0; i < iterations; i++) {
        Utils.server.scheduleInTicks(10*i, () => {
            if (!target.alive) return
            caster.lookAt("eyes",new Vec3d(target.x, target.y+1, target.z))
        })
    }
}


/** castMultipleSame:
 * - Used for casting multiple of the SAME spell. 
 * - Can be used for spells with any cast time as long as the correct cooldown is provided
 * - Spell level CANNOT be 0
 * - Cooldown is in TICKS
 */
function castMultipleSame (caster, spell, target, casts, cooldown, spell_level) {
    for (let i = 0; i < casts; i++) {
        Utils.server.scheduleInTicks(cooldown*i, () => {
            caster.lookAt("eyes",new Vec3d(target.x, target.y, target.z))
            let dim = caster.level.dimension
            Utils.server.runCommandSilent(`/execute in ${dim} run cast ${caster.uuid} ${spell} ${spell_level}`)
        })
    }
}




/**
 * Casts multiple spells from a caster entity onto a target entity with proper scheduling and orientation.
 *
 * This function schedules the casting of multiple spells, ensuring that each spell is cast
 * at the correct time based on its cast time. If a spell is categorized as 'persistent',
 * the caster will continuously face the target entity at regular intervals during the spell's duration.
 *
 * @param {Entity} caster - The entity that will cast the spells.
 * @param {Object} spells - An object containing spell definitions. Each key represents a unique identifier for the spell, and its value is the spell's name as a string.
 * @param {Entity} target - The entity that is the target of the spells.
 * @param {number} spell_level - The level of the spells being cast, influencing their potency or effects.
 */
function castMultipleDifferent(caster, spells, target, spell_level) {
    if (!caster || !target) return;

    // Make the caster face the target
    caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
    faceEntity(caster, target);

    let dim = caster.level.dimension;
    let cumulativeCooldown = 0;

    let sortedKeys = Object.keys(spells).sort((a, b) => a.localeCompare(b));

    sortedKeys.forEach(key => {
        let spellName = spells[key];
        if (typeof spellName !== 'string' || spellName.trim() === "") {
            console.log(`Invalid spell name for key "${key}":`, spellName);
            return;
        }

        // Retrieve spell data from spellcastingData
        let spellInfo = getSpellData(spellName);
        if (!spellInfo) {
            console.log(`Spell "${spellName}" not found in spellcastingData.`);
            return;
        }

        let castTime = getCastTimeInTicks(spellInfo.cast_time);
        let spellCastTick = cumulativeCooldown;

        Utils.server.scheduleInTicks(spellCastTick, () => {
            if (!caster.isAlive()) return;

            caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
            faceEntity(caster, target);

            Utils.server.scheduleInTicks(1, () => {
                Utils.server.runCommandSilent(`/execute in ${dim} run cast ${caster.uuid} ${spellName} ${spell_level}`);
            });
        });

        Utils.server.scheduleInTicks(spellCastTick + castTime - 1, () => {
            if (!caster.isAlive()) return;

            caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
            faceEntity(caster, target);
        });

        if (isPersistentSpell(spellInfo)) {
            let durationTicks = spellInfo.cast_time * 20;

            for (let i = 0; i < durationTicks; i += 2) {
                Utils.server.scheduleInTicks(spellCastTick + i, () => {
                    if (!caster.isAlive()) return;

                    caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
                    faceEntity(caster, target);
                });
            }
        }

        cumulativeCooldown += castTime;
    });
}




/**
 * generalCast:
 * Inputs: 
 *  - caster: the entity that is casting the spell
 *  - spell: the spell that is being cast
 *  - target: the entity that is being targeted
 *  - spell_level: the level of the spell
 *  - casts: the number of times the spell is cast
 * 
 * Uses the spelltypes object to determine what targeting of spell it is.
 *  - if casts is 1, it will use castOnceShort or castOnceLong
 *  - if casts is greater than 1, it will use castMultipleSame or castMultipleDifferent
 *  - uses the spelltypes object to determine what targeting of spell it is and the cast time
 */



function generalCast(caster, spell, target, spell_level, casts) {
    // find the spell targeting by checking the spelltypes object for the spell
   // Utils.server.tell(`spell: ${spell}`)
    // get the cooldown of the spell
    let cooldown = spelltypes[spell]
    //Utils.server.tell(`cooldown: ${cooldown}`)
    if (casts == 1) {
        if (cooldown == 0.15) {
            castOnceShort(caster, spell, target, spell_level)
        } else {
            castOnceLong(caster, spell, target, spell_level, 2+cooldown*20)
        }
    } else if (casts >= 2) {
        castMultipleSame(caster, spell, target, casts, 2+cooldown*20, spell_level)
    }
}
/**
 * ItemEvents.entityInteracted('minecraft:iron_sword', event => {
    let player = event.player
    let target = event.target
    let spell = 'firecracker'
    let spell_level = 3
    let casts = 3
    generalCast(target, spell, player, spell_level, casts)
})
 * 
 */


let spelltypes = {
        'acupuncture': 0.15,
        'ball_lightning': 0.15,
        'blood_needles': 0.15,
        'blood_slash': 0.15,
        'blood_step': 0.15,
        'burning_dash': 0.15,
        'chain_lightning': 0.15,
        'charge': 0.15,
        'counterspell': 0.15,
        'devour': 0.15,
        //
        'frost_step': 0.15,
        'echoing_strikes': 0.15,
        'eldritch_blast': 0.15,
        'evasion': 0.15,
        'firebolt': 0.15,
        'firecracker': 0.15,
        'icicle': 0.15,
        'lightning_bolt': 0.15,
        'lob_creeper': 0.15,
        'magic_missile': 0.15,
        'ray_of_frost': 0.15,
        'shield': 0.15,
        'thunderstorm': 0.15,
        'gluttony': 0.15,
        'guiding_bolt': 0.15,
        'heal': 0.15,
        'heartstop': 0.15,
        'oakskin': 0.15,
        'planar_sight': 0.15,
        'traveloptics:blood_howl': 0.15,
        'traveloptics:despair': 0.15,
//_____________________________________________________________________________________
        'acid_orb': 1,
        'arrow_volley': 1.5,
        'blaze_storm': 3,
        'blessing_of_life': 2,
        'blight': 2.5,
        'chain_creeper': 1.5,
        'cloud_of_regeneration': 10,
        'cone_of_cold': 5,
        'divine_smite': 1,
        //
        'dragon_breath': 5,
        'earthquake': 2,
        'electrocute': 5,
        'fang_strike': 1,
        'fire_breath': 5,
        'fireball': 2,
        'firefly_swarm': 2,
        'flaming_strike': 0.5,
        'fortify': 2,
        'frostwave': 1,
        'greater_heal': 6,
        'gust': 0.7,
        'haste': 2,
        'healing_circle': 1,
        'heat_surge': 1,
        'ice_block': 2,
        'lightning_lance': 2,
        'magic_arrow': 1.5,
        'magma_bomb': 1,
        'poison_arrow': 1,
        'poison_breath': 5,
        'poison_splash': 1,
        'raise_dead': 2,
        'root': 2,
        'scorch': 1,
        'sculk_tentacles': 8,
        'shockwave': 0.7,
        'slow': 1.5,
        'starfall': 8,
        'stomp': 0.5,
        'summon_vex': 1,
        'traveloptics:abyssal_blast': 2.5,
        'traveloptics:ashen_breath': 5,
        
        'traveloptics:burning_judgement': 2.5,
        'traveloptics:stele_cascade': 1,
        'traveloptics:death_laser': 1,
        
        'traveloptics:meteor_storm': 0.7,
        'traveloptics:orbital_void': 2,


}




// Define the spell_data object
let spell_data = {
    instant: {
        'acupuncture': 0.15,
        'ball_lightning': 0.15,
        'blood_needles': 0.15,
        'blood_slash': 0.15,
        'blood_step': 0.15,
        'burning_dash': 0.15,
        'chain_lightning': 0.15,
        'charge': 0.15,
        'counterspell': 0.15,
        'devour': 0.15,
        'frost_step': 0.15,
        'echoing_strikes': 0.15,
        'eldritch_blast': 0.15,
        'evasion': 0.15,
        'firebolt': 0.15,
        'firecracker': 0.15,
        'icicle': 0.15,
        'lightning_bolt': 0.15,
        'lob_creeper': 0.15,
        'magic_missile': 0.15,
        'ray_of_frost': 0.15,
        'shield': 0.15,
        'thunderstorm': 0.15,
        'gluttony': 0.15,
        'guiding_bolt': 0.15,
        'heal': 0.15,
        'heartstop': 0.15,
        'oakskin': 0.15,
        'planar_sight': 0.15,
        'traveloptics:blood_howl': 0.15,
        'traveloptics:despair': 0.15,
        'traveloptics:blood_howl': 0.15,
        'traveloptics:despair': 0.15
    },
    charge: {
        'acid_orb': 1,
        'arrow_volley': 1.5,
        'blight': 2.5,
        'chain_creeper': 1.5,
        'cloud_of_regeneration': 10,
        'divine_smite': 1,
        'earthquake': 2,
        'blessing_of_life': 2,
        'fang_strike': 1,
        'fireball': 2,
        'firefly_swarm': 2,
        'flaming_strike': 0.5,
        'fortify': 2,
        'frostwave': 1,
        'greater_heal': 6,
        'gust': 0.7,
        'haste': 2,
        'healing_circle': 1,
        'heat_surge': 1,
        'ice_block': 2,
        'lightning_lance': 2,
        'magic_arrow': 1.5,
        'magma_bomb': 1,
        'poison_arrow': 1,
        'poison_splash': 1,
        'raise_dead': 2,
        'root': 2,
        'scorch': 1,
        'sculk_tentacles': 8,
        'shockwave': 0.7,
        'slow': 1.5,
        'starfall': 8,
        'stomp': 0.5,
        'summon_vex': 1,
        'traveloptics:burning_judgement': 2.5,
        'traveloptics:stele_cascade': 1,
        'traveloptics:death_laser': 6,
        'traveloptics:meteor_storm': 0.7,
        'traveloptics:orbital_void': 2,
        'traveloptics:abyssal_blast': 8,
        'traveloptics:aerial_collapse': 4,
        'traveloptics:annihilation': 4.5,
        'traveloptics:axe_of_the_doomed': 4.5,
        'traveloptics:cursed_minefield': 2.5,
        'traveloptics:burning_judgment': 2.7,
        'traveloptics:cursed_revenants': 2.5,
        'traveloptics:eternal_sentinel': 4.2,
    },
    persistent: {
        'blaze_storm': 3,
        'cone_of_cold': 5,
        'dragon_breath': 5,
        'electrocute': 5,
        'fire_breath': 5,
        'poison_breath': 5,
        'traveloptics:ashen_breath': 5,
        
    }
};

/**
 * Spell data guide:
 * 
 * Tiers are difficulty level thats added with §aLevel 1 when it comes to boss variants. Increasing the level by 1 increases the difficulty of the boss variant by 1 as well
 *    Tier 1: 
 *      - Low damage 
 *      - Usually instant casts that arent healing or protection spells.
 *    Tier 2: 
 *      - Medium damage or prolonged damage. 
 *      - Includes minion summons that arent great like raise_dead.
 *      - All Movement spells
 *      - enemy AOE spells
 *      - Weaker targeted spells
 *      - Support spells like shield, fortify, etc.
 *    Tier 3: 
 *      - High damage enemys or aoe spells
 *      - Stronger minion summons like cursed_revenants
 *      - Stronger targeted spells
 *      - Stronger support spells like healing circle, haste, etc.
 *    Tier 4: 
 *      - Very high damage
 *      - Very strong minions like eternal_sentinel which summons a golem and primal_pack which summons a pack of dinosaurs
 *      - Very strong support spells like greater_heal, slow, etc.
 * 
 * 

 */


let spellcastingData = {
    instant: {
        'acupuncture': {
            id: 'acupuncture',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'blood',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'ball_lightning': {
            id: 'ball_lightning',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'blood_needles': {
            id: 'blood_needles',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'blood',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'blood_slash': {
            id: 'blood_slash',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 5,
            school: 'blood',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'blood_step': {
            id: 'blood_step',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'blood',
            distance: 'close',
            use: ['movement'],
            type: 'instant'
        },
        'burning_dash': {
            id: 'burning_dash',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'close',
            use: ['movement'],
            type: 'instant'
        },
        'chain_lightning': {
            id: 'chain_lightning',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'close',
            use: ['damage'],
            type: 'instant'
        },
        'charge': {
            id: 'charge',
            cast_time: 0.15,
            tier: 3,
            targeting: 'self',
            max_level: 3,
            school: 'lightning',
            distance: 'close',
            use: ['empower'],
            type: 'instant'
        },
        'counterspell': {
            id: 'counterspell',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 1,
            school: 'ender',
            distance: 'close',
            use: ['reflect'],
            type: 'instant'
        },
        'devour': {
            id: 'devour',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'blood',
            distance: 'close',
            use: ['damage'],
            type: 'instant'
        },
        'frost_step': {
            id: 'frost_step',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'ice',
            distance: 'any',
            use: ['movement'],
            type: 'instant'
        },
        'echoing_strikes': {
            id: 'echoing_strikes',
            cast_time: 0.15,
            tier: 4,
            targeting: 'enemy',
            max_level: 5,
            school: 'ender',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'eldritch_blast': {
            id: 'eldritch_blast',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'eldritch',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'instant'
        },
        'evasion': {
            id: 'evasion',
            cast_time: 0.15,
            tier: 4,
            targeting: 'self',
            max_level: 5,
            school: 'ender',
            distance: 'any',
            use: ['empower'],
            type: 'instant'
        },
        'firebolt': {
            id: 'firebolt',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'firecracker': {
            id: 'firecracker',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'evocation',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'icicle': {
            id: 'icicle',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'ice',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'lightning_bolt': {
            id: 'lightning_bolt',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'lob_creeper': {
            id: 'lob_creeper',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'evocation',
            distance: 'close',
            use: ['damage'],
            type: 'instant'
        },
        'magic_missile': {
            id: 'magic_missile',
            cast_time: 0.15,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'ender',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'ray_of_frost': {
            id: 'ray_of_frost',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'ice',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'shield': {
            id: 'shield',
            cast_time: 0.15,
            tier: 3,
            targeting: 'reflect',
            max_level: 10,
            school: 'evocation',
            distance: 'close',
            use: ['reflect'],
            type: 'instant'
        },
        'thunderstorm': {
            id: 'thunderstorm',
            cast_time: 0.15,
            tier: 4,
            targeting: 'enemy',
            max_level: 8,
            school: 'lightning',
            distance: 'close',
            use: ['damage'],
            type: 'instant'
        },
        'guiding_bolt': {
            id: 'guiding_bolt',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'holy',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'heal': {
            id: 'heal',
            cast_time: 0.15,
            tier: 2,
            targeting: 'self',
            max_level: 10,
            school: 'holy',
            distance: 'close',
            use: ['empower'],
            type: 'instant'
        },
        'oakskin': {
            id: 'oakskin',
            cast_time: 0.15,
            tier: 2,
            targeting: 'self',
            max_level: 8,
            school: 'nature',
            distance: 'close',
            use: ['empower'],
            type: 'instant'
        },
        'blood_howl': {
            id: 'traveloptics:blood_howl',
            cast_time: 0.15,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'blood',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        },
        'despair': {
            id: 'traveloptics:despair',
            cast_time: 0.15,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'ice',
            distance: 'any',
            use: ['damage'],
            type: 'instant'
        }
        // Add other instant spells here following the same structure
    },
    charge: {
        'acid_orb': {
            id: 'acid_orb',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'poison',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'arrow_volley': {
            id: 'arrow_volley',
            cast_time: 1.5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'nature',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'blight': {
            id: 'blight',
            cast_time: 2.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 8,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'chain_creeper': {
            id: 'chain_creeper',
            cast_time: 1.5,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'evocation',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'cloud_of_regeneration': {
            id: 'cloud_of_regeneration',
            cast_time: 10,
            tier: 4,
            targeting: 'ally',
            max_level: 3,
            school: 'holy',
            distance: 'close',
            use: ['empower'],
            type: 'charge'
        },
        'divine_smite': {
            id: 'divine_smite',
            cast_time: 1,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'holy',
            distance: 'very_close',
            use: ['damage'],
            type: 'charge'
        },
        'earthquake': {
            id: 'earthquake',
            cast_time: 2,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'earth',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'blessing_of_life': {
            id: 'blessing_of_life',
            cast_time: 2,
            tier: 3,
            targeting: 'ally',
            max_level: 5,
            school: 'holy',
            distance: 'close',
            use: ['empower'],
            type: 'charge'
        },
        'fang_strike': {
            id: 'fang_strike',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 7,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'fireball': {
            id: 'fireball',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'firefly_swarm': {
            id: 'firefly_swarm',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'flaming_strike': {
            id: 'flaming_strike',
            cast_time: 0.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'fire',
            distance: 'very_close',
            use: ['damage'],
            type: 'charge'
        },
        'fortify': {
            id: 'fortify',
            cast_time: 2,
            tier: 3,
            targeting: 'self',
            max_level: 10,
            school: 'holy',
            distance: 'any',
            use: ['empower'],
            type: 'charge'
        },
        'frostwave': {
            id: 'frostwave',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'ice',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'greater_heal': {
            id: 'greater_heal',
            cast_time: 6,
            tier: 4,
            targeting: 'ally',
            max_level: 1,
            school: 'holy',
            distance: 'close',
            use: ['empower'],
            type: 'charge'
        },
        'gust': {
            id: 'gust',
            cast_time: 0.7,
            tier: 1,
            targeting: 'enemy',
            max_level: 10,
            school: 'evocation',
            distance: 'close',
            use: ['hinder'],
            type: 'charge'
        },
        'haste': {
            id: 'haste',
            cast_time: 2,
            tier: 3,
            targeting: 'self',
            max_level: 6,
            school: 'holy',
            distance: 'any',
            use: ['empower'],
            type: 'charge'
        },
        'healing_circle': {
            id: 'healing_circle',
            cast_time: 1,
            tier: 3,
            targeting: 'self',
            max_level: 10,
            school: 'holy',
            distance: 'close',
            use: ['empower'],
            type: 'charge'
        },
        'heat_surge': {
            id: 'heat_surge',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'fire',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'ice_block': {
            id: 'ice_block',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'ice',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'lightning_lance': {
            id: 'lightning_lance',
            cast_time: 2,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'charge'
        },
        'magic_arrow': {
            id: 'magic_arrow',
            cast_time: 1.5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'ender',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'magma_bomb': {
            id: 'magma_bomb',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'fire',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'poison_arrow': {
            id: 'poison_arrow',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'poison',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'poison_splash': {
            id: 'poison_splash',
            cast_time: 1,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'poison',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'raise_dead': {
            id: 'raise_dead',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'blood',
            distance: 'any',
            use: ['summon_multiple'],
            type: 'charge'
        },
        'root': {
            id: 'root',
            cast_time: 2.2,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'nature',
            distance: 'any',
            use: ['control'],
            type: 'charge'
        },
        'scorch': {
            id: 'scorch',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'charge'
        },
        'sculk_tentacles': {
            id: 'sculk_tentacles',
            cast_time: 2,
            tier: 4,
            targeting: 'enemy',
            max_level: 4,
            school: 'eldritch',
            distance: 'any',
            use: ['followup'],
            type: 'charge'
        },
        'shockwave': {
            id: 'shockwave',
            cast_time: 0.7,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'lightning',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'slow': {
            id: 'slow',
            cast_time: 1.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 6,
            school: 'evocation',
            distance: 'close',
            use: ['control', 'hinder'],
            type: 'charge'
        },
        'starfall': {
            id: 'starfall',
            cast_time: 8,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'ender',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'charge'
        },
        'stomp': {
            id: 'stomp',
            cast_time: 0.5,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'summon_vex': {
            id: 'summon_vex',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'evocation',
            distance: 'any',
            use: ['summon_multiple'],
            type: 'charge'
        },
        'burning_judgement': {
            id: 'traveloptics:burning_judgement',
            cast_time: 2.7,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'fire',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'stele_cascade': {
            id: 'traveloptics:stele_cascade',
            cast_time: 1.2,
            tier: 2,
            targeting: 'enemy',
            max_level: 6,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'death_laser': {
            id: 'traveloptics:death_laser',
            cast_time: 6,
            tier: 3,
            targeting: 'enemy',
            max_level: 3,
            school: 'lightning',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'charge'
        },
        'meteor_storm': {
            id: 'traveloptics:meteor_storm',
            cast_time: 0.7,
            tier: 3,
            targeting: 'enemy',
            max_level: 6,
            school: 'fire',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'orbital_void': {
            id: 'traveloptics:orbital_void',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'ender',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'abyssal_blast': {
            id: 'traveloptics:abyssal_blast',
            cast_time: 8,
            tier: 4,
            targeting: 'enemy',
            max_level: 3,
            school: 'eldritch',
            distance: 'any',
            use: ['damage', 'followup'],
            type: 'charge'
        },
        'aerial_collapse': {
            id: 'traveloptics:aerial_collapse',
            cast_time: 4,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'nature',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'annihilation': {
            id: 'traveloptics:annihilation',
            cast_time: 4.5,
            tier: 4,
            targeting: 'enemy',
            max_level: 3,
            school: 'fire',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'axe_of_the_doomed': {
            id: 'traveloptics:axe_of_the_doomed',
            cast_time: 4.5,
            tier: 4,
            targeting: 'enemy',
            max_level: 3,
            school: 'ice',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'cursed_minefield': {
            id: 'traveloptics:cursed_minefield',
            cast_time: 2.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 8,
            school: 'ender',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'cursed_revenants': {
            id: 'traveloptics:cursed_revenants',
            cast_time: 2.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'ice',
            distance: 'any',
            use: ['summon_multiple'],
            type: 'charge'
        },
        'eternal_sentinel': {
            id: 'traveloptics:eternal_sentinel',
            cast_time: 4.2,
            tier: 4,
            targeting: 'enemy',
            max_level: 2,
            school: 'eldritch',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'halberd_horizon': {
            id: 'traveloptics:halberd_horizon',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 7,
            school: 'earth',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'ignited_onslaught': {
            id: 'traveloptics:ignited_onslaught',
            cast_time: 3.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'fire',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'lava_bomb': {
            id: 'traveloptics:lava_bomb',
            cast_time: 2.5,
            tier: 2,
            targeting: 'enemy',
            max_level: 3,
            school: 'fire',
            distance: 'close',
            use: ['damage'],
            type: 'charge',
            
        },
        /**
         *         'magnetron_deployment': {
            id: 'traveloptics:magnetron_deployment',
            cast_time: 3,
            tier: 2,
            targeting: 'enemy',
            max_level: 1,
            school: 'lightning',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
         * 
         * 
         */

        'mechanized_predator': {
            id: 'traveloptics:mechanized_predator',
            cast_time: 4.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'lightning',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'nocturnal_swarm': {
            id: 'traveloptics:nocturnal_swarm',
            cast_time: 2,
            tier: 2,
            targeting: 'enemy',
            max_level: 8,
            school: 'blood',
            distance: 'any',
            use: ['summon_multiple'],
            type: 'charge'
        },
        /**
         *         'the_forgotten_beast': {
            id: 'traveloptics:the_forgotten_beast',
            cast_time: 5,
            tier: 4,
            targeting: 'enemy',
            max_level: 2,
            school: 'eldritch',
            distance: 'close',
            use: ['summon_once'],
            type: 'charge'
        },
         * 
         * 
         */

        'void_eruption': {
            id: 'traveloptics:void_eruption',
            cast_time: 2.2,
            tier: 2,
            targeting: 'enemy',
            max_level: 3,
            school: 'ender',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        },
        'nullflare': {
            id: 'traveloptics:nullflare',
            cast_time: 1,
            tier: 2,
            targeting: 'enemy',
            max_level: 5,
            school: 'holy',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'sword_of_the_ancients': {
            id: 'traveloptics:sword_of_the_ancients',
            cast_time: 4.2,
            tier: 4,
            targeting: 'enemy',
            max_level: 3,
            school: 'holy',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'summon_desert_dwellers': {
            id: 'traveloptics:summon_desert_dwellers',
            cast_time: 3.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'holy',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'em_pulse': {
            id: 'traveloptics:em_pulse',
            cast_time: 5,
            tier: 3,
            targeting: 'enemy',
            max_level: 5,
            school: 'lightning',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'primal_pack': {
            id: 'traveloptics:primal_pack',
            cast_time: 5,
            tier: 4,
            targeting: 'enemy',
            max_level: 5,
            school: 'nature',
            distance: 'any',
            use: ['summon_once'],
            type: 'charge'
        },
        'solar_flare': {
            id: 'traveloptics:solar_flare',
            cast_time: 1.5,
            tier: 3,
            targeting: 'self',
            max_level: 8,
            school: 'holy',
            distance: 'close',
            use: ['damage'],
            type: 'charge'
        },
        'vortex_punch': {
            id: 'traveloptics:vortex_punch',
            cast_time: 1.5,
            tier: 3,
            targeting: 'enemy',
            max_level: 3,
            school: 'ender',
            distance: 'any',
            use: ['damage'],
            type: 'charge'
        }
        // Add other charge spells here following the same structure
    },
    persistent: {
        'blaze_storm': {
            id: 'blaze_storm',
            cast_time: 3,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'any',
            use: ['damage'],
            type: 'persistent'
        },
        'cone_of_cold': {
            id: 'cone_of_cold',
            cast_time: 5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'ice',
            distance: 'close',
            use: ['damage'],
            type: 'persistent'
        },
        'dragon_breath': {
            id: 'dragon_breath',
            cast_time: 5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'ender',
            distance: 'close',
            use: ['damage'],
            type: 'persistent'
        },
        'electrocute': {
            id: 'electrocute',
            cast_time: 5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'close',
            use: ['damage'],
            type: 'persistent'
        },
        'fire_breath': {
            id: 'fire_breath',
            cast_time: 5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'fire',
            distance: 'close',
            use: ['damage'],
            type: 'persistent'
        },
        'poison_breath': {
            id: 'poison_breath',
            cast_time: 5,
            tier: 2,
            targeting: 'enemy',
            max_level: 10,
            school: 'poison',
            distance: 'close',
            use: ['damage'],
            type: 'persistent'
        },
        'ashen_breath': {
            id: 'traveloptics:ashen_breath',
            cast_time: 5,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'evocation',
            distance: 'close',
            use: ['hinder'],
            type: 'persistent'
        },
        'rapid_laser': {
            id: 'traveloptics:rapid_laser',
            cast_time: 6,
            tier: 3,
            targeting: 'enemy',
            max_level: 10,
            school: 'lightning',
            distance: 'any',
            use: ['damage'],
            type: 'persistent'
        }
        // Add other persistent spells here following the same structure
    }
};











/**
 * Casts multiple spells from a caster entity onto a target entity with proper scheduling and orientation.
 *
 * This function schedules the casting of multiple spells, ensuring that each spell is cast
 * at the correct time based on its cast time. If a spell is categorized as 'persistent',
 * the caster will continuously face the target entity at regular intervals during the spell's duration.
 *
 * @param {Entity} caster - The entity that will cast the spells.
 * @param {Object} spells - An object containing spell definitions. Each key represents a unique identifier for the spell, and its value is the spell's name as a string.
 * @param {Entity} target - The entity that is the target of the spells.
 * @param {number} spell_level - The level of the spells being cast, influencing their potency or effects.
 */
function overhauledCast(caster, spells, target, spell_level) {
    if (!caster || !target) return;
    //Utils.server.tell('casting')
    // Make the caster face the target
    caster.lookAt("eyes", new Vec3d(target.x, target.y, target.z));
    faceEntity(caster, target);

    let dim = caster.level.dimension;
    let cumulativeCooldown = 0;

    let sortedKeys = Object.keys(spells).sort((a, b) => a.localeCompare(b));

    sortedKeys.forEach(key => {
        let spellName = spells[key];
        if (typeof spellName !== 'string' || spellName.trim() === "") {
            console.log(`Invalid spell name for key "${key}":`, spellName);
            return;
        }

        // Retrieve spell data from spellcastingData
        let spellInfo = getSpellData(spellName);
        if (!spellInfo) {
            console.log(`Spell "${spellName}" not found in spellcastingData.`);
            return;
        }
        let spellID = spellInfo.id;
        let castTime = getCastTimeInTicks(spellInfo.cast_time);
        let spellCastTick = cumulativeCooldown;

        Utils.server.scheduleInTicks(spellCastTick+2, () => {
            if (!caster.isAlive()) return;

            caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
            faceEntity(caster, target);

            Utils.server.scheduleInTicks(1, () => {
                Utils.server.runCommandSilent(`/execute in ${dim} run cast ${caster.uuid} ${spellID} ${spell_level}`);
                //Utils.server.tell(`/execute in ${dim} run cast ${caster.uuid} ${spellID} ${spell_level}`);
            });
        });

        Utils.server.scheduleInTicks(spellCastTick + castTime - 1, () => {
            if (!caster.isAlive()) return;

            caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
            faceEntity(caster, target);
        });

        if (spellInfo.type === 'persistent') {
            let durationTicks = spellInfo.cast_time * 20;

            for (let i = 0; i < durationTicks; i += 2) {
                Utils.server.scheduleInTicks(spellCastTick + i, () => {
                    if (!caster.isAlive()) return;

                    caster.lookAt("eyes", new Vec3d(target.x, target.y + 0.5, target.z));
                    faceEntity(caster, target);
                });
            }
        }

        cumulativeCooldown += castTime;
    });
}

/**
 * Retrieves the cast time of a spell in ticks (20 ticks = 1 second)
 * 
 * @param {number} castTimeInSeconds - The cast time in seconds
 * @returns {number} The cast time in ticks
 */
function getCastTimeInTicks(castTimeInSeconds) {
    return Math.floor(castTimeInSeconds * 20);
}

/**
 * Retrieves spell data from the spellcastingData object
 * 
 * @param {string} spellName - The name of the spell to retrieve
 * @returns {Object} The spell data if it exists, otherwise null
 */
function getSpellData(spellName) {
    // Look for the spell in instant, charge, or persistent categories
    if (spellcastingData.instant[spellName]) return spellcastingData.instant[spellName];
    if (spellcastingData.charge && spellcastingData.charge[spellName]) return spellcastingData.charge[spellName];
    if (spellcastingData.persistent && spellcastingData.persistent[spellName]) return spellcastingData.persistent[spellName];
    return null;
}

/**
 * Determines if a spell is persistent
 * 
 * @param {Object} spellInfo - The spell information object from spellcastingData
 * @returns {boolean} True if the spell is persistent, false otherwise
 */


function chooseDamageSpells(maxTier, maxLevel) {
    let chosenSpells = [];
    // Choose 3 instant damage spells
    let instantDamageSpells = Object.keys(spellcastingData.instant).filter(key => spellcastingData.instant[key].use.includes('damage'));
    for (let i = 0; i < 3; i++) {
        chosenSpells.push(instantDamageSpells[Math.floor(Math.random() * instantDamageSpells.length)]);
    }
    // Choose 4 charge damage spells
    let chargeDamageSpells = Object.keys(spellcastingData.charge).filter(key => spellcastingData.charge[key].use.includes('damage'));
    for (let i = 0; i < 4; i++) {
        chosenSpells.push(chargeDamageSpells[Math.floor(Math.random() * chargeDamageSpells.length)]);
    }
    // Choose 1 persistent damage spell
    let persistentDamageSpells = Object.keys(spellcastingData.persistent).filter(key => spellcastingData.persistent[key].use.includes('damage'));
    chosenSpells.push(persistentDamageSpells[Math.floor(Math.random() * persistentDamageSpells.length)]);

    if (chosenSpells == null || chosenSpells.length === 0) {
        //Utils.server.tell('Re-rolling Damage Spells...');
        return chooseDamageSpells(maxTier, maxLevel);
    }
    return chosenSpells;
}

function chooseEmpowerSpells(maxTier, maxLevel) {
    let chosenSpells = [];
    // Choose 2 self-empower instant spells
    let selfEmpowerSpells = Object.keys(spellcastingData.instant).filter(key => 
        spellcastingData.instant[key].use.includes('empower') && spellcastingData.instant[key].targeting === 'self'
    );
    chosenSpells.push(selfEmpowerSpells[Math.floor(Math.random() * selfEmpowerSpells.length)]);

    // Choose 1 ally-empower charge spell
    let allyEmpowerSpells = Object.keys(spellcastingData.charge).filter(key => 
        spellcastingData.charge[key].use.includes('empower') && spellcastingData.charge[key].targeting === 'ally'
    );
    chosenSpells.push(allyEmpowerSpells[Math.floor(Math.random() * allyEmpowerSpells.length)]);

    if (chosenSpells == null || chosenSpells.length === 0) {
        //Utils.server.tell('Re-rolling Empower Spells...');
        return chooseEmpowerSpells(maxTier, maxLevel);
    }
    return chosenSpells;
}

function chooseControlSpells(maxTier, maxLevel) {
    let controlSpells = Object.keys(spellcastingData.charge).filter(key => 
        spellcastingData.charge[key].use.includes('control')
    );
    // Choose 1 control charge spell
    let result = [controlSpells[Math.floor(Math.random() * controlSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Control Spells...');
        return chooseControlSpells(maxTier, maxLevel);
    }
    return result;
}

function chooseFollowupSpells(maxTier, maxLevel) {
    let followupSpells = Object.keys(spellcastingData.charge).filter(key => 
        spellcastingData.charge[key].use.includes('followup')
    );
    // Choose 1 followup charge spell
    let result = [followupSpells[Math.floor(Math.random() * followupSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Followup Spells...');
        return chooseFollowupSpells(maxTier, maxLevel);
    }
    return result;
}

function chooseHinderSpells(maxTier, maxLevel) {
    let potentialSpells = [];
    // Choose 2 hinder spells
    let hinderSpells = Object.keys(spellcastingData.charge).filter(key => 
        spellcastingData.charge[key].use.includes('hinder')
    );

    potentialSpells.push(hinderSpells[Math.floor(Math.random() * hinderSpells.length)]);

    let persistentHinderSpells = Object.keys(spellcastingData.persistent).filter(key => 
        spellcastingData.persistent[key].use.includes('hinder')
    );
    potentialSpells.push(persistentHinderSpells[Math.floor(Math.random() * persistentHinderSpells.length)]);

    let result = [potentialSpells[Math.floor(Math.random() * potentialSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Hinder Spells...');
        return chooseHinderSpells(maxTier, maxLevel);
    }
    return result;
}

function chooseMovementSpells(maxTier, maxLevel) {
    let potentialSpells = [];
    // Choose 1 movement spell
    let movementSpells = Object.keys(spellcastingData.instant).filter(key => 
        spellcastingData.instant[key].use.includes('movement')
    );
    potentialSpells.push(movementSpells[Math.floor(Math.random() * movementSpells.length)]);

    let result = [potentialSpells[Math.floor(Math.random() * potentialSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Movement Spells...');
        return chooseMovementSpells(maxTier, maxLevel);
    }
    return result;
}

function chooseReflectSpells(maxTier, maxLevel) {
    let potentialSpells = [];
    // Choose 1 reflect spell
    let reflectSpells = Object.keys(spellcastingData.instant).filter(key => 
        spellcastingData.instant[key].use.includes('reflect')
    );
    potentialSpells.push(reflectSpells[Math.floor(Math.random() * reflectSpells.length)]);

    let result = [potentialSpells[Math.floor(Math.random() * potentialSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Reflect Spells...');
        return chooseReflectSpells(maxTier, maxLevel);
    }
    return result;
}

function chooseSummonMultipleSpells(maxTier, maxLevel) {
    // Choose 1 summon multiple charge spell
    let summonMultipleSpells = Object.keys(spellcastingData.charge).filter(key => 
        spellcastingData.charge[key].use.includes('summon_multiple')
    );
    let result = [summonMultipleSpells[Math.floor(Math.random() * summonMultipleSpells.length)]];
    if (result == null || result.length === 0) {
        //Utils.server.tell('Re-rolling Summon Multiple Spells...');
        return chooseSummonMultipleSpells(maxTier, maxLevel);
    }
    return result;
}

/**
 * Each can have:
 * - 3 instant damage spells
 * - 4 charge damage spells
 * - 1 persistent damage spell
 * - 2 self-empower instant spells
 * - 1 ally-empower charge spell
 * - 1 control charge spell
 * - 1 followup charge spell
 * - 2 hinder spells
 * - 1 movement spell
 * - 1 reflect spell
 * - 1 summon multiple charge spell
 * 
 */

/**
 * 
 * 
 * function chooseAllSpells(maxTier, maxLevel, entity) {
    let chosenSpells = {}
    chosenSpells.damage = chooseDamageSpells(maxTier, maxLevel);
    chosenSpells.empower = chooseEmpowerSpells(maxTier, maxLevel);
    chosenSpells.control = chooseControlSpells(maxTier, maxLevel);
    chosenSpells.followup = chooseFollowupSpells(maxTier, maxLevel);
    chosenSpells.hinder = chooseHinderSpells(maxTier, maxLevel);
    chosenSpells.movement = chooseMovementSpells(maxTier, maxLevel);
    chosenSpells.reflect = chooseReflectSpells(maxTier, maxLevel);
    chosenSpells.summonMultiple = chooseSummonMultipleSpells(maxTier, maxLevel);
    entity.persistentData.put('chosenSpells', chosenSpells);
}

// When a variant boss is summoned, choose the spells

EntityEvents.spawned(event => {
    if (!event.entity.tags.contains('boss_variant')) return;
    chooseAllSpells(4, 10, event.entity);
    event.entity.persistentData.put('defensive_spell_timer', 0);
    event.entity.persistentData.put('offensive_spell_timer', 0);
    // Choose a summon_multiple spell from the chosenSpells persistent data
    let chosenSpells = event.entity.persistentData.get('chosenSpells');
    //Utils.server.tell(chosenSpells);
    let summonMultipleSpells = chosenSpells.summonMultiple;
    let spell = summonMultipleSpells[Math.floor(Math.random() * summonMultipleSpells.length)];
    let spellData = spellcastingData.charge[spell];
    // Cast the spell
    let target = event.entity;
    let caster = event.entity;
    let spellsToCast = {
        spell1: spell,
    }
    overhauledCast(caster, spellsToCast, target, 1);
    Utils.server.scheduleInTicks(100, e => {
        if (caster.alive) {
            e.repeating = true;
            overhauledCast(caster, spellsToCast, target, 1);
        } else {
            e.repeating = false;
        }
    })
})

EntityEvents.hurt(event => {
    if (!event.source.player) return;
    //if (!event.source.player.cooldowns.addCooldown())
    if (!event.entity.tags.contains('boss_variant')) return;
    if (event.entity.persistentData.defensive_spell_timer != 0) return; 
    let chosenSpells = event.entity.persistentData.get('chosenSpells');
    //let potentialSpells = [chosenSpells.damage, chosenSpells.empower, chosenSpells.hinder, chosenSpells.movement, chosenSpells.reflect];
    let potentialSpells = `${chosenSpells.damage},${chosenSpells.empower},${chosenSpells.hinder},${chosenSpells.movement},${chosenSpells.reflect}`;
    // remove all brackets from the string
    let bracketPotentialSpells = potentialSpells.replace(/[\[\]']+/g,'');
    // remove all "" from the string
    let cleanedPotentialSpells = bracketPotentialSpells.replace(/[""]+/g,'');
    // split the string into an array
    let potentialSpellsArray = cleanedPotentialSpells.split(',');
    // choose a random spell from potentialSpells
    let spell = potentialSpellsArray[Math.floor(Math.random() * potentialSpellsArray.length)];
    let spellData = spellcastingData.instant[spell];
    // Cast the spell
    let target = event.source.player;
    let caster = event.entity;
    let spellsToCast = {
        spell1: spell,
    }
    //Utils.server.tell(`Casting ${spell} on ${target}`);
    overhauledCast(caster, spellsToCast, target, 1);
    event.entity.persistentData.defensive_spell_timer+=1;
    Utils.server.scheduleInTicks(100, e => {
        if (caster.alive) {
            caster.persistentData.defensive_spell_timer = 0;
        } else {
            caster.persistentData.remove('defensive_spell_timer');
        }
    })

        
})

EntityEvents.hurt('player', event => {
    if (event.source == null) return;
    if (event.source.actual == null) return;
    if (!event.source.actual.tags.contains('boss_variant')) return;
    if (event.source.actual.persistentData.offensive_spell_timer != 0) return;
    // Choose a damage spell from the chosenSpells persistent data
    let possibilities = ['combo']
    // choose a random possibility
    let possibility = possibilities[Math.floor(Math.random() * possibilities.length)];
    let target = event.entity
    let caster = event.source.actual
    let chosenSpells = event.entity.persistentData.get('chosenSpells');
    if (possibility === 'damage') {
        let damageSpells = chosenSpells.damage;
        let spell = damageSpells[Math.floor(Math.random() * damageSpells.length)];
        let spellData = spellcastingData.instant[spell];
        // Cast the spell
        let spellsToCast = {
            spell1: spell,
        }
        overhauledCast(caster, spellsToCast, target, 1)
    } else {
        let controlSpells = chosenSpells.control;
        let followupSpells = chosenSpells.followup;
        let controlSpell = controlSpells[Math.floor(Math.random() * controlSpells.length)];
        let followupSpell = followupSpells[Math.floor(Math.random() * followupSpells.length)];
        let spellsToCast = {
            spell1: controlSpell,
            spell2: followupSpell
        }
        overhauledCast(caster, spellsToCast, target, 1)

    }
    event.source.actual.persistentData.offensive_spell_timer+=1;
    Utils.server.scheduleInTicks(100, e => {
        if (caster.alive) {
            caster.persistentData.offensive_spell_timer = 0;
        } else {
            caster.persistentData.remove('offensive_spell_timer');
        }
    })
})
 * 
 * 
 */








/**
 * Targeting overhaul: Tell what the target will be
 *  - Self
 *  - Enemy
 *  - Ally
 * 
 * 
 * 
 * Properties and their meaning:
 *  - cast_time: The time it takes to cast the spell. This can be the time it takes to charge it or the full time you spend casting it if its persistent
 *  - tier: The difficulty of the spell. This is used to determine the strength of the spell at §aLevel 1. The higher the tier, the stronger the spell
 *  - targeting: Who the spell is meant to affect. This can be the caster, an enemy, or an ally
 *  - max_level: The maximum level the spell can be. This is used to determine the strength of the spell at the highest level
 *  - school: The type of spell. This is used to determine the type of spell. Currently unused but may be used for spellcaster cults in the future
 *  - distance: The range that the enemy can be from you and the spell will still be effective. very close is less than 3 blocks, close is less than 10 blocks, any is any distance
 *  
 * Cast types:
 * > Instant: Spells that are casted instantly
 * > Charge: Spells that require a charge time
 * > Persistent: Spells that are casted and last for a certain amount of time
 * 
 * 
 * Types: 
 * 
 *  * Uses: (Add these to the bosses using "Aspects of the Eldritch") 
 *  - damage: Just damage
 *  - empower: healing or buffs
 *  - control: movement debuffs
 *  - followup: attacks meant to be used after a control spell
 *  - hinder: any debuff other than root
 *  - movement: spells that move the caster. Can be used for offense and defense. For defense just have the caster look at the player, turn 180 degrees, and then cast
 *  - reflect: spells that shield or reflect damage
 *  - summon_once: summon spells that the caster can only use once
 *  - summon_multiple: summon spells that the caster can use multiple times
 * 
 * How to Summon Boss Variants:
 *  - Use the Boss Effigy with the selected boss
 *  - Add a Spell Tier item to the altar. This determines what tier of spells the boss will randomly cast
 *  - Add a Spell Power item to the altar. This determines what power level the boss will cast the spells at. 5 types. Takes the spell max level, divides it by 5, and takes the slice that is the spell power level. Rounds down
 *  
 * Boss mods:
 *  - Burst Mod, 4 tiers: The boss will cast the spell multiple times in a row. Only works with instant spells. 2x, 3x, 4x, 5x
 *  - sacrifice mod: Allows the boss to summon a "summon_once" spell one time at the beginning of the fight.
 * 
 * 
 * 
 *  > Randomly choose:
 *   - 8 Spells from the damageSpells array
 *   - 1 spell from the persistentSpells array
 *   - 1 spell from the controlSpells array
 *   - 1 spell from the followupSpells array
 *   - 1 spell from the empowerSpells array
 *   - 1 spell from the movementSpells array
 *   - 1 spell from the summonMultipleSpells array
 *   - 1 spell from the hinderSpells array
 */
// 1. All spells from 'instant' or 'charge' that have the 'damage' use
// Define your spellcastingData object with 'id' fields elsewhere in your scripts
// Example: Include this script after the spellcastingData object is defined
