



/**
 * Ability slots are from right to left
 *  - Since L is the farthest right key, it is slot 1
 *  - Since K is the next key to the left, it is slot 2
 *  - Since J is the next key to the left, it is slot 3
 *  - Since H is the next key to the left, it is slot 4
 *  - Since G is the farthest left key, it is slot 5
 */

// Shared layout constants (right-aligned HUD, negative Y is higher)
const ABILITY_COLUMNS = 5;
const ABILITY_BASE_X = -5;     // right-most column
const ABILITY_X_STEP = -17;    // distance between columns (to the left)
const NON_KEY_BASE_Y = -74;    // first row of non-key abilities
const ROW_Y_STEP = -20;        // distance between rows (going up)

// Slot → hotkey letter mapping for key abilities
const KEY_SLOT_KEYS = {
    1: 'l',
    2: 'k',
    3: 'j',
    4: 'h',
    5: 'g',
};

// Key abilities that have separate swappable/cycleable effect icons
const CYCLE_EFFECT_ABILITIES = ['gjallarhorn', 'inspire'];

/**
 * Compute the on-screen position for a key ability slot.
 *
 * @param {number} slot        Slot index 1–5 (right to left)
 * @param {number} keyRowIndex Row index where keys should be drawn
 * @returns {{x:number,y:number}|null}
 */
function getKeySlotPosition(slot, keyRowIndex) {
    slot = parseInt(slot);
    if (isNaN(slot) || slot < 1 || slot > 5) return null;

    if (typeof keyRowIndex !== 'number' || keyRowIndex < 1) {
        keyRowIndex = 1;
    }

    let x = ABILITY_BASE_X + ABILITY_X_STEP * (slot - 1);
    let y = NON_KEY_BASE_Y + ROW_Y_STEP * keyRowIndex;

    return { x: x, y: y };
}

/**
 * Compute the on-screen position for a non-key ability slot.
 * This reproduces the existing 0–20 coordinate table and
 * extends naturally to additional rows.
 *
 * @param {number} slot Slot index 0+
 * @returns {{x:number,y:number}|null}
 */
function getNonKeySlotPosition(slot) {
    slot = parseInt(slot);
    if (isNaN(slot) || slot < 0) return null;

    let col = slot % ABILITY_COLUMNS;
    let row = Math.floor(slot / ABILITY_COLUMNS);

    let x = ABILITY_BASE_X + ABILITY_X_STEP * col;
    let y = NON_KEY_BASE_Y + ROW_Y_STEP * row;

    return { x: x, y: y };
}

function getKeySlotKey(slot) {
    slot = parseInt(slot);
    if (isNaN(slot)) return null;
    return KEY_SLOT_KEYS[slot] || null;
}

// Hide the small effect icon for cycleable key abilities when they are not active
function hideCycleEffect(player, ability) {
    if (player == null || ability == null) return;
    ability = ability.toString();
    if (CYCLE_EFFECT_ABILITIES.indexOf(ability) === -1) return;

    let paint_data = {};
    paint_data[ability] = {
        type: 'rectangle',
        visible: 'false',
    };
    player.paint(paint_data);
}

/**
 * Determine which row the key abilities should sit on for this player.
 * Keys are always exactly one row above the highest occupied non-key row.
 *
 * Row indexing:
 *  - Non-key row 0 starts at NON_KEY_BASE_Y
 *  - Row index increases upward by ROW_Y_STEP
 *  - Keys occupy row (highestNonKeyRow + 1), with a minimum of 1
 */
function getKeyRowIndexForPlayer(player) {
    if (player == null) return 1;

    let highestRow = -1;
    let non_key_abilities = skill_data.paint_nonkey_ability;

    non_key_abilities.forEach(function (ability) {
        if (!player.persistentData.get(ability + `_unlocked`)) return;
        if (typeof getNonKeySlot !== 'function') return;

        let slot = getNonKeySlot(player, ability);
        if (slot == null || slot === 'null') return;
        slot = parseInt(slot);
        if (isNaN(slot) || slot < 0) return;

        let row = Math.floor(slot / ABILITY_COLUMNS);
        if (row > highestRow) highestRow = row;
    });

    if (highestRow < 0) return 1;   // no non-key abilities yet: keep keys on baseline row 1
    return highestRow + 1;          // one row above the highest non-key row
}

function paint_ability_cycle (player, icon, ability) {
    if (player == null) return
    if (icon == null) return
    if (ability == null) return
    let slot = getAbilityKeySlot(player, ability)
    if (slot == null) return
    let keyRowIndex = getKeyRowIndexForPlayer(player)
    let pos = getKeySlotPosition(slot, keyRowIndex)
    if (pos == null) return
    let x = pos.x
    let y = pos.y
    let path = `kubejs:textures/icons/skill_tree/${ability}/${icon}.png`
    let paint_data = {}
    paint_data[ability] = {
        type: 'rectangle',
        x: x-2,
        y: y-30,
        w: 12,
        h: 12,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: `${path}`
    }
 
    player.paint(paint_data)
}

/**
 * ItemEvents.rightClicked('kubejs:testing', event => {
    let runes_to_give = [
        'kubejs:rune_of_the_mercenary',
        'kubejs:rune_of_the_plague_doctor',
        'kubejs:rune_of_the_nomad',
        'kubejs:rune_of_the_mystic',
        'kubejs:rune_of_the_disciple',
    ]
    runes_to_give.forEach(rune => {
        event.player.give(rune)
    })
    repaintAbilities(event.player)
    
})
 * 
 */



/**
 * Repaints every ability for the given player by invoking your paint_ability routine.
 * @param {ServerPlayerEntity} player
 */
function repaintAbilities(player) {
    // Replace this with however you store your full list of abilities;
    // here we assume you have an array of all ability IDs in skill_data.paint_key_abilities
    let paint_key_abilities = skill_data.paint_key_abilities;
    paint_key_abilities.forEach(ability => {
        if (player.persistentData.get(ability+`_unlocked`)) {
            //tell('repainting ability: ' + ability)
            if (isSkillCoolingDown(player, ability)) {
                runAbilityAction(player, ability, 'hide')
            } else {
                runAbilityAction(player, ability, 'paint')
            }
            
        }
    });
    let non_key_abilities = skill_data.paint_nonkey_ability
    non_key_abilities.forEach(ability => {
        if (player.persistentData.get(ability+`_unlocked`)) {
            //tell('repainting ability: ' + ability)
            if (isSkillCoolingDown(player, ability)) {
                runAbilityAction(player, ability, 'hide')
            } else {
                runAbilityAction(player, ability, 'paint')
            }
        }
    });
}

ServerEvents.customCommand('repaintAbilities', event => {
    let player = event.player;
    if (!player) return;
    repaintAbilities(player);
})


PlayerEvents.loggedIn(event => {
    repaintAbilities(event.player)
})




function paint_ability (player, icon, ability) {
    //console.log(ability)
    if (player == null) return
    if (icon == null) return
    if (ability == null) return

    let slot = getAbilityKeySlot(player, ability)

    if (slot == null) {
        let slot_attempt = setAbilityKeySlot(player, ability)
        if (!slot_attempt) {
            return
        } else {
            slot = getAbilityKeySlot(player, ability)
        }
    }

    if (ability == 'gjallarhorn') {
        let icon = gjallarhorn_icons[player.persistentData.getInt('current_gjallarhorn')]
        hide_ability(player, 'gjallarhorn')
        paint_ability_cycle(player, icon, 'gjallarhorn')
    } else {
        if (slot == getAbilityKeySlot(player, 'gjallarhorn') || getAbilityKeySlot(player, 'gjallarhorn') == null) {
            hide_ability(player, 'gjallarhorn')
        }
    }
    if (ability == 'inspire') {
        let icon = inspire_icons[player.persistentData.getInt('current_inspire')]
        hide_ability(player, 'inspire')
        paint_ability_cycle(player, icon, 'inspire')
    } else {
        if (slot == getAbilityKeySlot(player, 'inspire') || getAbilityKeySlot(player, 'inspire') == null) {
            hide_ability(player, 'inspire')
        }
    }
    if (ability == 'base_parry') {
        let icon = player.persistentData['current_parry']
        hide_ability(player, 'base_parry')
        paint_ability_cycle(player, icon, 'base_parry')
    } else {
        if (slot == getAbilityKeySlot(player, 'base_parry') || getAbilityKeySlot(player, 'base_parry') == null) {
            hide_ability(player, 'base_parry')
        }
    }

    // If we just swapped away from a cycleable ability (gjallarhorn/inspire),
    // and it no longer has any key slot, hide its effect icon so it doesn't
    // float above the new ability in that slot.
    CYCLE_EFFECT_ABILITIES.forEach(function (cycleAbility) {
        if (cycleAbility === ability) return;
        let cycleSlot = getAbilityKeySlot(player, cycleAbility);
        if (cycleSlot == null || cycleSlot == undefined) {
            hideCycleEffect(player, cycleAbility);
        }
    });

    let keyRowIndex = getKeyRowIndexForPlayer(player)
    let pos = getKeySlotPosition(slot, keyRowIndex)
    if (pos == null) return
    let x = pos.x
    let y = pos.y
    let path = `kubejs:textures/icons/skill_tree/${ability}/${icon}.png`

    let keyLetter = getKeySlotKey(slot)
    if (keyLetter != null) {
        paintKey(player, keyLetter, x, y)
    }
    let paint_data = {}
    paint_data[slot] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 16,
        h: 16,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: `${path}`
    }
    player.paint(paint_data)
    if (ability == 'gjallarhorn') {}
    getAbilityKeySlot(player, ability)
    /**
     * 
     *     if (ability == 'gjallarhorn') {
        let icon = gjallarhorn_icons[player.persistentData.getInt('current_gjallarhorn')]
        paint_ability_cycle(player, icon, 'gjallarhorn')
    } else {
        if (slot == getKeySlot(player, 'gjallarhorn')) {
            hide_ability(player, 'gjallarhorn')
        }
    }
    if (ability == 'inspire') {
        let icon = inspire_icons[player.persistentData.getInt('current_inspire')]
        paint_ability_cycle(player, icon, 'inspire')
    } else {
        if (slot == getKeySlot(player, 'inspire')) {
            hide_ability(player, 'inspire')
        }
    }

    if (ability == 'base_parry') {
        let icon = player.persistentData['current_parry']
        paint_ability_cycle(player, icon, 'base_parry')
    } else {
        if (slot == getKeySlot(player, 'base_parry')) {
            hide_ability(player, 'base_parry')
        }
    }
     * 
     */

}


function hide_ability(player, ability) {
    // let paint_data = {}
    // ability = ability.toString()

    // // Always hide the effect icon for cycleable abilities when we hide the base icon
    // hideCycleEffect(player, ability)

    // let slot = getAbilityKeySlot(player, ability)
    // if (slot == null || slot == undefined) return

    // let path = `kubejs:textures/icons/skill_tree/${ability}/${ability}_gray.png`
    // paint_data[slot] = {
    //     texture: path
    // }
    // if (paint_data[slot] == null || paint_data[slot] == undefined) return
    // player.paint(paint_data)

    // let key = getKeySlotKey(slot)
    // if (key == null || key == undefined) return
    // //hideKey(player, key)

    // Always hide the effect icon for cycleable abilities when we hide the base icon
    let paint_data = {}
    ability = ability.toString()
    hideCycleEffect(player, ability)

    let slot = getAbilityKeySlot(player, ability)
    if (slot == null || slot == undefined) return

    let keyRowIndex = getKeyRowIndexForPlayer(player)
    let pos = getKeySlotPosition(slot, keyRowIndex)
    if (pos == null) return
    let x = pos.x
    let y = pos.y

    let path = `kubejs:textures/icons/skill_tree/${ability}/${ability}_gray.png`
    paint_data[slot] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 16,
        h: 16,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: path
    }
    player.paint(paint_data)

    let key = getKeySlotKey(slot)
    if (key == null || key == undefined) return
    //hideKey(player, key)
}


/**
 * ItemEvents.rightClicked('kubejs:testing', event => {
    skill_data.paint_key_abilities.forEach(ability => {
        paint_ability(event.player, ability, ability)
    })
})
 * 
 */



/**
 * Returns the slot number (1–5) for the given ability, or null if not assigned.
 *
 * @param {ServerPlayerEntity} player
 * @param {string} ability  The ability name to look up
 * @returns {number|null}
 */
function getAbilityKeySlot(player, ability) {
    if (player.persistentData[`gKeyCurrent`] == ability) {
        return 5;
    }
    if (player.persistentData[`hKeyCurrent`] == ability) {
        return 4;
    }
    if (player.persistentData[`jKeyCurrent`] == ability) {
        return 3;
    }
    if (player.persistentData[`kKeyCurrent`] == ability) {
        return 2;
    }
    if (player.persistentData[`lKeyCurrent`] == ability) {
        return 1;
    }
    //tell(`Ability ${ability} not bound to any key slot`);
    return null;

}

/**
 * Attempts to bind an ability to the next available key slot.
 * @param {ServerPlayerEntity} player 
 * @param {string} ability 
 * @returns {boolean} true if bound to a slot, false if already bound or no empty slot available
 */
function setAbilityKeySlot(player, ability) {
    if (getAbilityKeySlot(player, ability) != null) {
        //tell('Already bound to a key slot');
        return false;
    }

    // 2️⃣ Define slots from bottom (L=1) up to top (G=5)
    let slotKeys = [
        'lKeyCurrent',  // slot 1
        'kKeyCurrent',  // slot 2
        'jKeyCurrent',  // slot 3
        'hKeyCurrent',  // slot 4
        'gKeyCurrent'   // slot 5
    ];

    // 3️⃣ Find the first empty one
    for (let key of slotKeys) {
        // if the tag doesn't exist or is the empty string, consider it free
        if (!player.persistentData.get(key) || player.persistentData.getString(key) === '') {
            player.persistentData.putString(key, ability);
            //tell(`Bound ${ability} to ${key}`);
            return true;
        }
    }

    // 4️⃣ No free slots
    //tell('No free slots available');
    return false;
}



function paintKey(player, key, x, y) {
    if (player == null) return
    if (key == null) return
    x = x-2.5
    y = y-17
    let paintData = {}
    paintData[`${key}_key`] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 10.5,
        h: 10.5,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: `kubejs:textures/icons/keys/${key}.png`
    }
    player.paint(paintData)
}


function hideKey(player, key) {
    let paintData = {}
    paintData[`${key}_key`] = {
        type: 'rectangle',
        visible: 'false',
    }
    player.paint(paintData)
}

let key_abilities = global.skillData.paint_key_abilities
let skill_data = global.skillData

function paint_non_key_ability (player, icon, slot, ability) {
    if (player == null) return
    if (icon == null) return
    if (slot == null) return
    if (ability == null) return
    slot = parseInt(slot)
    //tell(slot)
    let pos = getNonKeySlotPosition(slot)
    if (pos == null) return
    let x = pos.x
    let y = pos.y
    let path = `kubejs:textures/icons/skill_tree/${ability}/${icon}.png`
    if (icon.includes('parry')) {
        path = `kubejs:textures/icons/swappables/parry/${icon}.png`
    }
    let paint_data = {}
    paint_data[ability] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 16,
        h: 16,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: path
    }
    player.paint(paint_data)
}

function hide_non_key_ability(player, ability) {
    // let paint_data = {}
    // let path = `kubejs:textures/icons/skill_tree/${ability}/${ability}_gray.png`
    // paint_data[ability] = {
    //     texture: path
    // }
    // player.paint(paint_data)

    let paint_data = {}
    if (ability == null) return

    let slot = getNonKeySlot(player, ability)
    if (slot == null) return
    slot = parseInt(slot)

    let pos = getNonKeySlotPosition(slot)
    if (pos == null) return
    let x = pos.x
    let y = pos.y

    let path = `kubejs:textures/icons/skill_tree/${ability}/${ability}_gray.png`
    paint_data[ability] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 16,
        h: 16,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: path
    }
    player.paint(paint_data)
}


/* ────────────────────────────────────────────────────────────────────────── */
/*  Ability dispatch table (Option A)                                        */
/*  Dynamically route paint/hide by ability kind (key vs non_key)            */
/* ────────────────────────────────────────────────────────────────────────── */
const AbilityFns = {
  key: {
    paint: function (player, ability) {
        //console.log(ability)
      // Default icon == ability id
      return paint_ability(player, ability, ability)
    },
    hide: function (player, ability) {
      return hide_ability(player, ability)
    }
  },
  non_key: {
    paint: function (player, ability) {
        //console.log(ability)
      const slot = (typeof getNonKeySlot === 'function') ? getNonKeySlot(player, ability) : null
      // Default icon == ability id
      return paint_non_key_ability_new(player, ability)
    },
    hide: function (player, ability) {
      return hide_non_key_ability(player, ability)
    }
  }
}

function getAbilityKind(ability) {
  const keys = (global.skillData && global.skillData.paint_key_abilities) || []
  const nks  = (global.skillData && global.skillData.paint_nonkey_ability) || []
  if (keys.includes(ability)) return 'key'
  if (nks.includes(ability))  return 'non_key'
  return 'unknown'
}

function runAbilityAction(player, ability, action /* 'paint' | 'hide' */) {
  const kind = getAbilityKind(ability)
  //console.log(ability)
  const table = AbilityFns[kind]
  if (!table) return false
  const fn = table[action]
  if (typeof fn !== 'function') return false
  fn(player, ability)
  return true
}

/* ────────────────────────────────────────────────────────────────────────── */
/*  Hot-key map & registration (Rhino-safe)                                 */
/* ────────────────────────────────────────────────────────────────────────── */
const HOT_KEYS = [
  { letter: 'g', channel: 'global.gKeySpecial.consumeClick' },
  { letter: 'h', channel: 'global.hKeySpecial.consumeClick' },
  { letter: 'j', channel: 'global.jKeySpecial.consumeClick' },
  { letter: 'k', channel: 'global.kKeySpecial.consumeClick' },
  { letter: 'l', channel: 'global.lKeySpecial.consumeClick' },
];

// No destructuring; plain function callback
HOT_KEYS.forEach(function (keyInfo) {
  NetworkEvents.dataReceived(keyInfo.channel, function (event) {
    //console.log(keyInfo.letter)
    handleHotKey(event, keyInfo.letter);
  });
});

// Count how many key abilities this player has unlocked
function countUnlockedKeyAbilities(player) {
  if (!player || !player.persistentData) return 0;
  let count = 0;
  key_abilities.forEach(function (ability) {
    if (player.persistentData.get(`${ability}_unlocked`)) {
      count++;
    }
  });
  return count;
}

/* ────────────────────────────────────────────────────────────────────────── */
/*  Core handler – identical behaviour for every key letter                 */
/* ────────────────────────────────────────────────────────────────────────── */
function handleHotKey(event, keyLetter) {
  const player = event.player;
  const tag    = `${keyLetter}KeyCurrent`;   // e.g. "gKeyCurrent"

  /* -----------------------------------------------------------
     ⇧-key → cycle to next unlocked + un-slotted ability
     ----------------------------------------------------------- */
  if (player.shiftKeyDown) {
    // Only attempt to cycle if there are more than 5 unlocked key abilities
    // (5 can all be shown at once; 6+ require cycling)
    if (countUnlockedKeyAbilities(player) <= 5) {
      return;
    }

    let current     = player.persistentData.getString(tag);
    let currentNum  = ability_slot_num(player, current);

    /* -- advance until we hit an ability that isn't slotted anywhere -- */
    let checked = 0, next = null, nextNum = currentNum;

    do {
      nextNum += 1;
      next = getAbilityFromNumber(player, nextNum);

      if (next == null) {               // ran past end → wrap to 1
        nextNum = 1;
        next = getAbilityFromNumber(player, nextNum);
      }

      checked++;
    } while (next && abilitySlotted(player, next) && checked <= key_abilities.length);
    if (!player.persistentData.get(`${next}_unlocked`)) return; // don't cycle if ability is not unlocked
    if (!next || abilitySlotted(player, next)) next = current;  // every ability is taken
    player.persistentData.putString(tag, next);
    paint_ability(player, next, next);
    Utils.server.runCommandSilent(
      `/execute in ${player.level.dimension} run playsound` +
      ` irons_spellbooks:cast.ice_block player ${player.username}` +
      ` ${player.x} ${player.y} ${player.z} 0.2 2`
    );
    return;
  }

  /* -----------------------------------------------------------
     Plain key → cast the currently selected ability
     ----------------------------------------------------------- */
  if (!player.persistentData.get(tag)) return;

  const abilityName = player.persistentData.getString(tag);
  const fn = this[abilityName + '_ability'];
  
  if (typeof fn === 'function') fn(player);
}


function removeAllKeyCurrent(player) {
    let keys = ['g', 'h', 'j', 'k', 'l'];
    keys.forEach(k => {
        player.persistentData.remove(`${k}KeyCurrent`);
    });
    // Utils.server.tell('Removed all key current tags');
}




/* ────────────────────────────────────────────────────────────────────────── */
/*  abilitySlotted stays exactly as you already have it                     */
/* ──────────────────────────────────────────────────────────────────────────
function abilitySlotted(player, ability) {
  if (!player || !ability) return false;
  let keys = ['g', 'h', 'j', 'k', 'l'];
  return keys.some(k => player.persistentData.getString(`${k}KeyCurrent`) === ability);
}
*/



/* ============================================================================
   Helper – Is this ability already slotted on any other hot-key?
   ========================================================================= */
function abilitySlotted(player, ability) {
    if (!player || !ability) return false;

    let keys = ['g', 'h', 'j', 'k', 'l'];
    return keys.some(k => player.persistentData.getString(`${k}KeyCurrent`) === ability);
}


function paint_non_key_ability_new (player, ability) {
    if (player == null) return
    let icon = ability
    let slot = getNonKeySlot(player, ability)
    if (icon == null) return
    if (slot == null) return
    if (ability == null) return
    slot = parseInt(slot)
    //tell(slot)
    let pos = getNonKeySlotPosition(slot)
    if (pos == null) return
    let x = pos.x
    let y = pos.y
    let path = `kubejs:textures/icons/skill_tree/${ability}/${icon}.png`
    if (icon.includes('parry')) {
        path = `kubejs:textures/icons/swappables/parry/${icon}.png`
    }
    let paint_data = {}
    paint_data[ability] = {
        type: 'rectangle',
        x: x,
        y: y,
        w: 16,
        h: 16,
        draw: 'ingame',
        alignX: 'right',
        alignY: 'bottom',
        visible: 'true',
        texture: path
    }
    //console.log(`Painting ${ability} for ${player.username}`)
    player.paint(paint_data)
}




