

/**
 * Infuses a Skill Totem with a new skill. If `skillId` is provided (e.g. 'kubejs:ice_beam'),
 * it will attempt to add that specific skill; otherwise it picks one at random that isn’t already present.
 * 
 * Totem capacity:
 *  - 'kubejs:lesser_skill_totem'  → max 1 skill
 *  - 'kubejs:greater_skill_totem' → max 3 skills
 *  - 'kubejs:exalted_skill_totem' → max 5 skills
 * 
 * Each infusion appends the skill’s “Display Name:” and its description as lore lines on the totem,
 * with a blank line after each entry.
 * 
 * @param player   The player performing the infusion.
 * @param totItem  The ItemStack of the Skill Totem to infuse.
 * @param skillId  (Optional) Full ID of the skill to add, e.g. 'kubejs:ice_beam'. If omitted, picks randomly.
 * @returns        true if infusion succeeded; false if the totem was full or already has that skill or no skill available.
 */
function infuseSkillTotem(player, totItem, skillId, ignoreCaps) {
  if (!totItem || totItem.empty) return false;

  // Normalize skillId
  if (skillId === 'random') skillId = null;
  if (skillId && !skillId.includes('kubejs:')) {
    skillId = 'kubejs:' + skillId;
  }
    Utils.server.runCommandSilent(`/ftbquests change_progress ${player.username} reset 61B672EF2B04B3EB`)
    Utils.server.runCommandSilent(`/ftbquests change_progress ${player.username} complete 61B672EF2B04B3EB`)
    player.persistentData.putBoolean('orb_of_discovery', true)

  let totemId = totItem.id;
  let maxSlots = {
    'kubejs:lesser_skill_totem': 5,
    'kubejs:greater_skill_totem': 5,
    'kubejs:exalted_skill_totem': 5
  }[totemId] || 0;
  if (maxSlots === 0) return false;

  // Ensure basic NBT shape
  totItem.nbt = totItem.nbt || {};
  totItem.nbt.skills = Array.isArray(totItem.nbt.skills) ? totItem.nbt.skills : (totItem.nbt.skills || []);
  totItem.nbt.display = totItem.nbt.display || {};
  let existingSkills = totItem.nbt.skills;

  if (!ignoreCaps) {
    if (existingSkills.length >= maxSlots) {
      if (existingSkills.length < 5) {
        player.tell(Text.of([
          Text.of('Upgrade ').green(),
          Text.of('this Totem to add more skills').gray(),
        ]));
      } else if (existingSkills.length === 5) {
        player.tell(Text.of([
          Text.of('Corrupt ').darkRed(),
          Text.of('this Totem to add more skills').gray(),
        ]));
      } else {
        player.tell(Text.red('This Totem is full!'));
      }
      return false;
    }
  }

  let skillKey;
  if (skillId) {
    skillKey = skillId.split(':')[1];
    if (existingSkills.some(s => s === skillKey)) return false;
    if (!(skillKey in skillMapping)) return false;
  } else {
    let allKeys = Object.keys(skillMapping);
    let available = allKeys.filter(k => !existingSkills.toString().includes(k));
    if (available.length === 0) return false;
    skillKey = available[Math.floor(Math.random() * available.length)];
  }

  if (hasTotemSkill(totItem, skillKey)) {
    // recurse to find a new one
    infuseSkillTotem(player, totItem, 'random', ignoreCaps);
    return true;
  }

  // --- Level roll & NBT write via CompoundTag API ---
  (function () {
    let propsToCheck = ['time', 'cooldown', 'percent', 'hits', 'entity_count', 'amplifier'];
    let def = skillMapping[skillKey] || {};
    let hasLevelEligibleProp = typeof def === 'object' && propsToCheck.some(p => Object.prototype.hasOwnProperty.call(def, p));

    let l2Chance = 0.30, l3Chance = 0.20;
    let totemLevel = 1;
    if (totemId === 'kubejs:greater_skill_totem') { totemLevel = 2; l2Chance = 0.40; l3Chance = 0.30; }
    if (totemId === 'kubejs:exalted_skill_totem') { totemLevel = 3; l2Chance = 0.50; l3Chance = 0.50; }

    let level = 1;
    if (hasLevelEligibleProp && totemLevel) {
      if (Math.random() < l2Chance) {
        level = 2;
        if (Math.random() < l3Chance) level = 3;
      }
    }

    if (!totItem.nbt.contains('skill_levels')) {
      totItem.nbt.put('skill_levels', {}); // creates a CompoundTag
    }
    let levelsTag = totItem.nbt.get('skill_levels'); // CompoundTag
    if (!levelsTag.contains(skillKey)) {
      levelsTag.putInt(skillKey, level);
    }
  })();
  // --- END level roll ---

  existingSkills.push(skillKey);
  totItem.nbt.skills = existingSkills;

  // Read back level for chat message
  let displayLevel = 1;
  if (totItem.nbt.contains('skill_levels')) {
    let levelsTag = totItem.nbt.get('skill_levels');
    if (levelsTag.contains(skillKey)) {
      displayLevel = Math.max(1, Math.min(3, levelsTag.getInt(skillKey)));
    }
  }

  player.tell(Text.of([
    Text.of('Skill added: ').gray(),
    Text.of(skillKey.split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' ')).green(),
    Text.of(` (Lv. ${displayLevel})`).gold()
  ]));

  rebuildTotemLore(totItem, !!ignoreCaps);
  return true;
}











/**
 * Returns the number of skills currently on a Skill Totem.
 *
 * @param {Internal.ItemStackJS} totItem
 * @returns {number}  count of skills, or 0 if none
 */
function countTotemSkills(totItem) {
  if (!totItem.nbt || !totItem.nbt.skills) return 0;

  let count = 0;
  for (let entry of totItem.nbt.skills) {
    // Only count non-empty entries
    if (entry != null && String(entry).trim() !== '') {
      count++;
    }
  }
  return count;
}


/**
 * Checks whether a Skill Totem has a given skill.
 *
 * @param {Internal.ItemStackJS} totItem
 * @param {string} skillKey key of the skill to check (with or without namespace)
 * @returns {boolean} true if the skill is present
 */
function hasTotemSkill(totItem, skillKey) {
  // Must have NBT skills list
  if (!totItem.nbt || !totItem.nbt.skills) return false;
  //tell(skillKey)
  // Normalize the skillKey (strip namespace if provided)
  let rawKey = String(skillKey);
  if (rawKey.includes(':')) rawKey = rawKey.split(':')[1];
  //tell(`Checking for skill: ${rawKey} against Skill Key: ${skillKey}`);
  // Iterate through stored skills
  for (let entry of totItem.nbt.skills) {
    let k = String(entry);
    if (k.includes(':')) k = k.split(':')[1];
    if (k === rawKey) {
      return true;
    }
  }

  return false;
}



/**
 * Ensures the locked skill is present in the totem’s skills list.
 * If there’s a lockedSkill NBT and it’s missing from skills, adds it.
 *
 * @param {Internal.ItemStackJS} totItem
 * @returns {boolean}  true if it fixed a discrepancy, false otherwise
 */
function fixLockedSkillDiscrep(totItem) {
  // Must have nbt and lock
  if (!totItem.nbt) return false;
  let rawLocked = String(totItem.nbt.lockedSkill ?? '').trim();
  if (!rawLocked) return false;

  // Normalize locked key
  let lockedKey = rawLocked.includes(':') ? rawLocked.split(':')[1] : rawLocked;

  // Ensure skills array exists
  totItem.nbt.skills = Array.isArray(totItem.nbt.skills) ? totItem.nbt.skills : [];
  
  // Normalize existing skills into bare keys
  let skills = [];
  for (let entry of totItem.nbt.skills) {
    let k = String(entry);
    if (k.includes(':')) k = k.split(':')[1];
    skills.push(k);
  }

  // If lockedKey already present, nothing to do
  if (skills.includes(lockedKey)) return false;

  // Otherwise, add it and rebuild lore
  skills.push(lockedKey);
  totItem.nbt.skills = skills;
  rebuildTotemLore(totItem, false);
  //tell(`Locked skill [${lockedKey}] was missing and has been re-added.`);
  return true;
}



function lockTotemSkill(player, totItem) {
  totItem.nbt = totItem.nbt || {};
  let skills = (totItem.nbt.skills || []).map(s => String(s).replace(/"/g, ''));
  
  // new code
  let maxLocks = 1
  if (totItem.id.includes('greater_skill_totem')) maxLocks = 2
  if (totItem.id.includes('exalted_skill_totem')) maxLocks = 3
  // 

  // Normalize existing locks into a JS array (handles Java List safely)
  let locked = [];
  if (totItem.nbt.lockedSkills) {
    let raw = totItem.nbt.lockedSkills;
    for (let i = 0; i < raw.length; i++) locked.push(String(raw[i]));
  }
  if (totItem.nbt.lockedSkill && !locked.includes(String(totItem.nbt.lockedSkill))) {
    locked.push(String(totItem.nbt.lockedSkill));
  }

  if (skills.length === 0 || locked.length >= maxLocks) return false;

  // pick from unlockeds only
  let available = skills.filter(s => !locked.includes(s));
  if (available.length === 0) {
    player.tell(Text.red('No skills can be locked'))
    return false;
  }

  let chosen = available[Math.floor(Math.random() * available.length)];
  locked.push(chosen);

  // persist multi + legacy single
  totItem.nbt.lockedSkills = locked;
  totItem.nbt.lockedSkill = locked[0];

  // display formatting
  let displayName = chosen
    .split('_')
    .map(word => {
      let s = String(word).trim();
      return s.length > 0 ? s.charAt(0).toUpperCase() + s.slice(1) : '';
    })
    .join(' ');

  // Rebuild lore so the locked skill's title is highlighted gold
  rebuildTotemLore(totItem, false)
  player.tell(Text.of([ Text.of('Locked skill: ').gray(), Text.of(displayName).gold() ]));
  return true;
}



/**
 * Removes a specific skill (or a random one) from the skill totem, skipping the locked one.
 *
 * @param {Internal.ItemStackJS} totItem
 * @param {string} [skillKey]   – key (without namespace) of the skill to remove, or "random"/undefined to pick one
 * @returns {boolean}           – true if a skill was removed
 */
function removeRandomTotemSkill(player, totItem) {
  if (!totItem.nbt || !totItem.nbt.skills) return false;
 
  let skills = totItem.nbt.skills
  //tell(`Skills before removal: ${skills}`)
  let locked = totItem.nbt['lockedSkill'] || null;
  if (locked && skills.toString().includes(locked)) {
    skills = skills.filter(s => s !== locked);
  } else {
    locked = null; // No locked skill, so we can remove any skill
  }
  if (skills.length === 0) {
    //tell('No skills to remove!')
    return false;
  }
  // Pick a random skill to remove
  let randomIndex = Math.floor(Math.random() * skills.length);
  let removedSkill = skills[randomIndex];

  let removedSkillKey = String(removedSkill).replace(/"/g, '').replace(/^\[|\]$/g, '').split('_')
    .map(word => word.charAt(0).toUpperCase() + word.slice(1))
    .join(' ');
  let msg = Text.of([
    Text.of('Removed skill: ').gray(),
    Text.of(removedSkillKey).red()
  ])
  player.tell(msg);
  // Remove the skill from the totem 
  totItem.nbt.skills = skills.filter(s => s !== removedSkill);
  fixLockedSkillDiscrep(totItem)
  // Rebuild the lore
  rebuildTotemLore(totItem, false);

  return true;
}

// Safer, normalized variant that respects multi-locks and cleans level entries
function removeRandomTotemSkillSafe(player, totItem) {
  if (!totItem || !totItem.nbt || !totItem.nbt.skills) return false;

  let norm = (raw) => {
    try {
      let s = String(raw == null ? '' : raw);
      s = s.replace(/\\/g, '');
      s = s.replace(/^[\s\[\]"']+/, '').replace(/[\s\[\]"']+$/, '');
      if (s.includes(':')) s = s.split(':')[1];
      return s.trim();
    } catch(_) { return ''; }
  };

  let skills = [];
  try {
    let raw = totItem.nbt.skills;
    for (let i = 0; i < raw.length; i++) {
      let k = norm(raw[i]);
      if (k) skills.push(k);
    }
  } catch(_) {}

  let lockedSet = new Set();
  try {
    let ls = totItem.nbt.lockedSkills;
    if (ls && typeof ls.length === 'number') {
      for (let i = 0; i < ls.length; i++) {
        let v = norm(ls[i]);
        if (v) lockedSet.add(v);
      }
    }
    if (totItem.nbt.lockedSkill) {
      let v = norm(totItem.nbt.lockedSkill);
      if (v) lockedSet.add(v);
    }
  } catch(_) {}

  let removable = skills.filter(k => !lockedSet.has(k));
  if (removable.length === 0) return false;

  let removedKey = removable[Math.floor(Math.random() * removable.length)];
  let removedDisplay = removedKey.split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' ');
  player.tell(Text.of([ Text.of('Removed skill: ').gray(), Text.of(removedDisplay).red() ]));

  let remaining = skills.filter(k => k !== removedKey);
  totItem.nbt.skills = remaining;

  try {
    if (totItem.nbt.contains && totItem.nbt.contains('skill_levels')) {
      let oldTag = totItem.nbt.get('skill_levels');
      let newLevels = {};
      for (let i = 0; i < remaining.length; i++) {
        let k = remaining[i];
        if (oldTag.contains && typeof oldTag.contains === 'function' && oldTag.contains(k)) newLevels[k] = oldTag.getInt(k);
        else if (typeof oldTag[k] === 'number') newLevels[k] = oldTag[k];
      }
      if (typeof totItem.nbt.put === 'function') {
        totItem.nbt.put('skill_levels', {});
        let newTag = totItem.nbt.get('skill_levels');
        for (let k in newLevels) {
          if (Object.prototype.hasOwnProperty.call(newLevels, k)) {
            if (typeof newTag.putInt === 'function') newTag.putInt(k, newLevels[k]); else newTag[k] = newLevels[k];
          }
        }
      } else {
        totItem.nbt.skill_levels = newLevels;
      }
    } else if (totItem.nbt.skill_levels) {
      let dst = {};
      for (let k of remaining) if (typeof totItem.nbt.skill_levels[k] === 'number') dst[k] = totItem.nbt.skill_levels[k];
      totItem.nbt.skill_levels = dst;
    }
  } catch(_) {}

  try {
    if (totItem.nbt.activeSkill && norm(totItem.nbt.activeSkill) === removedKey) delete totItem.nbt.activeSkill;
    if (totItem.nbt.selectedSkill && norm(totItem.nbt.selectedSkill) === removedKey) delete totItem.nbt.selectedSkill;
  } catch(_) {}

  rebuildTotemLore(totItem, false);
  return true;
}

/**
 * Removes all skills except the locked one.
 */
// Keep ONLY the locked skill; remove all other skills, levels, and leftover lore/markers
function removeAllTotemSkills(player, totItem) {
  if (!totItem || !totItem.nbt) return false;
  let nbt = totItem.nbt;
  // Collect all locked skills (multi-support)
  let lockedSet = new Set();
  let legacyLocked = nbt.lockedSkill ? String(nbt.lockedSkill) : null;
  let skillCount = countTotemSkills(totItem)
  function collectLocked(raw) {
    if (!raw) return;
    try {
      let len = raw.length;
      for (let i = 0; i < len; i++) {
        let v = String(raw[i])
          .replace(/^["'\[]+/, '')
          .replace(/["'\]]+$/, '')
          .trim();
        if (v) lockedSet.add(v);
      }
    } catch(_){}
  }

  if (Array.isArray(nbt.lockedSkills)) collectLocked(nbt.lockedSkills);
  else if (nbt.lockedSkills && typeof nbt.lockedSkills.length === 'number') collectLocked(nbt.lockedSkills);
  if (legacyLocked) {
    legacyLocked = legacyLocked.replace(/^["'\[]+/, '').replace(/["'\]]+$/, '').trim();
    if (legacyLocked) lockedSet.add(legacyLocked);
  }

  if (lockedSet.size === 0) {
    nbt.skills = [];
    if (nbt.contains && nbt.contains('skill_levels')) {
      if (typeof nbt.remove === 'function') {
        try { nbt.remove('skill_levels'); } catch(_) { delete nbt.skill_levels; }
      } else delete nbt.skill_levels;
    } else if (nbt.skill_levels) delete nbt.skill_levels;
    if (nbt.activeSkill) delete nbt.activeSkill;
    if (nbt.selectedSkill) delete nbt.selectedSkill;
  } else {
    nbt.skills = Array.from(lockedSet);
    // Filter skill_levels to only locked
    try {
      if (nbt.contains && nbt.contains('skill_levels')) {
        let oldTag = nbt.get('skill_levels');
        let newLevels = {};
        for (let k of lockedSet) {
          if (oldTag.contains && oldTag.contains(k)) newLevels[k] = oldTag.getInt(k);
          else if (typeof oldTag[k] === 'number') newLevels[k] = oldTag[k];
        }
        if (typeof nbt.put === 'function') {
          nbt.put('skill_levels', {});
          let newTag = nbt.get('skill_levels');
          for (let k in newLevels) {
            if (newLevels.hasOwnProperty(k) && typeof newTag.putInt === 'function') newTag.putInt(k, newLevels[k]);
            else newTag[k] = newLevels[k];
          }
        } else nbt.skill_levels = newLevels;
      } else if (nbt.skill_levels) {
        let filtered = {};
        for (let k of lockedSet) if (typeof nbt.skill_levels[k] === 'number') filtered[k] = nbt.skill_levels[k];
        nbt.skill_levels = filtered;
      }
    } catch(_) {
      if (nbt.skill_levels) {
        let filtered = {};
        for (let k of lockedSet) if (typeof nbt.skill_levels[k] === 'number') filtered[k] = nbt.skill_levels[k];
        nbt.skill_levels = filtered;
      }
    }
    if (nbt.activeSkill && !lockedSet.has(String(nbt.activeSkill))) delete nbt.activeSkill;
    if (nbt.selectedSkill && !lockedSet.has(String(nbt.selectedSkill))) delete nbt.selectedSkill;
  }

  if (nbt.display && nbt.display.Lore) delete nbt.display.Lore;
  if (nbt.display && Object.keys(nbt.display).length === 0) delete nbt.display;
  if (typeof rebuildTotemLore === 'function') rebuildTotemLore(totItem, false);

  if (player) {
    //Utils.server.tell(lockedSet.size)

    let locksize = lockedSet.size || 0
    //Utils.server.tell(locksize)
    //Utils.server.tell(skillCount)

    if (skillCount == 0) {
      player.tell(Text.red('No skills can be removed'))
      return false
    }

    if (skillCount > 0 && locksize == 0) {
      player.tell(Text.darkRed('Removed all skills'))
      return true
    }

    if (Number(locksize.toFixed(1)) == Number(skillCount.toFixed(1))) {
      player.tell(Text.gold('All skills are locked'))
      return false
    } else {
      let listDisp = Array.from(lockedSet).map(k => k.split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' '));
      player.tell(Text.gray('Removed all non-locked skills. Kept: ').append(Text.gold(listDisp.join(', '))));
    }
  }
  return true;
}





/**
 * Rebuilds the lore based on current skills and locked skill.
 */
/**
 * Rebuilds the totem’s lore, coloring the last skill red if ignoreCaps is true.
 *
 * @param {Internal.ItemStackJS} totItem
 * @param {boolean} [ignoreCaps=false] – if true, the final skill is colored red instead of green
 */
function rebuildTotemLore(totItem, ignoreCaps) {
  // Normalize skills & locked skills
    let rawSkills = totItem.nbt.skills || [];
    let skills = [];
    for (let s of rawSkills) {
      try {
        let k = String(s)
          .replace(/^["'\[]+/, '')
          .replace(/["'\]]+$/, '')
          .trim();
        if (k && skillMapping[k]) skills.push(k); // keep only valid keys
      } catch(_){}
    }
    let lockedSkills = [];
    // gather lockedSkills (array or java list)
    let lsRaw = totItem.nbt.lockedSkills;
    if (lsRaw && typeof lsRaw.length === 'number') {
      for (let i=0;i<lsRaw.length;i++) {
        try {
          let v = String(lsRaw[i])
            .replace(/^["'\[]+/, '')
            .replace(/["'\]]+$/, '')
            .trim();
          if (v) lockedSkills.push(v);
        } catch(_){}
      }
    }
    let legacyLocked = totItem.nbt.lockedSkill ? String(totItem.nbt.lockedSkill) : '';
    if (legacyLocked) {
      legacyLocked = legacyLocked.replace(/^["'\[]+/, '').replace(/["'\]]+$/, '').trim();
      if (legacyLocked) lockedSkills.push(legacyLocked);
    }
    // Deduplicate lockedSkills
    lockedSkills = Array.from(new Set(lockedSkills));

  // Helper: format numbers (no trailing .0, keep 1 decimal if needed)
  let fmt = (n) => {
    if (n == null || isNaN(n)) return n;
    let r = Math.round(n * 10) / 10;
    return (Math.abs(r - Math.round(r)) < 1e-9) ? String(Math.round(r)) : String(r);
  };

  // Read levels tag (CompoundTag) if present
  let levelsTag = null;
  if (totItem.nbt && totItem.nbt.contains('skill_levels')) {
    levelsTag = totItem.nbt.get('skill_levels'); // CompoundTag
  }

  // Build dynamic lore for a single skill
  let buildLore = (key, level) => {
    let def = skillMapping[key] || {};
    let text = (def.lore || '').toString();

    let has = (p) => Object.prototype.hasOwnProperty.call(def, p);

    let timeVal         = has('time')         ? (def.time * level) : null;
    let percentVal      = has('percent')      ? (def.percent * level) : null;
    let hitsVal         = has('hits')         ? (def.hits * level) : null;
    let entityCountVal  = has('entity_count') ? (def.entity_count * level) : null;
    let amplifierVal    = has('amplifier')    ? (def.amplifier * level) : null;
    let cooldownVal     = has('cooldown')     ? ((def.cooldown * 3) - (def.cooldown * (level - 1))) : null;

    if (timeVal != null)        
      text = text.replace(/TIME/g, `§a${fmt(timeVal)}s§7`);
    if (percentVal != null)     
      text = text.replace(/PERCENT/g, `§a${fmt(percentVal)}%§7`);
    if (hitsVal != null)        
      text = text.replace(/HITS/g, `§a${fmt(hitsVal)} hits§7`);
    if (entityCountVal != null) 
      text = text.replace(/ENTITY_COUNT/g, `§a${fmt(entityCountVal)}§7`);
    if (amplifierVal != null)   
      text = text.replace(/AMPLIFIER/g, `§a${fmt(amplifierVal)}§7`);
    if (cooldownVal != null)    
      text = text.replace(/COOLDOWN/g, `§a${fmt(cooldownVal)}s§7`);


    return text;
  };

  let newLore = [];

  // Iterate skills
  skills.forEach((skillKey) => {
    let words = skillKey.split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1));
    let displayName = words.join(' ');
    // Base color logic; override to gold if locked
    let nameColor = 'blue';
    if (skills.indexOf(skillKey) > 4 && ignoreCaps) nameColor = 'red';
    if (lockedSkills.includes(skillKey)) nameColor = 'gold';

    // Determine level (default 1) via NBT API
    let level = 1;
    if (levelsTag && levelsTag.contains(skillKey)) {
      level = Math.max(1, Math.min(3, levelsTag.getInt(skillKey)));
    }

    let romanLevel = toRoman(level);
    // Title line with level
    newLore.push(JSON.stringify({
      text: '§o' + displayName + ' '+ romanLevel + '§7',
      color: nameColor,
      italic: false
    }));

    // Description line (dynamically rendered)
    let rendered = buildLore(skillKey, level);
    newLore.push(JSON.stringify({
      text: '  §e•§7 ' + rendered,
      color: 'gray',
      italic: false
    }));

    // Spacing
    newLore.push(JSON.stringify({
      text: '',
      color: 'gray',
      italic: false
    }));
  });

  // Removed separate locked skill section; locked skills are now indicated by gold title color.

  // Write back
  totItem.nbt.display = totItem.nbt.display || {};
  totItem.nbt.display.Lore = newLore;
}

// Simple converter for 1–10 (expand if you need higher)
function toRoman(num) {
  const romans = [
    "", "I", "II", "III", "IV",
    "V", "VI", "VII", "VIII", "IX", "X"
  ];
  return romans[num] || num; // fallback to normal number if out of range
}


/**
 * Removes a random non-locked skill from the totem.
 *
 * @param totItem The totem ItemStack
 * @returns true if a skill was removed; false if no valid skills to remove
 */

/**
 * function removeRandomTotemSkill(totItem) {
  totItem.nbt = totItem.nbt || {};
  let skills = (totItem.nbt.skills || []).map(s => String(s).replace(/\"/g, '').replace(/^\[|\]$/g, ''));
  let lockedSkill = String(totItem.nbt.lockedSkill || '').replace(/\"/g, '').replace(/^\[|\]$/g, '');

  // Filter out the locked skill
  let removableSkills = skills.filter(skill => skill !== lockedSkill);
  if (removableSkills.length === 0) return false;

  // Pick a random skill to remove
  let idx = Math.floor(Math.random() * removableSkills.length);
  let removedSkill = removableSkills[idx];

  // Remove from skill list
  totItem.nbt.skills = skills.filter(s => s !== removedSkill);

  // Rebuild the lore
  rebuildTotemLore(totItem);
  return true;
} 
 * @param {
 * 
 * } totItem 
 * @returns 
 */




function swapTotemSkill(player, totItem) {

    return removeRandomTotemSkillSafe(player, totItem) && infuseSkillTotem(player, totItem, 'random', false)

}


function validSkillTotemOrb(item) {
  if (!item || !item.id) return false;
  return global.validSkillTotemOrbs.includes(item.id);
}

function validSkillTotem(item) {
  if (!item || !item.id) return false;
  return item.id.includes('skill_totem') && item.id.includes('kubejs:');
}


function skillTotemRarity(item) {
  if (!item || !item.id) return false;
  if (!item.id.includes('skill_totem')) return false;
  if (item.id.includes('lesser_skill_totem')) return 'lesser';
  if (item.id.includes('greater_skill_totem')) return 'greater';
  if (item.id.includes('exalted_skill_totem')) return 'exalted';
}








function upgradeTotemRarity(player, item, orbId, slotIndex) {
  if (!item || item.empty) return false
  Utils.server.runCommandSilent(`/ftbquests change_progress ${player.username} reset 2083FCDC7D2A5D87`)
  Utils.server.runCommandSilent(`/ftbquests change_progress ${player.username} complete 2083FCDC7D2A5D87`)
  player.persistentData.putBoolean('orb_of_knowledge', true)
  let totemChain = [
    'kubejs:empty_skill_totem',
    'kubejs:lesser_skill_totem',
    'kubejs:greater_skill_totem',
    'kubejs:exalted_skill_totem'
  ]

  let id = item.id
  let tier = totemChain.indexOf(id)
  if (tier === -1) {
    player.tell(Text.red("This is not a valid skill totem."))
    return false
  }

  let targetTier = -1
  if (orbId === 'kubejs:orb_of_knowledge' && tier === 0) {
    targetTier = 1
  } else if (orbId === 'kubejs:orb_of_revelation' && (tier === 0 || tier === 1)) {
    targetTier = 2
  } else if (orbId === 'kubejs:orb_of_divinity' && tier < 3) {
    targetTier = 3
  }

  if (targetTier === -1) {
    player.tell(Text.red("This totem cannot be upgraded with that orb."))
    return false
  }

  let nextId = totemChain[targetTier]
  let upgraded = Item.of(nextId)
  if (item.nbt) upgraded = upgraded.withNBT(item.nbt)
  

  item.count -= 1
  Utils.server.scheduleInTicks(1, () => {
    player.inventory.setStackInSlot(slotIndex, upgraded)
    let name = nextId.split(':').pop().replace(/_/g, ' ')
    player.tell(Text.green(`Totem upgraded to ${name}`))
  })

  return true
}




/**
 * Corrupts a Skill Totem with weighted outcomes:
 *  - 40% add another random, non-duplicate skill (still marks CORRUPTED)
 *  - 40% do nothing but still marks CORRUPTED
 *  - 20% destroy the totem
 *
 * If Arcane Luck is active:
 *  - 60% add a skill
 *  - 30% do nothing
 *  - 10% destroy
 *
 * @param {Internal.PlayerJS}    player
 * @param {Internal.ItemStackJS} totItem
 * @returns {boolean} true if the orb should be consumed
 */
function corruptTotem(player, totItem) {
  if (!totItem || totItem.empty) return false;

  totItem.nbt = totItem.nbt || {}

  // Gather the existing skills (bare keys)
  let skills = [];
  if (Array.isArray(totItem.nbt?.skills)) {
    for (let entry of totItem.nbt.skills) {
      let key = String(entry);
      if (key.includes(':')) key = key.split(':')[1];
      skills.push(key);
    }
  }

  function hasArcaneLuck(p) {
    if (!p) return false
    try { if (typeof p.hasEffect === 'function' && p.hasEffect('kubejs:arcane_luck')) return true } catch (e) {}
    try { if (p.potionEffects && typeof p.potionEffects.isActive === 'function' && p.potionEffects.isActive('kubejs:arcane_luck')) return true } catch (e2) {}
    return false
  }

  function markTotemCorrupted() {
    let lore = []
    if (totItem.nbt.display?.Lore) {
      let raw = totItem.nbt.display.Lore
      for (let i = 0; i < raw.length; i++) lore.push(raw[i])
    }
    if (!lore.some(line => String(line).includes('CORRUPTED'))) {
      lore.push(JSON.stringify({ text: 'CORRUPTED', color: 'dark_red', italic: true }))
    }
    totItem.nbt.display = totItem.nbt.display || {}
    totItem.nbt.display.Lore = lore
    try { if (typeof totItem.nbt.putBoolean === 'function') totItem.nbt.putBoolean('itemCorrupted', true) } catch (e) {}
    try { totItem.nbt.itemCorrupted = true } catch (e2) {}
  }

  function playCorruptionFx() {
    Utils.server.runCommandSilent(
      `/execute in ${player.level.dimension} run playsound witherreincarnated:entity.wither.idle_weak ` +
      `ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.5`
    )
    Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run particle fathomless:crimson_aura ${player.x} ${player.y+1} ${player.z} 0.3 0.5 0.3 1 100 force ${player.username}`)
  }

  let luck = hasArcaneLuck(player)
  let roll = Math.random()

  let addChance = luck ? 0.60 : 0.40
  let nothingChance = luck ? 0.30 : 0.40
  let destroyChance = luck ? 0.10 : 0.20

  // Normalize just in case someone tweaks values incorrectly
  let total = addChance + nothingChance + destroyChance
  if (total <= 0) { addChance = 0.40; nothingChance = 0.40; destroyChance = 0.20; total = 1.0 }
  addChance /= total
  nothingChance /= total

  if (roll < addChance) {
    // ─── Add a random non-duplicate skill (if possible) ─────────
    let allKeys = Object.keys(skillMapping)
    let available = allKeys.filter(k => !skills.includes(k))
    let newSkill = available.length ? available[Math.floor(Math.random() * available.length)] : null
    if (newSkill) infuseSkillTotem(player, totItem, newSkill, true)

    markTotemCorrupted()
    playCorruptionFx()

    if (newSkill) {
      let disp = newSkill.split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' ')
      player.tell(Text.darkRed('The totem has been corrupted! Added skill: ').append(Text.red(disp)))
    } else {
      player.tell(Text.darkRed('The totem has been corrupted!'))
    }
    return true
  }

  if (roll < addChance + nothingChance) {
    // ─── Do nothing, but still mark corrupted ───────────────────
    markTotemCorrupted()
    playCorruptionFx()
    player.tell(Text.darkRed('The corruption failed to alter the totem!'))
    return true
  }

  // ─── Destroy ──────────────────────────────────────────────────
  totItem.count = 0
  player.tell(Text.red('The totem has been destroyed by the corruption!'))
  Utils.server.runCommandSilent(`/execute in ${player.level.dimension} run particle fathomless:crimson_aura ${player.x} ${player.y+1} ${player.z} 0.3 0.5 0.3 1 100 force ${player.username}`)
  let dim = player.level.dimension
  Utils.server.runCommandSilent(`/execute in ${dim} run playsound minecraft:entity.wither.spawn ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.5`)
  return true
}

const skillMapping = {
  'ice_beam': {
    lore: 'Enemies hit by a Ray of Frost are frozen for TIME seconds. Cooldown: COOLDOWN',
    time: 3, 
    cooldown: 10,
    categories: ['cold']
  },
  'frost_repulse': {
    lore: 'PERCENT chance to freeze enemies around you when you are hit',
    percent: 5,
    categories: ['cold']
  },
  'aegis': {
    lore: 'Block and look at the ground while falling to negate fall damage',
    categories: ['block']
  },
  'bulwark': {
    lore: 'Blocked physical projectiles will be reflected back at the enemy',
    categories: ['block']
  },
  'lava_dancer': {
    lore: 'Gain a permanent lava walker effect',
    categories: ['skill']
  },
  'speed_runner': {
    lore: 'Gives you TIME of Speed after performing a ParCool move',
    time: 1,
    categories: ['speed']
  },
  'merchant': {
    lore: 'Items sell for PERCENT more',
    percent: 25,
    categories: ['economy']
  },
  'bounty_hunter': {
    lore: 'Monster kills give PERCENT more coins',
    percent: 20,
    categories: ['economy']
  },
  'woods_walker': {
    lore: 'Quantity of item drops from animals is doubled',
    categories: ['economy']
  },
  'tunnel_vision': {
    lore: 'Marks a random nearby enemy upon each kill. Killing a marked enemy gives stacking speed and haste. Resets after 5 seconds of no marked kills',
    categories: ['speed', 'haste']
  },
  'head_hunter': {
    lore: 'Enemies under 10% health have a 50% chance to be Culled. Player gains PERCENT max health as healing',
    percent: 3,
    categories: ['skill', 'regeneration']
  },
  'momentum': {
    lore: 'Gain TIME of Haste and Strength after Combat Dashing',
    time: 2,
    categories: ['haste', 'strength']
  },
  'shattering_blow': {
    lore: 'Combat dashing into an enemy will stun them for TIME',
    time: 1.5,
    categories: ['stun', 'roll']
  },
  'cloaked_blade': {
    lore: 'Hitting a mob while invisible has a PERCENT chance to apply Venom for TIME',
    percent: 30,
    time: 3,
    categories: ['venom', 'invisibility']
  },
  'hexblood': {
    lore: 'Killing a mob with a spell while they have Venom applies Hex to the next HITS',
    hits: 3,
    categories: ['hex', 'venom']
  },
  'witchglass': {
    lore: 'Killing a mob with Hex via a spell applies Venom to ENTITY_COUNT nearby enemies for TIME',
    entity_count: 1,
    time: 3,
    categories: ['hex', 'venom']
  },
  'poisonous_rot': {
    lore: 'Killing a mob with Venom has a PERCENT chance per level of Venom to give nearby enemies Venom and Plague for 5s per level',
    percent: 10,
    categories: ['venom', 'plague']
  },
  'bloodshroud': {
    lore: 'Gain Invisibility and Speed while Bleeding. Hitting a mob consumes the Bleeding, Invisibility, and Speed.',
    categories: ['bleed', 'invisibility', 'speed']
  },
  'plague_of_rust': {
    lore: 'Hitting a mob with Plague has a PERCENT chance to apply Rend for TIME',
    percent: 25,
    time: 3,
    categories: ['plague', 'rend']
  },
  'final_gasp': {
    lore: 'Killing a mob has a PERCENT chance to give YOU Venom, Bleeding, Chilled, or Slowness for 15 seconds',
    percent: 5,
    categories: ['venom', 'bleed', 'cold', 'slow']
  },
  'frost_wraith': {
    lore: 'Hitting a mob while Invisible has a PERCENT chance to apply Chilled for TIME',
    percent: 30,
    time: 3,
    categories: ['cold', 'invisibility']
  },
  'bloody_shards': {
    lore: 'Killing a Frozen mob has a PERCENT chance per level of Frozen to apply Bleeding to nearby enemies',
    percent: 10,
    categories: ['cold', 'bleed']
  },
  'dreadmark': {
    lore: 'Permafrost Charges work with any projectile',
    categories: ['skill']
  },
  'death_grip': {
    lore: 'Projectile attacks consume 10 mana to stack 1 level of Chilled on the enemy. Each stack of Chilled has a PERCENT chance to consume Chilled and Freeze the enemy for 2s per stack',
    percent: 5,
    categories: ['cold', 'mana', 'ranged']
  },
  'frost_surge': {
    lore: 'Kills on mobs with the Chilled effect have a PERCENT chance to give you Speed for TIME',
    percent: 20,
    time: 3,
    categories: ['cold', 'speed']
  },
  'lucky_hex': {
    lore: 'While you have Luck, attacks have a PERCENT chance to give the target Hexed for TIME',
    percent: 15,
    time: 3,
    categories: ['hex']
  },
  'sanguine': {
    lore: 'Hitting a bleeding mob has a PERCENT chance to apply Marked for Death for TIME',
    percent: 15,
    time: 3,
    categories: ['bleed']
  },
  'bloodhunt': {
    lore: 'Melee killing a Slowed mob applies Bleeding to your next HITS',
    hits: 3,
    categories: ['slow', 'bleed']
  },
  'grim_reaper': {
    lore: 'Hitting a Hexed mob has a PERCENT chance per level of Hex to apply Marked for Death to nearby mobs (4s per level)',
    percent: 7,
    categories: ['hex']
  },
  'blood_splatter': {
    lore: 'Using §4Blood Harvest§7 has a PERCENT chance to apply Bleeding to each enemy in a 15 block radius',
    percent: 10,
    categories: ['bleed']
  },
  'sacrifice': {
    lore: 'Killing while Charged has a PERCENT chance per enemy to apply Hexed within 20 blocks',
    percent: 20,
    categories: ['hex', 'charged']
  },
  'esoteric_poison': {
    lore: 'Killing while Charged has a PERCENT chance per enemy to apply Plague within 20 blocks',
    percent: 15,
    categories: ['hex', 'charged']
  },
  'synapse': {
    lore: 'Killing a Stunned mob with a ranged attack stuns ENTITY_COUNT nearby enemies for TIME',
    entity_count: 1,
    time: 3,
    categories: ['stun', 'ranged']
  },
  'aftershock': {
    lore: 'Killing a Stunned mob with a melee attack applies Rend to ENTITY_COUNT nearby enemies for TIME',
    entity_count: 1,
    time: 3,
    categories: ['stun', 'melee', 'rend']
  },
  'temporal_chains': {
    lore: 'Killing a Stunned mob with a spell has a PERCENT chance per enemy to apply Slowness to nearby enemies',
    percent: 15,
    categories: ['stun', 'spell', 'slow']
  },
  'ghost_draw': {
    lore: 'The §eLethal Shadows§7 skill works with bows instead of only crossbows',
    categories: ['skill']
  },
  'stoneform': {
    lore: 'Blocking has a PERCENT chance to give 5s of Resistance',
    percent: 15,
    categories: ['block', 'resistance']
  },
  'spectral_splinter': {
    lore: 'PERCENT chance to spawn 2 spectral arrows that target nearby enemies when hitting with an arrow',
    percent: 15,
    categories: ['ranged']
  },
  'slipstream': {
    lore: 'Hitting enemies with projectiles gives a TIME stack of Speed. Getting hit removes it',
    time: 4,
    categories: ['ranged', 'speed']
  },
  'phantom_step': {
    lore: 'Gain TIME of Invisibility and Speed for TIME after using Combat Dash',
    time: 3,
    categories: ['invisibility', 'speed', 'roll']
  },
  'dripping_maw': {
    lore: 'Killing a Hexed mob has a PERCENT chance to make nearby mobs cast Fire Breath at each other, based on Hex level',
    percent: 15,
    categories: ['hex', 'spell']
  },
  'ancestral_strength': {
    lore: 'Melee kills have a PERCENT chance to stack TIME of Strength',
    percent: 20,
    time: 2,
    categories: ['melee', 'strength']
  },
  'giants_blood': {
    lore: 'Melee hit enemies to gain a stacking Lesser Strength effect that lasts TIME. Max 5 stacks',
    time: 3,
    categories: ['melee', 'strength']
  },
  'mana_veil': {
    lore: 'Upon taking fatal damage, consume all mana and regain 1% of your health for every 40 mana consumed. You must have at least 200 mana to use this',
    categories: ['mana', 'revive', 'regeneration']
  },
  'wounding_strike': {
    lore: 'Hitting a mob with the Rend effect has a PERCENT chance to give them Bleeding for TIME',
    percent: 15,
    time: 3,
    categories: ['rend', 'bleed']
  },
  'gravemark': {
    lore: 'Killing a Slowed enemy with a projectile has a PERCENT chance to give you TIME of Impact',
    percent: 20,
    time: 5,
    categories: ['slow', 'ranged']
  },
  'shock_transfer': {
    lore: 'Killing a mob with a spell while you have the Charged effect has a PERCENT chance to consume your Charged effect and make mobs cast Ball Lightning at each other',
    percent: 15,
    categories: ['charged', 'spell']
  },
  'siphoning_steel': {
    lore: 'PERCENT chance to gain mana equal to 1% of damage from Melee attacks',
    percent: 25,
    categories: ['melee', 'mana']
  },
  'arcane_bargain': {
    lore: 'Killing a Hexed mob recharges 5% of your health and 5% of your mana per level of Hexed consumed. Each level consumed adds a PERCENT chance to apply a negative effect to you',
    percent: 15,
    categories: ['hex', 'mana', 'regeneration']
  },
  'arcane_dash': {
    lore: 'Gain a TIME stack of Speed after casting a spell. Max stacks: 3',
    time: 2,
    categories: ['spell', 'speed']
  },
  'sprint_shot': {
    lore: 'Hitting a mob with a projectile while sprinting gives you a TIME stacking Speed effect',
    time: 1,
    categories: ['spell', 'speed']
  },
  'scouring_winds': {
    lore: 'Using a dodge roll near a Hexed mob consumes Hexed and applies a 5 second speed effect with an amplifier equal to the level of Hex consumed',
    categories: ['hex', 'roll', 'speed']
  },
  'dead_rush': {
    lore: 'Killing an enemy that has Hexed, Chilled, Bleeding, or Plague will transfer the effect to you but apply a Speed boost for TIME',
    time: 5,
    categories: ['hex', 'cold', 'bleed', 'plague', 'speed']
  },
  'second_wind': {
    lore: 'Upon taking damage below 20% health, gain a Speed AMPLIFIER effect for TIME seconds. 30 second cooldown',
    time: 4,
    amplifier: 1,
    categories: ['speed']
  },
  'last_stand': {
    lore: 'Upon taking damage below 10% health, gain a Strength IV effect for TIME. 1 minute cooldown',
    time: 5,
    categories: ['strength']
  },
  'sentry': {
    lore: 'The §eSentinel§7 skill now works with bows',
    categories: ['skill']
  },
  'stim_shot': {
    lore: 'Using any Health Elixir will give you TIME of Speed II',
    time: 5,
    categories: ['elixir', 'speed']
  },
  'unbreakable_will': {
    lore: 'Using any Mana Elixir will give you TIME of Resistance I',
    time: 5,
    categories: ['elixir', 'resistance']
  },
  'commanders_concoction': {
    lore: 'Using any Health Elixir will give you and allies within a 15 block radius TIME of Strength 2',
    time: 10,
    categories: ['elixir', 'strength', 'allies']
  },
  'volatile_mix': {
    lore: 'PERCENT chance to double the duration and amplifier of any Elixir effect you apply, or cancel it entirely',
    percent: 20,
    categories: ['elixir']
  },
  'windlash_elixir': {
    lore: 'Using any Mana Elixir vertically launches any non-boss enemy within a 5 block radius',
    categories: ['elixir']
  },
  'bottomless_quiver': {
    lore: 'Modular Arrows teleport back to you if you miss a shot',
    categories: ['skill']
  },
  // Added in 1.3
  'dormant_rage': {
    lore: 'Killing a mob while you are Slowed will remove the effect and give you TIME of Strength',
    time: 4,
    categories: ['strength']
  },
  'satiate': {
    lore: 'Killing a mob while you have the Hunger effect will remove the it and give you TIME of Saturation',
    time: 5,
    categories: ['saturation']
  },
  'overdrive': {
    lore: 'If you are mounted, you and the mount gain a Speed effect',
    categories: ['speed', 'mount']
  },
  'trample': {
    lore: 'Hitting a mob with a melee attack while mounted will give them TIME of Slowness',
    time: 4,
    categories: ['slow', 'mount', 'melee']
  },
  'mangudai': {
    lore: 'Hitting a mob with a projectile while mounted will give you TIME of Haste',
    time: 3,
    categories: ['haste', 'mount', 'ranged']
  },
  'immovable_object': {
    lore: 'Gain a Resistance and Slowed effect while blocking',
    categories: ['block', 'resistance', 'slow']
  },
  'open_wounds': {
    lore: 'Each nearby bleeding mob adds a stack of Lesser Strength to you, up to 5 stacks',
    categories: ['bleed', 'strength']
  },
  'inertial_reversal': {
    lore: 'Combat dashing into an enemy that has a Slowness effect will consume the effect and give you TIME of Speed II',
    time: 5,
    categories: ['slow', 'roll', 'speed']
  },
  'frost_blast': {
    lore: 'Combat dashing into an enemy that has a Chilled effect will consume the effect and give them TIME of Frozen',
    time: 5,
    categories: ['roll', 'cold']
  },
  'arrow_storm': {
    lore: 'Hitting an enemy with a physical projectile has a PERCENT chance to spawn 4 homing arrows that target that enemy',
    percent: 15,
    categories: ['ranged', 'skill']
  },
  'conductive_arcane': {
    lore: 'Hitting an enemy with a spell has a PERCENT chance to spawn 2 homing orbs that target that enemy',
    percent: 20,
    categories: ['spell', 'skill']
  },
  'pyroclasm': {
    lore: 'Blocking a melee attack has a PERCENT chance for you to rapidly cast Flaming Strike at the attacker',
    percent: 15,
    categories: ['block', 'spell']
  },
  'voltage': {
    lore: 'Blocking a ranged attack has a PERCENT chance for you to rapidly cast Lightning Bolt at the attacker',
    percent: 15,
    categories: ['block', 'spell']
  },
  'freezing_retreat': {
    lore: 'Blocking a ranged attack has a 100% chance for you to rapidly cast Frost Step',
    categories: ['block', 'spell']
  },
  'arcane_barrage': {
    lore: 'Blocking a melee attack has a PERCENT chance for you to rapidly cast 2 Magic Missles at the attacker',
    percent: 15,
    categories: ['block', 'spell']
  },
  // Added in 1.4
  'gravewalker': {
    lore: 'When you are revived with an Effigy of the Undying, there is a PERCENT chance to not consume it',
    percent: 20,
    categories: ['revive']
  },
  'revenant': {
    lore: 'When you are revived with an Effigy of the Undying, you gain TIME of Invisibility and Speed',
    time: 5,
    categories: ['revive']
  },
  'divinity': {
    lore: 'When you are revived with an Effigy of the Undying, you and nearby allies are fully healed',
    categories: ['revive', 'allies']
  },
  // Added in 1.4.2
  'mana_bloom': {
    lore: 'Using a spell to kill a mob that has the Hexed effect has a PERCENT chance to give you 25 mana',
    percent: 30,
    categories: ['mana', 'hex', 'spell']
  },
  'coward': {
    lore: 'When you are hit while blocking, you gain TIME of Speed and Weakness',
    time: 10,
    categories: ['block', 'speed']
  },
  'phantom_vengeance': {
    lore: 'Killing a mob while Invisible has a PERCENT chance to give them Bleeding for TIME',
    percent: 30,
    time: 3,
    categories: ['bleed', 'invisibility']
  },
  'pathogenic_hex': {
    lore: 'Attacks against a mob with Hexed have a PERCENT chance to apply Venom for TIME',
    percent: 20,
    time: 3,
    categories: ['venom', 'hex']
  },
  'biting_cold': {
    lore: 'Attacks against a mob with Chilled have a PERCENT chance to apply Slowness for TIME',
    percent: 20,
    time: 4,
    categories: ['cold', 'slow']
  },
  'reapers_call': {
    lore: 'Melee killing a Chilled mob applies Marked for Death to your next HITS',
    hits: 1,
    categories: ['cold', 'melee']
  },
  'spell_break': {
    lore: 'Killing a Hexed mob with a spell applies Rend to your next HITS',
    hits: 3,
    categories: ['hex', 'spell', 'rend']
  },

  'hexchain': {
    lore: 'Killing a mob with Hex via a spell applies Hexed to ENTITY_COUNT nearby enemies for TIME',
    entity_count: 2,
    time: 3,
    categories: ['hex', 'spell']
  },
  'soul_tithe': {
    lore: 'Killing a Hexed mob with a melee attack has a PERCENT chance to give you 15 mana',
    percent: 30,
    categories: ['mana', 'hex', 'melee']
  },

  'spellweaver': {
    lore: 'Casting spells grants a stacking Echoing Strikes buff that lasts TIME. Getting hit removes it. Max 3 stacks',
    time: 4,
    categories: ['spell']
  },

  'arcane_vent': {
    lore: 'Killing while Charged has a PERCENT chance per enemy to apply Slowness within 12 blocks for TIME',
    percent: 25,
    time: 5,
    categories: ['charged', 'slow']
  },
  'frost_chain': {
    lore: 'Killing a Chilled mob has a PERCENT chance to make nearby enemies cast Ray of Frost at each other',
    percent: 20,
    categories: ['cold', 'spell']
  },

  'power_vacuum': {
    lore: 'Killing a mob while you have Weakness grants stacking Strength for TIME',
    time: 4,
    categories: ['weakness', 'strength']
  },

  'runeguard': {
    lore: 'Casting a spell grants Resistance for TIME (can stack up to 2 times)',
    time: 3,
    categories: ['spell', 'resistance']
  },

  'iron_will': {
    lore: 'When below 30% health, gain Resistance for TIME (Cooldown applies)',
    time: 8,
    categories: ['resistance']
  },

  'heavy_strike': {
    lore: 'While sprinting, landing a melee attack will trigger Flaming Strike',
    categories: ['melee', 'sprint', 'spell']
  },
  'windrunner': {
    lore: 'While sprinting, melee attacks have a PERCENT chance to cast Gust',
    percent: 25,
    categories: ['melee', 'sprint', 'spell']
  },
  'spell_snipe': {
    lore: 'While sprinting, ranged attacks have a PERCENT chance to cast Magic Arrow',
    percent: 10,
    categories: ['ranged', 'sprint', 'spell']
  },
  'gaias_boon': {
    lore: 'While sprinting, melee attacks have a PERCENT chance to cast Earthquake',
    percent: 5,
    categories: ['melee', 'sprint', 'spell']
  },
  'frost_snare': {
    lore: 'While sprinting, melee attacks have a PERCENT chance to cast Frostwave',
    percent: 30,
    categories: ['melee', 'sprint', 'spell']
  },
  'arcane_rebound': {
    lore: 'Killing a Slowed or Chilled enemy grants you Charged for TIME',
    time: 4,
    categories: ['charged', 'cold', 'slow']
  },

  'battle_march': {
    lore: 'Using any Mana Elixir will give you and allies within a 12 block radius TIME of Speed II',
    time: 8,
    categories: ['elixir', 'speed', 'allies']
  },

  'iron_rider': {
    lore: 'While mounted, you and the mount gain TIME of Resistance',
    time: 4,
    categories: ['resistance', 'mount']
  },
  'lancer': {
    lore: 'Hitting a mob with a melee attack while mounted will give them TIME of Weakness',
    time: 4,
    categories: ['weakness', 'mount', 'melee']
  },
  'venom_whip': {
    lore: 'Combat dashing into an enemy that has Venom applies Bleeding for TIME',
    time: 4,
    categories: ['bleed', 'roll', 'venom']
  },
  'thunder_riposte': {
    lore: 'Blocking a melee attack has a PERCENT chance for you to rapidly cast Chain Lightning',
    percent: 20,
    categories: ['block', 'spell']
  },
  'searing_wound': {
    lore: 'Hitting with Flaming Strike applies Bleeding for TIME. Cooldown: COOLDOWN',
    time: 3,
    cooldown: 8,
    categories: ['bleed', 'spell']
  },
  'stormwalker': {
    lore: 'Hitting enemies with spells while you are sprinting gives you a PERCENT chance to rapidly cast Chain Lightning.',
    percent: 20,
    categories: ['sprint', 'spell']
  },
  'frostbarb': {
    lore: 'Hitting enemies with ranged attacks while crouching gives you a PERCENT chance to rapidly cast Icicle. Cooldown: COOLDOWN',
    percent: 25,
    cooldown: 5,
    categories: ['crouch', 'ranged', 'spell']
  },
  'raging_inferno': {
    lore: 'Hitting enemies with spells while you are crouching gives you a PERCENT chance to rapidly cast Scorch. Cooldown: COOLDOWN',
    percent: 30,
    cooldown: 30,
    categories: ['crouch', 'spell']
  },
  'frozen_lightning': {
    lore: 'If you have a Permafrost Charge, you also gain a Shock Charge.',
    categories: ['skill']
  },
  // Added in 1.5
  'frostnova': {
    lore: 'Landing a critical hit with a melee attack has a PERCENT chance to rapidly cast Frostwave.',
    percent: 20,
    categories: ['cold', 'melee', 'spell']
  },
  'flame_burst': {
    lore: 'Landing a critical hit with a melee attack has a PERCENT chance to rapidly cast Flaming Strike.',
    percent: 20,
    categories: ['fire', 'melee', 'spell']
  },
  
};
