// ─────────────────────────────────────────────────────────────────────────────
// CONFIG
let CORRUPTION = {
  // Resonance + re-corruption are currently on hold.
  // (Kept here so the feature can be restored later without losing tuning values.)
  RESONANCE_PER_TIER_MULT: 50,
  RESONANCE_LEVEL_THRESHOLD: 100,
  MAX_RESONANCE_LEVEL: 1000,
  RECORRUPT_CONSUME_LEVELS: 1,

  // Purify rules
  PURIFY_BASE_DESTROY_CHANCE: 0.50, // your existing system was ~50/50
  PURIFY_DESTROY_BONUS_PER_TIER: 0.10, // +10% per corruption tier
  PURIFY_LUCK_REDUCTION: 0.10, // arcane luck reduces destroy chance by 10% (optional, tuneable)
  PURIFY_MAX_DESTROY_CHANCE: 0.95,

  // Empower scaling
  // max spell level grows with corruption tier (cap at 10 by default)
  BASE_SPELL_MAX_LEVEL: 5,
  MAX_SPELL_MAX_LEVEL: 10
}

// ─────────────────────────────────────────────────────────────────────────────
// NBT helpers (Rhino-safe)
function ensureObj(obj, key) {
  if (!obj[key] || typeof obj[key] !== 'object') obj[key] = {}
  return obj[key]
}

function ensureArr(obj, key) {
  if (!obj[key] || !Array.isArray(obj[key])) obj[key] = []
  return obj[key]
}

// ─────────────────────────────────────────────────────────────────────────────
// Corruption state storage (Rhino-safe)
// We store our corruption metadata as a JSON string in NBT to avoid OrderedCompoundTag
// incompatibilities with `typeof` / plain-object mutations.
function kcDefault() {
  return {
    corrupted: 0,
    tier: 0,
    resonancePoints: 0,
    resonanceLevel: 0,
    cursor: 0,
    owned: { spells: [], affixes: [], socketsAdded: 0 }
  }
}

function getKubeCorruption(item) {
  if (!item || item.empty) return kcDefault()

  item.nbt = item.nbt || {}

  let raw = null
  try { raw = item.nbt.kubejs_corruption_json } catch (e) { raw = null }

  let kc = null
  if (raw && typeof raw === 'string') {
    try {
      kc = JSON.parse(raw)              
    } catch (e2) {
      kc = null
    }
  }

  if (!kc || typeof kc !== 'object') kc = kcDefault()

  // Normalize
  if (typeof kc.corrupted !== 'number') kc.corrupted = 0
  if (typeof kc.tier !== 'number') kc.tier = 0
  if (typeof kc.resonancePoints !== 'number') kc.resonancePoints = 0
  if (typeof kc.resonanceLevel !== 'number') kc.resonanceLevel = 0
  if (typeof kc.cursor !== 'number') kc.cursor = 0

  if (!kc.owned || typeof kc.owned !== 'object') kc.owned = {}
  if (!Array.isArray(kc.owned.spells)) kc.owned.spells = []
  if (!Array.isArray(kc.owned.affixes)) kc.owned.affixes = []
  if (typeof kc.owned.socketsAdded !== 'number') kc.owned.socketsAdded = 0

  return kc
}

function saveKubeCorruption(item, kc) {
  if (!item || item.empty) return false
  item.nbt = item.nbt || {}
  item.nbt.kubejs_corruption_json = JSON.stringify(kc)  // ✅ JS stringify
  return true
}

function getLegacyCorruptedFlag(item) {
  if (!item || item.empty) return false
  item.nbt = item.nbt || {}

  try {
    if (typeof item.nbt.getBoolean === 'function' && item.nbt.contains && item.nbt.contains('itemCorrupted')) {
      return !!item.nbt.getBoolean('itemCorrupted')
    }
  } catch (e) {}

  try {
    if (Object.prototype.hasOwnProperty.call(item.nbt, 'itemCorrupted')) return !!item.nbt.itemCorrupted
  } catch (e2) {}

  return false
}

function setLegacyCorruptedFlag(item, val) {
  if (!item || item.empty) return false
  item.nbt = item.nbt || {}

  if (val) {
    try { if (typeof item.nbt.putBoolean === 'function') item.nbt.putBoolean('itemCorrupted', true) } catch (e) {}
    try { item.nbt.itemCorrupted = true } catch (e2) {}
    return true
  }

  try { if (typeof item.nbt.remove === 'function') item.nbt.remove('itemCorrupted') } catch (e3) {}
  try { if (item.nbt && item.nbt.itemCorrupted) delete item.nbt.itemCorrupted } catch (e4) {}
  return true
}


function isCorrupted(item) {
  let kc = getKubeCorruption(item)
  return kc.corrupted === 1 || getLegacyCorruptedFlag(item)
}

function setCorrupted(item, val) {
  let kc = getKubeCorruption(item)
  kc.corrupted = val ? 1 : 0
  saveKubeCorruption(item, kc)
  setLegacyCorruptedFlag(item, !!val)
}

function getCorruptionTier(item) {
  let kc = getKubeCorruption(item)
  return kc.tier || 0
}

function setCorruptionTier(item, tier) {
  let kc = getKubeCorruption(item)
  kc.tier = tier
  saveKubeCorruption(item, kc)
}

function clearKubeCorruption(item) {
  if (!item || item.empty) return false
  saveKubeCorruption(item, kcDefault())
  setLegacyCorruptedFlag(item, false)
  return true
}

// ─────────────────────────────────────────────────────────────────────────────
// Corruption gate: once corrupted, only Repentance can be used
function canOrbModifyItem(player, targetItem, orbId) {
  if (!targetItem || targetItem.empty) return false

  // Not corrupted -> allow anything
  if (!isCorrupted(targetItem)) return true

  // Corrupted -> only allow repentance
  if (orbId === 'kubejs:orb_of_repentance') return true

  // Block everything else
  player.tell(Text.of([
    Text.darkRed('FAILED: '),
    Text.gray('This item is '),
    Text.darkRed('CORRUPTED'),
    Text.gray('. Only '),
    Text.yellow('Orb of Repentance'),
    Text.gray(' can affect it.')
  ]))
  return false
}

// Export gate for other scripts (load-order safe)
global.canOrbModifyItem = canOrbModifyItem


// ─────────────────────────────────────────────────────────────────────────────
// Snapshot helpers to detect what corruption ADDED (without modifying mod NBT formats)
function snapshotState(item) {
  let out = { affixIds: {}, spellIds: {}, sockets: 0 }

  let tag = item.nbt || {}
  let affixData = tag.affix_data
  let affixes = (affixData && affixData.affixes) ? affixData.affixes : {}
  let affixKeys = Object.keys(affixes)
  for (let i = 0; i < affixKeys.length; i++) out.affixIds[String(affixKeys[i])] = true

  let spells = (tag.ISB_Spells && tag.ISB_Spells.data) ? tag.ISB_Spells.data : []
  for (let j = 0; j < spells.length; j++) {
    let sid = spells[j] && spells[j].id ? String(spells[j].id) : ''
    if (sid) out.spellIds[sid] = true
  }

  out.sockets = (affixData && typeof affixData.sockets === 'number') ? affixData.sockets : 0
  return out
}

function recordOwnedAdditions(item, beforeSnap) {
  let kc = getKubeCorruption(item)
  let owned = kc.owned || { spells: [], affixes: [], socketsAdded: 0 }
  owned.spells = owned.spells || []
  owned.affixes = owned.affixes || []
  if (typeof owned.socketsAdded !== 'number') owned.socketsAdded = 0

  let after = snapshotState(item)

  // Newly added affixes
  let tag = item.nbt || {}
  let affixData = tag.affix_data
  let affixes = (affixData && affixData.affixes) ? affixData.affixes : {}
  let affixKeys = Object.keys(affixes)
  for (let i = 0; i < affixKeys.length; i++) {
    let id = String(affixKeys[i])
    if (!beforeSnap.affixIds[id]) {
      // add if not already tracked
      let exists = false
      for (let a = 0; a < owned.affixes.length; a++) {
        if (String(owned.affixes[a].id) === id) { exists = true; break }
      }
      if (!exists) owned.affixes.push({ id: id })
    }
  }

  // Newly added spells
  let spells = (tag.ISB_Spells && tag.ISB_Spells.data) ? tag.ISB_Spells.data : []
  for (let j = 0; j < spells.length; j++) {
    let sid = spells[j] && spells[j].id ? String(spells[j].id) : ''
    if (sid && !beforeSnap.spellIds[sid]) {
      let exists2 = false
      for (let s = 0; s < owned.spells.length; s++) {
        if (String(owned.spells[s].id) === sid) { exists2 = true; break }
      }
      if (!exists2) owned.spells.push({ id: sid })
    }
  }

  // Newly added sockets (track only sockets that corruption added)
  let addedSockets = after.sockets - beforeSnap.sockets
  if (addedSockets > 0) owned.socketsAdded += addedSockets

  kc.owned = owned
  saveKubeCorruption(item, kc)
}

/*
// ─────────────────────────────────────────────────────────────────────────────
// Resonance + re-corruption (ON HOLD)
// Commented out per request so corruption focuses only on:
// - initial corruption outcome (positive/negative)
// - lockout from other orbs until repentance

// Empowerment: upgrade ONLY corruption-owned additions when resonance levels up
function getSpellMaxLevelForTier(tier) {
  let max = CORRUPTION.BASE_SPELL_MAX_LEVEL + Math.max(0, tier - 1)
  if (max > CORRUPTION.MAX_SPELL_MAX_LEVEL) max = CORRUPTION.MAX_SPELL_MAX_LEVEL
  return max
}

function upgradeOwnedSpellOnce(player, item, spellId, maxLevel) {
  let tag = item.nbt || {}
  if (!tag.ISB_Spells || !tag.ISB_Spells.data) return false

  let list = tag.ISB_Spells.data
  for (let i = 0; i < list.length; i++) {
    let s = list[i]
    if (!s || !s.id) continue
    if (String(s.id) === String(spellId)) {
      let lvl = parseInt(s.level, 10)
      if (!lvl || lvl < 1) lvl = 1
      if (lvl >= maxLevel) return false
      s.level = lvl + 1
      list[i] = s
      tag.ISB_Spells.data = list
      item.nbt = tag
      player.tell(Text.of([Text.aqua('[Resonance] '), Text.gray('Spell empowered: '), Text.green(String(spellId).split(':').pop())]))
      return true
    }
  }
  return false
}

function upgradeOwnedAffixOnce(player, item, affixId) {
  let tag = item.nbt || {}
  let affixData = tag.affix_data
  if (!affixData || !affixData.affixes) return false

  let rarity = affixData.rarity || 'apotheosis:common'
  let caps = rarityCapsAndLevels[rarity]
  if (!caps) return false

  let affixes = affixData.affixes
  if (affixes[affixId] == null) return false

  let cur = parseFloat(affixes[affixId])
  if (isNaN(cur)) return false

  let maxCap = caps.maxAffixValue
  if (typeof maxCap !== 'number') maxCap = 0.10

  if (cur >= maxCap) return false

  let remaining = maxCap - cur
  let boosted = cur + (remaining * 0.20)
  if (boosted > maxCap) boosted = maxCap

  affixes[affixId] = boosted
  tag.affix_data.affixes = affixes
  item.nbt = tag

  player.tell(Text.of([Text.aqua('[Resonance] '), Text.gray('Affix empowered: '), Text.lightPurple(String(affixId).split('/').pop())]))
  return true
}

function upgradeOwnedSocketsOnce(player, item) {
  let kc = getKubeCorruption(item)
  if (!kc.owned || typeof kc.owned.socketsAdded !== 'number' || kc.owned.socketsAdded <= 0) return false

  let ok = modifySockets(player, item, 1, true)
  if (ok) {
    player.tell(Text.of([Text.aqua('[Resonance] '), Text.gray('Sockets empowered.')]))
    return true
  }
  return false
}

function applyResonanceEmpower(player, item, levelsGained) {
  if (!levelsGained || levelsGained <= 0) return false

  let kc = getKubeCorruption(item)
  let tier = kc.tier || 1
  let maxSpellLevel = getSpellMaxLevelForTier(tier)

  let owned = kc.owned || { spells: [], affixes: [], socketsAdded: 0 }
  if (!Array.isArray(owned.spells)) owned.spells = []
  if (!Array.isArray(owned.affixes)) owned.affixes = []
  if (typeof owned.socketsAdded !== 'number') owned.socketsAdded = 0

  let any = false

  for (let n = 0; n < levelsGained; n++) {
    let attempts = 0
    let didOne = false

    while (attempts < 12 && !didOne) {
      attempts++

      let mode = kc.cursor % 3 // 0 spells, 1 affixes, 2 sockets
      kc.cursor = (kc.cursor + 1) % 300000

      if (mode === 0 && owned.spells.length > 0) {
        let si = kc.cursor % owned.spells.length
        let sid = owned.spells[si] && owned.spells[si].id ? owned.spells[si].id : null
        if (sid && upgradeOwnedSpellOnce(player, item, sid, maxSpellLevel)) {
          didOne = true; any = true
        } else {
          let stillExists = false
          let tag = item.nbt || {}
          let list = (tag.ISB_Spells && tag.ISB_Spells.data) ? tag.ISB_Spells.data : []
          for (let x = 0; x < list.length; x++) {
            if (list[x] && String(list[x].id) === String(sid)) { stillExists = true; break }
          }
          if (!stillExists) owned.spells.splice(si, 1)
        }
        continue
      }

      if (mode === 1 && owned.affixes.length > 0) {
        let ai = kc.cursor % owned.affixes.length
        let aid = owned.affixes[ai] && owned.affixes[ai].id ? owned.affixes[ai].id : null
        if (aid && upgradeOwnedAffixOnce(player, item, aid)) {
          didOne = true; any = true
        } else {
          let tag2 = item.nbt || {}
          let affixData = tag2.affix_data
          let affixes = (affixData && affixData.affixes) ? affixData.affixes : {}
          if (affixes[aid] == null) owned.affixes.splice(ai, 1)
        }
        continue
      }

      if (mode === 2) {
        if (upgradeOwnedSocketsOnce(player, item)) {
          didOne = true; any = true
        }
        continue
      }
    }
  }

  kc.owned = owned
  saveKubeCorruption(item, kc)
  return any
}

// Resonance add + level up
function addResonanceToItem(player, item, points) {
  if (!item || item.empty) return false
  if (!isCorrupted(item)) return false

  let kc = getKubeCorruption(item)
  let threshold = CORRUPTION.RESONANCE_LEVEL_THRESHOLD

  kc.resonancePoints = (kc.resonancePoints || 0) + points
  if (kc.resonancePoints < 0) kc.resonancePoints = 0

  let gained = 0
  while (kc.resonancePoints >= threshold && kc.resonanceLevel < CORRUPTION.MAX_RESONANCE_LEVEL) {
    kc.resonancePoints -= threshold
    kc.resonanceLevel += 1
    gained += 1
  }

  saveKubeCorruption(item, kc)
  applyResonanceLore(item, kc)

  if (gained > 0) {
    player.tell(Text.of([
      Text.aqua('[Resonance] '),
      Text.gray('Item resonance increased: '),
      Text.green('+' + gained),
      Text.gray(' level(s).')
    ]))

    player.persistentData.putBoolean('corruption_in_progress', true)
    try {
      applyResonanceEmpower(player, item, gained)
    } finally {
      player.persistentData.putBoolean('corruption_in_progress', false)
    }
  }

  return true
}

function applyResonanceLore(item, kc) {
  if (!item || item.empty) return

  let cap = CORRUPTION.RESONANCE_LEVEL_THRESHOLD
  if (typeof cap !== 'number' || cap <= 0) cap = 100

  let pts = 0
  try { pts = kc && typeof kc.resonancePoints === 'number' ? kc.resonancePoints : 0 } catch (e) { pts = 0 }
  if (pts < 0) pts = 0
  if (pts > cap) pts = cap

  let pct = Math.floor((pts / cap) * 100)

  item.nbt = item.nbt || {}
  item.nbt.display = item.nbt.display || {}

  let lore = []
  let raw = item.nbt.display.Lore
  if (raw && raw.length) {
    for (let i = 0; i < raw.length; i++) {
      let line = String(raw[i])
      if (line.indexOf('Resonance:') !== -1) continue
      lore.push(raw[i])
    }
  }

  let txt = 'Resonance: ' + pct + '% (' + pts + '/' + cap + ')'
  lore.push(JSON.stringify({ text: txt, color: 'red', italic: true }))

  item.nbt.display.Lore = lore
}

// Boss kill hook: add tier*5 resonance to a CORRUPTED item.
EntityEvents.death(event => {
  if (!event.source.player) return
  if (!event.entity.tags.contains('boss')) return
  let isBoss
  let tier
  Object.entries(global.boss_data).forEach(([entity, data]) => {
    if (event.entity.type.toString().includes(data.id)) {
      isBoss = true
      tier = data.tier
    }
  })
  if (!isBoss) return
  let player = event.source.player
  let gain = tier * CORRUPTION.RESONANCE_PER_TIER_MULT

  let main = player.mainHandItem
  let off = player.offHandItem

  let target = null
  if (main && !main.empty && isCorrupted(main)) target = main
  else if (off && !off.empty && isCorrupted(off)) target = off
  else {
    let foundSlot = -1
    let foundCount = 0
    for (let i = 0; i < 36; i++) {
      let it = player.inventory.getItem(i)
      if (it && !it.empty && isCorrupted(it)) {
        foundCount++
        foundSlot = i
        if (foundCount > 1) break
      }
    }
    if (foundCount === 1 && foundSlot >= 0) target = player.inventory.getItem(foundSlot)
  }

  if (!target) {
    player.tell(Text.gray('[Resonance] Hold a corrupted item to absorb resonance from boss kills.'))
    return
  }

  addResonanceToItem(player, target, gain)
})



function destroyItem(item) {
    let slot = player.inventory.findSlotMatchingItem(item)
    if (slot >= 0) player.inventory.setStackInSlot(slot, 'supplementaries:ash')
    item.count = 0
    player.tell('§cThe Corruption devours the item entirely.')
    destroyed = true
    return true
}
*/

function destroyItem(player, item) {
  if (!player || !item || item.empty) return false
  item.count = 0
  player.tell('§cThe Corruption devours the item entirely.')
  return true
}



// ─────────────────────────────────────────────────────────────────────────────
// Corruption orb behavior
// - If item NOT corrupted: perform initial corruption (can be positive/negative).
function useCorruptionOrb(player, item) {
  if (!item || item.empty) return false

  // Not corrupted -> initial corruption
  return initialCorrupt(player, item)
}

function initialCorrupt(player, item) {
  // Must be a valid target (optional: keep your checks here if you want)
  if (!checkCorruption(player, item, false)) return false // should pass because not corrupted

  // Prevent other orbs once corrupted: we mark only after actions succeed/complete
  let before = snapshotState(item)

  // Allow caps override during corruption transaction
  player.persistentData.putBoolean('corruption_override_caps', true)
  player.persistentData.putBoolean('corruption_in_progress', true)
  try {
    // Requirements: at least uncommon, at least 1 affix and 1 spell (same spirit as your old system)
    item.nbt = item.nbt || {}
    let rarity = (item.nbt.affix_data && item.nbt.affix_data.rarity) ? String(item.nbt.affix_data.rarity) : 'apotheosis:common'

    // rarity >= uncommon by key order
    let keys = Object.keys(rarityCapsAndLevels)
    let curIdx = keys.indexOf(rarity)
    let minIdx = keys.indexOf('apotheosis:uncommon')
    if (curIdx < 0) curIdx = 0
    if (minIdx < 0) minIdx = 1
    if (curIdx < minIdx) {
      player.tell(Text.red('Corruption requires the item to be at least UNCOMMON rarity.'))
      return false
    }

    let affixes = (item.nbt.affix_data && item.nbt.affix_data.affixes) ? item.nbt.affix_data.affixes : {}
    if (Object.keys(affixes).length === 0) {
      player.tell(Text.red('Corruption requires the item to have at least one affix.'))
      return false
    }

    let spells = (item.nbt.ISB_Spells && item.nbt.ISB_Spells.data) ? item.nbt.ISB_Spells.data : []
    if (!spells || spells.length === 0) {
      player.tell(Text.red('Corruption requires the item to have at least one imbued spell.'))
      return false
    }

    // 50/50 outcome
    let hasLuck = player.hasEffect('kubejs:arcane_luck')
    let positive = Math.random() > 0.50

    let anyChange = false
    let destroyed = false



    function noChange() {
      player.tell('The Corruption latches to your item... but does nothing else.')
      return false
    }

    // NOTE: keep these aligned to your real function signatures.
    // Your file currently uses applyRandomAffix(player, item) and imbueItem(player, item) etc.
    let positiveActions = [
      function () { return upgradeItemRarity(player, item) },
      function () { return applyRandomAffix(player, item) },
      function () { return imbueItem(player, item) },
      function () { return upgradeImbuedSpell(player, item) },
      function () { return modifySockets(player, item, hasLuck ? 2 : 1, true) },
      function () { return lockAffix(player, item) },
      function () { return lockImbuedSpell(player, item) }
    ]

    let negativeActions = [
      function () { return destroyItem(player, item) },
      function () { return noChange() },
      function () { return removeRandomAffix(player, item) },
      function () { return removeRandomSpell(player, item) },
      function () { return downgradeItemRarity(player, item, true) },
      function () { return downgradeRandomAffix(player, item) },
      function () { return downgradeRandomSpell(player, item) },
      function () { return modifySockets(player, item, -1, true) }
    ]

    if (positive) {
      // Guarantee exactly ONE positive action on initial corruption (keeps it readable/controlled)
      let fn = positiveActions[Math.floor(Math.random() * positiveActions.length)]
      if (fn && fn()) anyChange = true
    } else {
      // Exactly one negative
      let fn2 = negativeActions[Math.floor(Math.random() * negativeActions.length)]
      if (fn2 && fn2()) anyChange = true
    }

    if (destroyed || item.count <= 0 || item.empty) {
      player.tell('§cThe Corruption leaves nothing behind…')
      return true
    }

    // Mark corrupted + initialize corruption data (tier 1)
    let kc = getKubeCorruption(item)
    kc.corrupted = 1
    kc.tier = 1
    kc.resonancePoints = 0
    kc.resonanceLevel = 0
    kc.cursor = 0
    kc.owned = kc.owned || { spells: [], affixes: [], socketsAdded: 0 }
    saveKubeCorruption(item, kc)

    // Track what corruption ADDED (without touching mod formats)
    recordOwnedAdditions(item, before)

    // Lore mark (optional; does not affect other mods)
    applyCorruptionLore(player, item)


    player.tell(positive
      ? '§cThe Corruption twists in your favour…'
      : (anyChange ? '§cThe Corruption lashes out with malevolence…' : Text.red('The Corruption stirs, but fails to alter the item.'))
    )

    return true
  } finally {
    player.persistentData.putBoolean('corruption_in_progress', false)
    player.persistentData.putBoolean('corruption_override_caps', false)
  }
}

/*
// Re-corrupt tier-up (ON HOLD)
function recorruptTierUp(player, item) {
  if (!item || item.empty) return false
  if (!isCorrupted(item)) return false

  let kc = getKubeCorruption(item)
  let need = CORRUPTION.RECORRUPT_CONSUME_LEVELS

  if ((kc.resonanceLevel || 0) < need) {
    player.tell(Text.red('This item lacks the resonance to deepen its corruption.'))
    return false
  }

  // Consume resonance levels and increase corruption tier
  kc.resonanceLevel -= need
  if (kc.resonanceLevel < 0) kc.resonanceLevel = 0
  kc.tier = (kc.tier || 1) + 1
  saveKubeCorruption(item, kc)

  // No negatives on re-corrupt: just apply one empowerment tick immediately
  // (This makes re-corrupt feel like a meaningful action.)
  applyResonanceEmpower(player, item, 1)

  // Update lore to reflect tier
  applyCorruptionLore(player, item)

  player.tell(Text.of([
    Text.darkRed('CORRUPTION DEEPENS '),
    Text.gray('(Tier ' + kc.tier + ')')
  ]))

  return true
}
*/

// Simple lore marker that shows tier (does not interfere with other mods)
function applyCorruptionLore(player, item) {
  let kc = getKubeCorruption(item)
  let tier = kc.tier || 1

  item.nbt = item.nbt || {}
  item.nbt.display = item.nbt.display || {}

  let lore = []
  let raw = item.nbt.display.Lore
  if (raw && raw.length) {
    for (let i = 0; i < raw.length; i++) {
      let line = String(raw[i])
      // Remove previous CORRUPTED lines (we rewrite them)
      if (line.indexOf('CORRUPTED') !== -1) continue
      lore.push(raw[i])
    }
  }

  lore.push(JSON.stringify({ text: 'CORRUPTED (Tier ' + tier + ')', color: 'dark_red', italic: true }))
  item.nbt.display.Lore = lore

  // Keep your old boolean too if you still want it for other checks
  if (typeof item.nbt.putBoolean === 'function') item.nbt.putBoolean('itemCorrupted', true)
  try { item.nbt.itemCorrupted = true } catch (e) {}
}

// ─────────────────────────────────────────────────────────────────────────────
// Repentance: 50/50 purify or destroy.
// - Clears corruption state so orbs can be used again
// - 50/50 to remove corruption or destroy item
// - Does not remove affixes/spells/skills the item already has
function repentance50_50(player, item) {
  if (!item || item.empty) return false

  if (!isCorrupted(item)) {
    player.tell(Text.gray('Corruption is not present on this item'))
    return false
  }

  // 50/50: destroy vs cleanse
  if (Math.random() < 0.50) {
    item.count = 0
    player.tell(Text.of([
      Text.of('The Orb of Repentance').gray(),
      Text.of(' FAILS').red(),
      Text.of(', shattering the item').gray(),
    ]))
    player.level.runCommandSilent(`/execute in ${player.level.dimension} run playsound minecraft:entity.wither.death ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.5`)
    return true
  }

  // Success: remove corruption state (JSON + legacy flag)
  clearKubeCorruption(item)

  // Clean lore: remove CORRUPTED + any leftover Resonance lines
  if (item.nbt && item.nbt.display && item.nbt.display.Lore) {
    let cleaned = []
    for (let i = 0; i < item.nbt.display.Lore.length; i++) {
      let line = String(item.nbt.display.Lore[i])
      if (line.indexOf('CORRUPTED') !== -1) continue
      if (line.indexOf('Resonance:') !== -1) continue
      cleaned.push(item.nbt.display.Lore[i])
    }
    item.nbt.display.Lore = cleaned
  }

  if (item.id && String(item.id).includes('totem')) {
    player.level.runCommandSilent(`/execute in ${player.level.dimension} run playsound kubejs:trigger ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 2`)
  }

  player.tell(Text.white('PURIFIED'))
  player.level.runCommandSilent(`/execute in ${player.level.dimension} run particle irons_spellbooks:spark 1 1 1 ${player.x} ${player.y+1.5} ${player.z} 0.5 0.5 0.5 0.25 250 force ${player.username}`)
  return true
}

function cleanseCorruptionOverhaul(player, item) {
  return repentance50_50(player, item)
}

function cleanseCorruption(player, item) {
  return repentance50_50(player, item)
}
