

//__________________________________________________________________________________________________________________________________________________________________________
function applyRandomAffix(player, item, dontTell) {
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false

  let rarity  = global.apothRarityHelper(item)
  let affixes = item.nbt?.affix_data?.affixes || {}

  /* ── cap logic ────────────────────────────────────────────────────── */
  let ignoreCaps = player?.persistentData?.corruption_override_caps === true
  let levelCap   = ignoreCaps ? Infinity : 2 + (rarityCapsAndLevels[rarity].maxAffixes)
  if (Object.keys(affixes).length >= levelCap && !ignoreCaps) {
    if (!dontTell) {
      player.tell(Text.of('Upgrade this item to add more Affixes.').red())
    }
    return false
  }

  /* ── choose a new affix ───────────────────────────────────────────── */
  let validPool = global.getAvailableApothAffixes(item)           // <‑‑ list of Affix objects
  validPool = validPool.filter(id => getAffixDetails(id)?.type !== 'mob_effect')
  //tell(validPool)
  if (validPool.length === 0) {
    if (!dontTell) {
      player.tell(Text.red('No valid affixes available for this item.'))
    }
    return false
  }
  player.level.runCommandSilent(`/ftbquests change_progress ${player.username} reset 291FCA909BE917D2`)
  player.level.runCommandSilent(`/ftbquests change_progress ${player.username} complete 291FCA909BE917D2`)
  player.persistentData.putBoolean('orb_of_infusion', true)

  /* convert Affix object → registry‑name string */
  let affixId = validPool[Math.floor(Math.random() * validPool.length)]
  //tell(description)
  let value = getAffixValue(rarity)
  //console.log(getAffixDetails(affixId).id)
  

  // Apply it
  applyAffix(item, affixId, value)
       // "Cursed Flame"
  let affixDetails = getAffixDetails(affixId)
  let affixType = String(affixDetails.type)
    .split('/').pop()                                  // "cursed_flame"
    .replace(/_/g, ' ')                                // "cursed flame"
    .replace(/\b\w/g, c => c.toUpperCase())   

  let added
  if (affixDetails.type != 'other') {
    added = String(affixDetails.id)
    if (added) {
        added = added
        .split('.').pop()
        .split(':').pop()
        .split('/').pop()                                  // "cursed_flame"
        .replace(/_/g, ' ')                                // "cursed flame"
        .replace(/\b\w/g, c => c.toUpperCase())
    }
  } else {
    added = affixDetails.affixClass
    .split('affix.effect.').pop()
    .split('Affix')[0]
  }

   
  //console.log(String(affixDetails.type))
  if (!dontTell && affixDetails) {
      player.tell(Text.green(`Added → `).append(Text.lightPurple(added)).append(Text.gray(` (${affixType})`)))
    }


  console.log(`${player.username} applies: ${affixId}`)
  
  /* ── surge roll (repeat until cap) ────────────────────────────────── */
  let currentCount = Object.keys(affixes).length
  if (surgeChance(player) && currentCount < levelCap) {
    if (!dontTell) {
      player.tell(Text.of([
        Text.of('[Arcane Surge] → ').aqua(),
        Text.of('A bonus affix is being infused...').gray()
      ]))
      applyRandomAffix(player, item)  // unlimited surges until cap
    } else {
      applyRandomAffix(player, item, true)  // unlimited surges until cap
    }
    
  }

  return true
}


function getAffixValue(rarity) {
  let max = rarityCapsAndLevels[rarity].maxAffixValue ?? 0.10
  return Math.random() * max
}




// Orb of Infusion – low‑level writer (no cap logic here)
function applyAffix(item, affixId, value) {
  if (!item || item.empty) return false

  let tag = item.nbt || {}
  if (!tag.affix_data)           tag.affix_data = { affixes: {}, uuids: [[0, 0, 0, 0]] }
  if (!tag.affix_data.affixes)   tag.affix_data.affixes = {}

  tag.affix_data.affixes[affixId] = value
  item.nbt = tag
  return true
}

////__________________________________________________________________________________________________________________________________________________________________________
function swapRandomAffixes(player, item) {
    let corruptionCheck = checkCorruption(player, item)
    if (!corruptionCheck) return
    if (!item || item.empty) return item
    removeRandomAffix(player, item)

    applyRandomAffix(player, item)

    return item
}


function removeAffix(item, affixId) {
  if (!item || item.empty) return item

  let tag = item.nbt || {}

  // If there's no affix_data or affixes object, nothing to remove
  if (!tag.affix_data || !tag.affix_data.affixes) return item

  // Delete the affix if it exists
  if (tag.affix_data.affixes[affixId]) {
    delete tag.affix_data.affixes[affixId]
  }

  item.nbt = tag
  return item
}

function removeRandomAffix(player, item) {
  if (!item || item.empty) return item

  let tag = item.nbt || {}
  let affixes = tag.affix_data?.affixes
  if (!affixes || Object.keys(affixes).length === 0) return item

  let locked = tag.affix_data.lockedAffix

  // Filter out the locked affix
  let affixIds = Object.keys(affixes).filter(id => id !== locked)
  if (affixIds.length === 0) return item // All affixes are locked

  let chosen = affixIds[Math.floor(Math.random() * affixIds.length)]
  let type = getAffixDetails(chosen).type
  let str = String(chosen)
    .split('.').pop()
    .split(':').pop()
    .split('/').pop()                                  // "cursed_flame"
    .replace(/_/g, ' ')                                // "cursed flame"
    .replace(/\b\w/g, c => c.toUpperCase())
  let typeStr = String(type)
    .split('/').pop()                                  // "cursed_flame"
    .replace(/_/g, ' ')                                // "cursed flame"
    .replace(/\b\w/g, c => c.toUpperCase())

  player.tell(Text.of([Text.of('Removed → ').darkRed(), Text.of(str).lightPurple(), Text.of(` (${typeStr})`).gray()]))
  return removeAffix(item, chosen)
}

//__________________________________________________________________________________________________________________________________________________________________________
function upgradeItemRarity(player, item, dontTell) { // Orb of Exaltation
  if (!item || item.empty) return false
  
  if (!checkCorruption(player, item)) {
    player.tags.remove('orb_of_exaltation')
    return false
  }

  let keys = Object.keys(rarityCapsAndLevels)
  global.apothRarityHelper(item)
  let currentRarity = item.nbt.affix_data.rarity || 'apotheosis:common'
  let index = keys.indexOf(currentRarity)
  let nextName = keys[index + 1]


  if (!nextName) {
    player.tell(Text.gray("This item has already reached its final form."))
    player.tags.remove('orb_of_exaltation')
    return false
  }

  let nextId = nextName
  applyRarity(item, nextId)
  let oldRarity = String(currentRarity)
    .split('.').pop()
    .split(':').pop()
    .split('/').pop()                                  // "cursed_flame"
    .replace(/_/g, ' ')                                // "cursed flame"
    .replace(/\b\w/g, c => c.toUpperCase())

  let newRarity = String(nextName)
    .split('.').pop()
    .split(':').pop()
    .split('/').pop()                                  // "cursed_flame"
    .replace(/_/g, ' ')                                // "cursed flame"
    .replace(/\b\w/g, c => c.toUpperCase())
  if (!dontTell) {
    player.tell(Text.gray('[Rarity upgraded]: ')
      .append(Text.red(capitalize(oldRarity)).strikethrough(true))
      .append(Text.white(' → '))
      .append(Text.green(capitalize(newRarity)))
    )
  }

  let surge = surgeChance(player)
  if (surge) {
    if (!dontTell) {
      player.tell(Text.of([
        Text.of('[Divine Surge] → ').yellow(),
        Text.of('A surge of divine energy flows through the item, upgrading it further...').gray()
      ]))
      upgradeItemRarity(player, item)
    }
  }
  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
function checkCorruption(player, item, isRepentance) {
  // Treat empty item as invalid target
  if (!item || item.empty) return false

  // Ensure NBT exists so property reads do not throw
  item.nbt = item.nbt || {}

  let corrupted = false
  try {
    // New system: KubeJS corruption JSON
    let raw = null
    try { raw = item.nbt.kubejs_corruption_json } catch (e0) { raw = null }
    if (raw && typeof raw === 'string') {
      try {
        let kc = JSON.parse(raw)
        if (kc && typeof kc === 'object' && kc.corrupted === 1) corrupted = true
      } catch (e1) {}
    }

    // Legacy marker: CompoundTag boolean accessor / plain property
    if (!corrupted) {
      if (typeof item.nbt.getBoolean === 'function' && item.nbt.contains && item.nbt.contains('itemCorrupted')) {
        corrupted = !!item.nbt.getBoolean('itemCorrupted')
      } else if (Object.prototype.hasOwnProperty.call(item.nbt, 'itemCorrupted')) {
        corrupted = !!item.nbt.itemCorrupted
      }
    }
  } catch (e) {
    corrupted = !!(item.nbt && item.nbt.itemCorrupted)
  }


  // Overhaul bypass: allow corruption/resonance to mutate while a transaction is in progress
  let inProgress = false
  try {
    inProgress = player && player.persistentData && player.persistentData.getBoolean('corruption_in_progress') === true
  } catch (e2) {}
  if (corrupted && inProgress) return true

  if (corrupted && !isRepentance) {
    player.tell(Text.of([
      //Text.gray('Modification failed because this item is'),
      //Text.red(' CORRUPTED'),
      Text.darkRed(`FAILED: `),
      Text.gray('Remove corruption with an '),
      Text.yellow('Orb of Repentance')
    ]))
    player.level.runCommandSilent(`/execute in ${player.level.dimension} run playsound minecraft:entity.wither.death ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 2`)
    player.level.runCommandSilent(`/execute in ${player.level.dimension} run particle fathomless:crimson_aura ${player.x} ${player.y+1} ${player.z} 0.3 0.5 0.3 1 100 force ${player.username}`)
    return false
  }
  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
function upgradeRarity(item) {
  if (!item || item.empty) return item

  let rarity  = global.apothRarityHelper(item)

  // Already at max rarity
  return item
}

// ─────────────────────────────────────────────────────────────────────────────
// Fixed downgrade rarity (Rhino-safe, key-order based)
function downgradeItemRarity(player, item, dontTell) {
  if (!item || item.empty) return false
  if (player && !checkCorruption(player, item)) return false

  item.nbt = item.nbt || {}
  item.nbt.affix_data = item.nbt.affix_data || {}

  let keys = Object.keys(rarityCapsAndLevels)
  let currentRarity = item.nbt.affix_data.rarity || 'apotheosis:common'
  let index = keys.indexOf(String(currentRarity))
  if (index < 0) index = 0

  if (index <= 0) {
    if (!dontTell && player) player.tell(Text.gray("This item can't be downgraded any further."))
    return false
  }

  let prevRarity = keys[index - 1]
  applyRarity(item, prevRarity)

  if (!dontTell && player) {
    function pretty(id) {
      id = String(id).split('.').pop().split(':').pop().split('/').pop()
      id = id.replace(/_/g, ' ')
      id = id.replace(/\b\w/g, function (c) { return c.toUpperCase() })
      return id
    }
    player.tell(
      Text.gray('[Rarity downgraded]: ')
        .append(Text.red(pretty(currentRarity)).strikethrough(true))
        .append(Text.white(' → '))
        .append(Text.darkRed(pretty(prevRarity)))
    )
  }

  return true
}


//__________________________________________________________________________________________________________________________________________________________________________
  
//________________________________________________________________________________________________________________
function applyRarity(item, rarityId) {
  if (!item || item.empty) return item

  let tag = item.nbt || {}

  if (!tag.affix_data) {
    tag.affix_data = {}
  }

  tag.affix_data.rarity = rarityId

  item.nbt = tag
  return item
}





//___________________________________________________________________________________________________________________
function swapAffix(item, affixToRemove, affixToAdd, affixToAddValue, player) {
  if (!item || item.empty) return item

  // Format names
  let formatAffixName = id => {
    let name = id.split('/').pop().split(':').pop()
    return name.split('_').map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(' ')
  }

  let removedName = formatAffixName(affixToRemove)
  let addedName = formatAffixName(affixToAdd)
  let addedPercent = (affixToAddValue * 100).toFixed(1)

  // Remove and apply
  item = removeAffix(item, affixToRemove)
  item = applyAffix(item, affixToAdd, affixToAddValue, player)

  // Message
  player.tell(
    Text.gray('[Affix Swapped]: ')
      .append(Text.red(`${removedName}`))
      .append(Text.white(' → '))
      .append(Text.green(`${addedName}`))
      .append(Text.gray(` (+${addedPercent}%)`))
  )

  return item
}
//__________________________________________________________________________________________________________________________________________________________________________
// Orb of Ascension – upgrades one random numeric affix
function upgradeRandomAffix(player, item) {
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false          // helper already notifies player
  // ─── fast sanity checks ───────────────────────────────────────────────────
  if (!item || !item.nbt?.affix_data?.affixes) {
    player.tell(Text.red('No affixes found to upgrade.'))
    return false
  }

  // ─── references and basic info ────────────────────────────────────────────
  let affixes    = item.nbt.affix_data.affixes
  let rarity     = item.nbt.affix_data.rarity
  let ignoreCap  = player.persistentData.corruption_override_caps === true

  // numeric keys only
  let keys = Object.keys(affixes).filter(k =>
    typeof affixes[k] === 'number' || affixes[k] instanceof java.lang.Number)

  if (keys.length === 0) {
    player.tell(Text.red('No numeric affixes available.'))
    return false
  }

  // filter out those already at cap (unless overrides are on)
  let upgradable = keys.filter(k =>
    ignoreCap || affixes[k] < rarityCapsAndLevels[rarity].maxAffixValue)

  if (upgradable.length === 0) {
    player.tell(Text.red('Upgrade your item rarity to continue upgrading affixes.'))
    return false
  }

  // ─── pick one affix and calculate new value ───────────────────────────────
  let selected   = upgradable[Math.floor(Math.random() * upgradable.length)]
  let current    = parseFloat(affixes[selected])
  let maxCap     = ignoreCap ? 1.0 : rarityCapsAndLevels[rarity].maxAffixValue
  // simple stepping: raise value by 20 % of the remaining distance to the cap
  let remaining  = maxCap - current
  let step       = remaining / 5             // 5 equal steps to reach the cap
  let   boosted    = current + step

  if (!ignoreCap && boosted > maxCap) boosted = maxCap

  // ─── write-back: remove then re‑apply via applyAffix ───────────────────────
  delete affixes[selected]
  item.nbt.affix_data.affixes = affixes
  removeAffix(item, selected)              // your existing helper
  applyAffix(item, selected, boosted, player)

  // ─── nice display name ────────────────────────────────────────────────────
  let displayName = selected
  if (displayName.includes('/')) displayName = displayName.split('/').pop()
  if (displayName.includes(':')) displayName = displayName.split(':').pop()
  displayName = displayName.split('_')
    .map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(' ')

  // ─── progress percentages ─────────────────────────────────────────────────
  let pctCurrent  = ((current  / maxCap) * 100).toFixed(1)
  let pctBoosted  = ((boosted  / maxCap) * 100).toFixed(1)

  // 1. Upgraded‑affix announcement
  player.tell(
    Text.gray('[Upgraded Affix]: ')
      .append(Text.gold(displayName))
  )

  
  let rarityDisplayName = rarity.split(':')[1]
  let rarityColor = {
    'common':   Text.gray,
    'uncommon': Text.green,
    'rare':     Text.blue,
    'epic':     Text.lightPurple,
    'mythic':   Text.gold,
    'ancient':  Text.red
  }
  // Capitalise the rarity for display (Rare, Epic, etc.)
  let prettyRarity = rarityDisplayName.charAt(0).toUpperCase() + rarityDisplayName.slice(1)

  // ─── Progress read‑out ───────────────────────────────────────────────────────
  player.tell(
    Text.of([
      Text.white('['),                                 // opening bracket
      rarityColor[rarityDisplayName](prettyRarity),    // coloured rarity
      Text.white('] '),                                // closing bracket + space
      Text.white('Progress: '),                         // label
      Text.red(`${pctCurrent}%`),
      Text.white(' → '),
      Text.green(`${pctBoosted}%`)
    ])
  )


  // ─── Arcane Surge roll – chain until fail or full cap ─────────────────────
  if (surgeChance(player) && !ignoreCap && boosted < maxCap) {
    player.tell(Text.of([
      Text.of('[Arcane Surge] → ').aqua(),
      Text.of('A bonus affix is being upgraded...').gray()
    ]))

    // delay 2 ticks so NBT writes are committed, then upgrade again
    Utils.server.scheduleInTicks(2, () => upgradeRandomAffix(player, item))
  }

  return true
}




// ─────────────────────────────────────────────────────────────────────────────
// Helpers – downgrade routines
function downgradeRandomAffix(player, item) {
  let affixes = item.nbt?.affix_data?.affixes || {}
  let locked = item.nbt?.affix_data?.lockedAffix || ''
  let keys = Object.keys(affixes).filter(k =>
    (typeof affixes[k] === 'number' || affixes[k] instanceof java.lang.Number) &&
    k !== locked
  )

  if (keys.length === 0) {
    if (Object.keys(affixes).length > 0 && locked) {
      player.tell(Text.gold('Fractured Affix rejected the corruption.'))
    }
    return false
  }

  let id = keys[Math.floor(Math.random() * keys.length)]
  let value = parseFloat(affixes[id])
  value *= 0.80
  if (value < 0) value = 0

  removeAffix(item, id)
  applyAffix(item, id, value, player)
  player.tell(Text.red(`The corruption weakens ${id.split(/[/:]/).pop()}…`))
  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
function downgradeRandomSpell(player, item) {
  let data = item.nbt?.ISB_Spells?.data || []
  let locked = getLockedSpellsArray(item.nbt)

  let unlockedIndices = []
  for (let i = 0; i < data.length; i++) {
    let id = String(data[i].id)
    if (!locked.includes(id)) unlockedIndices.push(i)
  }

  if (unlockedIndices.length === 0) {
    if (data.length > 0) {
      player.tell(Text.gold('Locked spell rejected the corruption.'))
    } else {
      player.tell(Text.gray('No spells found on this item.'))
    }
    return false
  }

  let idx = unlockedIndices[Math.floor(Math.random() * unlockedIndices.length)]
  let s = data[idx]

  if (s.level && s.level > 1) {
    s.level -= 1
    player.tell(Text.red('A spell falters, losing a level…'))
  } else {
    data.splice(idx, 1)
    player.tell(Text.red('A spell is ripped from the item!'))
  }

  item.nbt.ISB_Spells.data = data
  return true
}


function noItemChange(player, item) {
  player.tell('The Corruption latches to your item... but does nothing else.')
  return false
}

function corruptionDestroysItem(player, item) {
  let slot = player.inventory.findSlotMatchingItem(item)
  item.count -= 1
  player.inventory.setStackInSlot(slot, 'supplementaries:ash')
  player.tell('§cThe Corruption devours the item entirely.')
  player.level.runCommandSilent(
    `/execute in ${player.level.dimension} run playsound minecraft:entity.generic.extinguish_fire ambient ${player.username} ${player.x} ${player.y} ${player.z} 1 1.1`
  )
  return false
}


//__________________________________________________________________________________________________________________________________________________________________________
function lockAffix(player, item) {
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false          // helper already notifies player
  if (!item || item.empty) return false
  item.nbt = item.nbt || {}
  item.nbt.affix_data = item.nbt.affix_data || {}
  item.nbt.affix_data.affixes = item.nbt.affix_data.affixes || {}

  let affixes = Object.keys(item.nbt.affix_data.affixes)
  if (affixes.length === 0 || item.nbt.affix_data.lockedAffix) {
    player.tell(Text.red('No affixes can be locked'))
    return false
  }

  let chosen = affixes[Math.floor(Math.random() * affixes.length)]
  item.nbt.affix_data.lockedAffix = chosen

  // Format display name
  let details = getAffixDetails(chosen)
  let type = details.type
  let str
  if (type == 'other') {
    str = String(chosen)
      .split('.').pop()
      .split(':').pop()
      .split('/').pop()                                  // "cursed_flame"
      .replace(/_/g, ' ')                              // "cursed flame"
      .replace(/\b\w/g, c => c.toUpperCase())
  } else {
    str = String(getAffixDetails(chosen).id)
        .split('.').pop()
        .split(':').pop()
        .split('/').pop()                                  // "cursed_flame"
        .replace(/_/g, ' ')                                // "cursed flame"
        .replace(/\b\w/g, c => c.toUpperCase())
  }

  let msg = Text.gold('[Locking Affix]: ').append(Text.red(str))
  player.tell(msg)
  // Add lore to the item
  item.nbt.display = item.nbt.display || {}
  let lore = item.nbt.display.Lore || []

  lore.push(JSON.stringify([
    { text: 'Locked Affix: ', color: 'gold', italic: false },
    { text: str, color: 'red', italic: false }
  ]))

  item.nbt.display.Lore = lore
  return true
}


function lockImbuedSpell(player, item) {
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false          // helper already notifies player
  if (!item || item.empty) return false

  // Make sure spell list exists
  if (!item.nbt || !item.nbt.ISB_Spells || !Array.isArray(item.nbt.ISB_Spells.data)) {
    player.tell(Text.red("This item has no spells to lock."))
    return false
  }

  let nbt = item.nbt
  let spellList = nbt.ISB_Spells.data

  // Normalize lockedSpells to a JS array (preserve existing entries from Java List)
  let lockedSpells = getLockedSpellsArray(nbt)
  nbt.lockedSpells = lockedSpells

  // Build list of spells not yet locked
  let unlocked = []
  for (let i = 0; i < spellList.length; i++) {
    let id = String(spellList[i].id)
    if (lockedSpells.indexOf(id) === -1) unlocked.push(id)
  }

  if (unlocked.length === 0) {
    player.tell(Text.gray("All spells are already locked."))
    return false
  }

  // Pick a random spell to lock
  let chosenId = unlocked[Math.floor(Math.random() * unlocked.length)]
  lockedSpells.push(chosenId)
  nbt.lockedSpells = lockedSpells   // write back
  item.nbt = nbt                    // commit to item

  // Mirror the lock flag on the spell entry for mod compatibility
  try {
    for (let i = 0; i < spellList.length; i++) {
      if (String(spellList[i].id) === String(chosenId)) {
        spellList[i].locked = 1
        break
      }
    }
    nbt.ISB_Spells.data = spellList
    item.nbt = nbt
  } catch (e) {}

  // Nicely formatted name for chat
  let displayName = chosenId.split(':').pop()
  if (displayName.indexOf('/') !== -1) displayName = displayName.split('/').pop()
  displayName = displayName.split('_').map(function (w) {
    return w.charAt(0).toUpperCase() + w.slice(1)
  }).join(' ')

  // Add lore line
  nbt.display = nbt.display || {}
  let lore = nbt.display.Lore || []
  lore.push(JSON.stringify([
    { text: 'Locked Spell: ', color: 'gold', italic: false },
    { text: displayName,    color: 'red',  italic: false }
  ]))
  nbt.display.Lore = lore
  item.nbt = nbt

  player.tell(Text.gold('[Locked Spell]: ').append(Text.red(displayName)))
  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
function removeSpellById(player, item, spellId) {
  if (!item || !item.nbt || !item.nbt.ISB_Spells || !item.nbt.ISB_Spells.data) {
    player.tell(Text.red("This item has no spells."))
    return false
  }

  let spellTag = item.nbt.ISB_Spells
  let spellList = spellTag.data
  if (!Array.isArray(spellList) || spellList.length === 0) {
    player.tell(Text.red("This item has no spells."))
    return false
  }

  // Also honor the lockedSpells list (in addition to per-entry flags)
  let lockedFromList = getLockedSpellsArray(item.nbt)
  if (lockedFromList.includes(String(spellId))) {
    player.tell(Text.red("That spell is locked and cannot be removed."))
    return false
  }

  let removed = false
  let newList = []

  for (let i = 0; i < spellList.length; i++) {
    let spell = spellList[i]
    let isLocked = spell.locked === 1 || spell.locked === 1.0 || spell.locked === true
    if (!removed && spell.id === spellId && !isLocked) {
      removed = true

      // Notify
      let name = spell.id.split(':').pop().split('_').map(w => w.charAt(0).toUpperCase() + w.slice(1)).join(' ')
      player.tell(Text.gray('[Removed Spell]: ').append(Text.red(name)))
      continue // Skip this one
    }
    newList.push(spell)
  }

  if (!removed) {
    player.tell(Text.red("No matching unlocked spell found to remove."))
    return false
  }

  spellTag.data = newList
  item.nbt.ISB_Spells = spellTag
  return true
}



function removeRandomSpell(player, item) {
  // Basic checks
  if (!item || !item.nbt || !item.nbt.ISB_Spells || !Array.isArray(item.nbt.ISB_Spells.data)) {
    player.tell(Text.red("That item holds no spells to modify."))
    return false
  }

  let nbt        = item.nbt
  let spellList  = nbt.ISB_Spells.data
  let lockedList = getLockedSpellsArray(nbt)

  if (spellList.length === 0) {
    player.tell(Text.red("That item holds no spells to modify."))
    return false
  }

  // Collect indices of spells NOT in lockedSpells
  let removable = []
  for (let i = 0; i < spellList.length; i++) {
    let id = String(spellList[i].id)
    if (lockedList.indexOf(id) === -1) removable.push(i)
  }

  if (removable.length === 0) {
    player.tell(Text.red("No unlocked spells to remove."))
    return false
  }

  // Choose random unlocked spell to remove
  let removeIndex  = removable[Math.floor(Math.random() * removable.length)]
  let removedSpell = spellList[removeIndex]

  // Build new spell list, omitting the removed one
  let newList = []
  for (let j = 0; j < spellList.length; j++) {
    if (j !== removeIndex) newList.push(spellList[j])
  }

  // Commit the rebuilt list
  nbt.ISB_Spells.data = newList
  nbt.ISB_Spells.maxSpells = Math.max(nbt.ISB_Spells.maxSpells || newList.length, newList.length)
  item.nbt = nbt

  // Notify player
  let removedName = removedSpell.id.split(':').pop().split('_').map(function (w) {
    return w.charAt(0).toUpperCase() + w.slice(1)
  }).join(' ')
  player.tell(Text.gray('[Removed Spell]: ').append(Text.red(removedName)))
  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
/**
 * swapSpell(player, item)
 * Removes one unlocked spell at random and replaces it with a new one.
 * Preconditions:
 *   • Item must contain at least THREE imbued spells.
 */
function swapSpell(player, item) {
  if (!item || item.empty) return false

  /* ── Spell‑count requirement ────────────────────────────────────────────── */
  let spellList = item.nbt?.ISB_Spells?.data || []
  if (spellList.length < 1) {
    player.tell(Text.red('Requires the item to have at least 1 imbued spell'))
    return false
  }

  /* ── Remove a random unlocked spell ─────────────────────────────────────── */
  let removed = removeRandomSpell(player, item)
  if (!removed) {
    player.tell(Text.red('No valid spell could be removed.'))
    return false
  }

  /* ── Add a new spell ────────────────────────────────────────────────────── */
  let added = imbueItem(player, item)
  if (!added) {
    player.tell(Text.red('Failed to imbue a new spell.'))
    return false
  }

  return true
}
//__________________________________________________________________________________________________________________________________________________________________________


// one‑time import (put near the other imports in this file)
let AffixRegistry = Java.loadClass('dev.shadowsoffire.apotheosis.adventure.affix.AffixRegistry')
let LootController = Java.loadClass("dev.shadowsoffire.apotheosis.adventure.loot.LootController");
let RarityRegistry = Java.loadClass("dev.shadowsoffire.apotheosis.adventure.loot.RarityRegistry");
let AffixHelper = Java.loadClass("dev.shadowsoffire.apotheosis.adventure.affix.AffixHelper");
let AffixType = Java.loadClass("dev.shadowsoffire.apotheosis.adventure.affix.AffixType");

/**
 * stack: ItemStack (weapon)
 * rarityId: Apotheosis rarity id, e.g. "apotheosis:rare" or "rare"
 * returns: [ "apotheosis:affix_id", ... ]  // all types merged
 */
global.getAvailableApothAffixes = function (stack) {
  let rarityId = String(global.apothRarityHelper(stack))
  let rarityHolder = RarityRegistry.byLegacyId(String(rarityId));
  if (!rarityHolder.isBound()) return [];

  let rarity = rarityHolder.get();

  // Existing affixes so we don’t suggest duplicates already on the item
  let affixMap = AffixHelper.getAffixes(stack); // java.util.Map
  let currentAffixes = affixMap.keySet();       // java.util.Set<DynamicHolder<? extends Affix>>

  let allIds = [];
  let types = AffixType.values(); // Java array of all AffixType values

  for (let i = 0; i < types.length; i++) {
    let type = types[i];

    let list = LootController.getAvailableAffixes(stack, rarity, currentAffixes, type);
    let it = list.iterator();
    while (it.hasNext()) {
      let holder = it.next();
      let id = String(holder.getId()); // e.g. "apotheosis:magical"
      if (isAffixBlacklisted(id)) {
        //console.log(`Blacklisted affix: ${id}`)
        continue
      
      };
      if (allIds.indexOf(id) === -1) allIds.push(id); // dedupe just in case
    }
  }

  return allIds;
};


/**
 * Ensures an item has an Apotheosis rarity.
 * - If it already has one, just returns it.
 * - If it has none, sets it to "common" and returns that.
 *
 * @param stack ItemStack
 * @return rarity id string, e.g. "apotheosis:common"
 */
global.apothRarityHelper = function (stack) {
  // Current rarity (may be unbound if none present)
  let holder = AffixHelper.getRarity(stack);
  if (!holder.isBound()) {
    // No rarity: default to "common"
    let commonHolder = RarityRegistry.byLegacyId("common"); // -> apotheosis:common
    if (!commonHolder.isBound()) {
      // Fallback: minimum configured rarity if "common" doesn’t exist for some reason
      commonHolder = RarityRegistry.getMinRarity();
    }

    let rarity = commonHolder.get();
    AffixHelper.setRarity(stack, rarity);
    let key = RarityRegistry.INSTANCE.getKey(rarity); // ResourceLocation
    return String(key);
  }

  // Has a rarity already, just return it
  let rarity = holder.get();
  let key = RarityRegistry.INSTANCE.getKey(rarity);
  return String(key);
};



global.blacklistedAffixStrings  = [
  'apotheotic_additions:ranged/attribute/crit_chance',
  'apotheotic_additions:ranged/attribute/crit_damage',
  'apotheosis_ascended:ranged/mob_effect/inaccurate',
  'apotheosis_ascended:ranged/mob_effect/precision',
  'apotheosis:heavy_weapon/attribute/annihilating',
  'apotheosis:heavy_weapon/attribute/decimating',
  'apotheosis:heavy_weapon/attribute/giant_slaying',
  'apotheosis_ascended:heavy_weapon/mob_effect/heartbreak',
  'apotheosis_ascended:heavy_weapon/mob_effect/vulnerability',
  'fallen_gems_affixes:heavy_weapon/attribute/annihilate',
  'shatter_spleen',
  'soulbound',
  //'apotheosis_ascended:sword/mob_effect/internal_bleeding',
  'apotheosis:sword/attribute/intricate',
  'apotheosis:sword/attribute/lacerating',
  'apotheosis_ascended:armor/mob_effect/perception',  
  'fallen_gems_affixes:armor/spell/',
  'fallen_gems_affixes:ranged/spell/',
  'fallen_gems_affixes:sword/spell/',
  'fallen_gems_affixes:heavy_weapon/spell/',
  'fallen_gems_affixes:soulbound',
  'apotheosis:armor/dmg_reduction/feathery',
  'apotheosis:telepathic',
  'flight'
]






// Helper: exact match, or prefix match if blacklist entry ends with "/"
function isAffixBlacklisted(id) {
  if (!global.blacklistedAffixStrings) return false;
  id = String(id);

  for (let i = 0; i < global.blacklistedAffixStrings.length; i++) {
    let pat = global.blacklistedAffixStrings[i];
    if (!pat) continue;
    pat = String(pat).trim();
    if (!pat) continue;

    // 1) Prefix mode: pattern ends with '/'
    if (pat.charAt(pat.length - 1) === '/') {
      if (id.indexOf(pat) === 0) return true;
      continue;
    }

    // 2) Short token: no ':' and no '/'
    if (pat.indexOf(':') === -1 && pat.indexOf('/') === -1) {
      if (id === pat || id.endsWith('/' + pat) || id.endsWith(':' + pat)) return true;
      continue;
    }

    // 3) Full ID match
    if (id === pat) return true;
  }

  return false;
}


let validOrbs = global.affix_orbs





//___________________________________________________________________________________________________________________________________________________________
// Orb of Socketing
  function getRandomSocketCount(chip) {
    return chip.min + Math.floor(Math.random() * (chip.max - chip.min + 1))
  }


// ─────────────────────────────────────────────────────────────────────────────
// Socket modifier — adds OR removes sockets (+count or –count)
//   • Positive count: behaves like the old addSockets (cap + 25 % failure + surge)
//   • Negative count: removes sockets, can’t go below 0, no failure/surge
//   • count = 0: does nothing and returns false
function modifySockets(player, item, count, ignoreCap) {
  if (ignoreCap === undefined) ignoreCap = false            // Rhino default
  if (!item || !item.nbt || count === 0) return false

  // ═══ Corruption lock ═══════════════════════════════════════════════════════
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false                      // helper already notifies player

  // ═══ Ensure affix_data path exists ═════════════════════════════════════════
  let tag = item.nbt
  if (tag.affix_data == null) tag.affix_data = {}
  let current = tag.affix_data.sockets ?? 0
    player.level.runCommandSilent(`/ftbquests change_progress ${player.username} reset 1E2AF8C0E3C458E2`)
    player.level.runCommandSilent(`/ftbquests change_progress ${player.username} complete 1E2AF8C0E3C458E2`)
    player.persistentData.putBoolean('orb_of_socketing', true)
  // ═══ Removal branch (count < 0) ════════════════════════════════════════════
  if (count < 0) {
    let newSockets = Math.max(current + count, 0)          // never < 0
    if (newSockets === current) {
      player.tell(Text.red('No sockets to remove.'))
      return false
    }

    tag.affix_data.sockets = newSockets
    item.nbt = tag

    player.tell(
      Text.gray('[Sockets removed]: ')
        .append(Text.red('' + current))
        .append(Text.white(' → '))
        .append(Text.green('' + newSockets))
    )
    return true
  }

  

  // ═══ Addition branch (count > 0) ══════════════════════════════════════════
  let corruptionMax = 10
  let rarity        = tag.affix_data.rarity
  let rarityLevel   = Object.keys(rarityCapsAndLevels).indexOf(rarity) || 0
  let maxSockets    = ignoreCap ? corruptionMax : 1 + rarityLevel

  if (current >= maxSockets) {
    player.tell(Text.red('This item already has the maximum number of sockets.'))
    return false
  }



  // Apply main addition
  let newSockets = Math.min(current + count, maxSockets)
  tag.affix_data.sockets = newSockets
  item.nbt = tag

  player.tell(
    Text.gray('[Sockets increased]: ')
      .append(Text.red('' + current))
      .append(Text.white(' → '))
      .append(Text.green('' + newSockets))
  )

  // One‑socket surge (same rules as before)
  if (surgeChance(player) && newSockets < maxSockets) {
    let surged = Math.min(newSockets + 1, maxSockets)
    tag.affix_data.sockets = surged
    item.nbt = tag

    player.tell(Text.of([
      Text.of('[Arcane Surge] → ').aqua(),
      Text.of('A bonus socket has been added!').gray()
    ]))
    player.tell(
      Text.gray('[Sockets increased]: ')
        .append(Text.red('' + newSockets))
        .append(Text.white(' → '))
        .append(Text.green('' + surged))
    )
  }

  return true
}


// -----------------------------------------------------------------------------
// Imbue one spell; may surge repeatedly until the item hits its spell cap

function normalizeSpellId(id) {
  if (id === null || id === undefined) return null
  let s = String(id).trim()
  if (!s) return null
  if (s.indexOf(':') === -1) s = 'irons_spellbooks:' + s
  return s.toLowerCase()
}

function imbueItem(player, item, dontTell) {

  if (!item || item.empty) return false
  let curio = isCurioItem(item)
  if (curio) {
    player.tell('§cCurios items cannot be imbued.')
    return false
  }
  /* ── Corruption gate ────────────────────────────────────────────────────── */
  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false          // helper already notifies player

  /* ── Ensure NBT scaffolding ─────────────────────────────────────────────── */
  let nbt = item.nbt || {}
  if (!nbt.affix_data) nbt.affix_data = {}
    player.level.runCommandSilent(`/ftbquests change_progress ${player.username} reset 1D416A491E4105EE`)
    player.level.runCommandSilent(`/ftbquests change_progress ${player.username} complete 1D416A491E4105EE`)
    player.persistentData.putBoolean('orb_of_imbuement', true)


  if (!nbt.ISB_Spells)      nbt.ISB_Spells      = {}
  if (!nbt.ISB_Spells.data) nbt.ISB_Spells.data = []

  let list  = nbt.ISB_Spells.data

  let ignoreCaps = player?.persistentData?.corruption_override_caps === true
  if (!ignoreCaps) {
    try {
      if (player && player.persistentData && typeof player.persistentData.getBoolean === 'function') {
        ignoreCaps = player.persistentData.getBoolean('corruption_override_caps') === true
      }
    } catch (e) {}
  }

  let rarity = nbt.affix_data.rarity ||
    (global.apothRarityHelper ? String(global.apothRarityHelper(item)) : 'apotheosis:common') ||
    'apotheosis:common'

  let caps = rarityCapsAndLevels && rarityCapsAndLevels[rarity]
  let spellCap = (caps && caps.maxSpells != null) ? caps.maxSpells : 1
  if (ignoreCaps) spellCap = Infinity

  let atCap = !ignoreCaps && list.length >= spellCap
  if (atCap) {
    if (!dontTell) {
      player.tell(Text.red("Upgrade this item's rarity to imbue more spells."))
    }
    return false
  }

  /* ── Pick a unique spell id ─────────────────────────────────────────────── */
  let existingIds = new Set()
  for (let i = 0; i < list.length; i++) {
    let norm = normalizeSpellId(list[i]?.id)
    if (norm) existingIds.add(norm)
  }
  let spellID = null
  for (let tries = 0; tries < 60; tries++) {
    let cand = normalizeSpellId(randomSpellID())
    if (!cand) continue
    if (!existingIds.has(cand)) { spellID = cand; break }
  }
  if (!spellID) {
    if (!dontTell) {
      player.tell(Text.red('No new spells are available to imbue on this item.'))
    }
    return false
  }

  /* ── Push spell & write NBT ─────────────────────────────────────────────── */
  let usedIdx = list.map(s => Number(s.index)).filter(i => !Number.isNaN(i))
  let idx = 0
  while (usedIdx.indexOf(idx) !== -1) idx++
  // level is 1 - maxLevel
  let rarityDef = rarityCapsAndLevels && rarityCapsAndLevels[rarity]
  let maxLevel = (rarityDef && rarityDef.maxAffixes != null) ? rarityDef.maxAffixes : 1
  let level = Math.min(maxLevel, Math.max(1, Math.floor(Math.random() * maxLevel)))
  list.push({ id: spellID, index: idx, level: level, locked: 1 })

  nbt.ISB_Spells.data      = list
  if (ignoreCaps) {
    let prev = Number(nbt.ISB_Spells.maxSpells)
    if (!Number.isFinite(prev)) prev = 0
    nbt.ISB_Spells.maxSpells = Math.max(prev, list.length)
  } else {
    nbt.ISB_Spells.maxSpells = Math.max(spellCap, list.length)
  }
  if (nbt.ISB_Spells.mustEquip  === undefined) nbt.ISB_Spells.mustEquip  = 0
  if (nbt.ISB_Spells.spellWheel === undefined) nbt.ISB_Spells.spellWheel = 1
  item.nbt = nbt

  /* ── Feedback ───────────────────────────────────────────────────────────── */
  let niceName = spellID.split(':').pop().split('/').pop()
    .split('_').map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(' ')
    if (!dontTell) {
      player.tell(Text.green('Imbued Spell → ').append(Text.blue(niceName)))
    }

  /* ── Surge chain — only if room remains ─────────────────────────────────── */
  if (surgeChance(player) && (ignoreCaps || list.length < spellCap)) {
    if (!dontTell) {
      player.tell(Text.of([
        Text.of('[Spell Surge] → ').aqua(),
        Text.of('Another spell is being etched...').gray()
      ]))
      Utils.server.scheduleInTicks(2, () => imbueItem(player, item))
    } else {
      Utils.server.scheduleInTicks(2, () => imbueItem(player, item, true))
    }
  }

  return true
}

//_________________________________________________________________________________________________________________________________________________________________________
// Orb of Sorcery
function upgradeImbuedSpell(player, item, dontTell) {
  if (!item || !item.nbt) return false

  let corruptionCheck = checkCorruption(player, item)
  if (!corruptionCheck) return false          // helper already notifies player

  let ignoreCap = player.persistentData.corruption_override_caps === true
  let maxLevel = ignoreCap ? 10 : 5

  // Ensure ISB_Spells and data exist
  let spellTag = item.nbt.ISB_Spells
  if (!spellTag || !spellTag.data || spellTag.data.length === 0) {
    if (!dontTell) {
      player.tell(Text.red("No imbued spells found on this item."))
    }
    return false
  }

  let spellList = spellTag.data

  let index = Math.floor(Math.random() * spellList.length)
  let spell = spellList[index]
  let baseLevel = parseInt(spell.level) || 1

  if (!ignoreCap && baseLevel >= maxLevel) {
    if (!dontTell) {
      player.tell(Text.red("That spell is already at max level."))
    }
    return false
  }

  spell.level = Math.min(baseLevel + 1, maxLevel)

  // Save updated spell data
  spellList[index] = spell
  spellTag.data = spellList
  item.nbt.ISB_Spells = spellTag

  let spellName = spell.id.split(':')[1]
  if (spellName.indexOf('/') !== -1) {
    spellName = spellName.split('/')[1]
  }
  spellName = spellName.split('_').map(function (w) {
    return w.charAt(0).toUpperCase() + w.slice(1)
  }).join(' ')

  let msg = Text.of([
    Text.yellow(spellName + ': '),
    Text.white('Level ' + baseLevel),
    Text.gray(' → '),
    Text.green('Level ' + spell.level)
  ])
  if (ignoreCap && spell.level > 5) {
    msg.append(Text.aqua(' [Chaos Infusion]').italic())
  }
  if (!dontTell) {
    player.tell(msg)
  }

  // Arcane Luck surge chance
  let surgeChance = player.hasEffect('kubejs:arcane_luck') ? 0.15 : 0.05
  if (Math.random() < surgeChance) {
    if (!dontTell) {
      player.tell(Text.lightPurple('[Spell Surge]: ').append(Text.gray('Another spell upgrade surges through the item...')))
      Utils.server.scheduleInTicks(2, function () {
        upgradeImbuedSpell(player, item)
      })
    } else {
      Utils.server.scheduleInTicks(2, function () {
        upgradeImbuedSpell(player, item, true)
      })
    }

  }

  return true
}
//__________________________________________________________________________________________________________________________________________________________________________
function surgeChance(player) {
  let baseChance = 0.05
  let luckBonus = player.hasEffect('kubejs:arcane_luck') ? 0.15 : 0.05
  return Math.random() < (baseChance + luckBonus)
}

// -----------------------------------------------------------------------------
// Spell lock helpers
// Read nbt.lockedSpells safely whether it's a JS array or a Java List.
function getLockedSpellsArray(nbt) {
  let out = []
  try {
    if (!nbt || !nbt.lockedSpells) return out
    let raw = nbt.lockedSpells
    // Rhino/Java lists generally expose a length and index accessor in this env
    let len = 0
    try { len = raw.length } catch (e) { len = 0 }
    if (typeof len === 'number' && len > 0) {
      for (let i = 0; i < len; i++) out.push(String(raw[i]))
      return out
    }
    // Fallback for plain JS arrays
    if (Array.isArray(raw)) {
      for (let i = 0; i < raw.length; i++) out.push(String(raw[i]))
    }
  } catch (e) {}
  return out
}





// KubeJS (server_scripts). Uses reflection because Apotheosis keeps these fields protected.


const AttributeAffix = Java.loadClass('dev.shadowsoffire.apotheosis.adventure.affix.AttributeAffix')
const PotionAffix = Java.loadClass('dev.shadowsoffire.apotheosis.adventure.affix.effect.PotionAffix')
const DamageReductionAffix = Java.loadClass('dev.shadowsoffire.apotheosis.adventure.affix.effect.DamageReductionAffix')

function getDeclaredField(obj, name) {
  let f = obj.getClass().getDeclaredField(name)
  f.setAccessible(true)
  return f.get(obj)
}

// returns null (unknown affix id) or an object describing what this affix is "about"
function getAffixDetails(affixId) {
    let ForgeRegistries = Java.loadClass('net.minecraftforge.registries.ForgeRegistries')
    let ResourceLocation = Java.loadClass('net.minecraft.resources.ResourceLocation')
  let holder = AffixRegistry.INSTANCE.holder(new ResourceLocation(String(affixId)))
  if (!holder.isBound()) return null

  let affix = holder.get()
  let affixClass = String(affix.getClass().getName())

  if (affix instanceof AttributeAffix) {
    let attr = getDeclaredField(affix, 'attribute')     // Attribute
    let op = getDeclaredField(affix, 'operation')       // AttributeModifier$Operation
    return {
      type: 'attribute',
      id: String(ForgeRegistries.ATTRIBUTES.getKey(attr)),
      operation: String(op.name()),
      affixClass: affixClass
    }
  }

  if (affix instanceof PotionAffix) {
    let effect = getDeclaredField(affix, 'effect')      // MobEffect
    let target = getDeclaredField(affix, 'target')      // PotionAffix$Target
    let targetId = String(getDeclaredField(target, 'id'))
    return {
      type: 'mob_effect',
      id: String(ForgeRegistries.MOB_EFFECTS.getKey(effect)),
      target: targetId,
      affixClass: affixClass
    }
  }

  if (affix instanceof DamageReductionAffix) {
    let drType = getDeclaredField(affix, 'type')        // DamageReductionAffix$DamageType
    return {
      type: 'damage_reduction',
      damage_type: String(drType.getId()), // physical|magic|fire|fall|explosion
      affixClass: affixClass
    }
  }

  return { type: 'other', affixClass: affixClass }
}

