


function detectCurios(player, item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional = CuriosApi.getCuriosHelper().getEquippedCurios(player);

  if (!optional.isPresent()) return false;

  let found = false;
  optional.ifPresent(handler => {
    for (let i = 0; i < handler.getSlots(); i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        found = true;
      }
    }
  });
  return found;
}

function isCurioItem(item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let helper = CuriosApi.getCuriosHelper();

  // Get all possible slot types the item is valid for
  let slotTypes = helper.getCurioTags(item); // this returns a java.util.Set

  // Convert to array and check if there's at least one slot
  return !slotTypes.isEmpty();
}


function removeCurio(player, item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional = CuriosApi.getCuriosHelper().getEquippedCurios(player);

  if (!optional.isPresent()) return false;

  let removed = false;
  optional.ifPresent(handler => {
    for (let i = 0; i < handler.getSlots(); i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        // Prefer extracting the full stack from the slot
        try {
          handler.extractItem(i, stack.getCount(), /*simulate=*/false);
        } catch (err) {
          // Fallback: force-clear the slot if extract isn't available
          let ItemStack = Java.type('net.minecraft.world.item.ItemStack');
          try {
            handler.setStackInSlot(i, ItemStack.EMPTY);
          } catch (err2) {
            // If both fail, leave removed=false
            return;
          }
        }
        removed = true;
        break; // remove only the first match
      }
    }
  });
  return removed;
}




function getCuriosItemNBT(player, item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional = CuriosApi.getCuriosHelper().getEquippedCurios(player);

  if (!optional.isPresent()) return null;

  let result = null;
  optional.ifPresent(handler => {
    for (let i = 0; i < handler.getSlots(); i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        result = stack.getNbt(); // ✅ Use getNbt() instead of getTag()
        break;
      }
    }
  });

  return result; // This will be null if no matching curio with NBT was found
}

/**
 * Returns true if the first equipped Curio matching `item` contains the given NBT key.
 *
 * @param player   The player to check.
 * @param item     The ItemStack or item ID to look for in Curio slots.
 * @param nbtKey   The NBT key to test for on that Curio’s CompoundTag.
 * @returns        Boolean indicating whether that Curio has a non-null NBT with the specified key.
 */
function hasCuriosNBT(player, item, nbtKey) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional  = CuriosApi.getCuriosHelper().getEquippedCurios(player);
  if (!optional.isPresent()) return false;

  let found = false;
  optional.ifPresent(handler => {
    for (let i = 0; i < handler.getSlots(); i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        let nbt = stack.getNbt();
        if (nbt != null && nbt.contains(nbtKey)) {
          found = true;
        }
        break;
      }
    }
  });

  return found;
}

/**
 * Returns true if the player has the specified Curio equipped AND
 * that Curio’s NBT.skills ListTag contains the given skillKey.
 * Includes debug messages via player.tell().
 *
 * @param player   The player to check.
 * @param item     The item ID (string) of the Curio to look for in Curio slots.
 * @param skillKey The key of the infused skill to test for (e.g. 'ice_beam').
 * @returns        true only if the player has the Curio AND its NBT.skills contains skillKey.
 */
function hasCurioWithNBT(player, item, skillKey) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional  = CuriosApi.getCuriosHelper().getEquippedCurios(player);
  
  if (!optional.isPresent()) {
    //player.tell("Debug: No Curios handler present.");
    return false;
  }

  let foundMatch = false;
  optional.ifPresent(handler => {
    let slots = handler.getSlots();
    //player.tell("Debug: Checking " + slots + " Curio slots for item " + item + " and skillKey " + skillKey + ".");

    for (let i = 0; i < slots; i++) {
      let stack = handler.getStackInSlot(i);
      if (stack.isEmpty()) {
        //player.tell("Debug: Slot " + i + " is empty.");
        continue;
      }
      if (!stack.is(item)) {
        //player.tell("Debug: Slot " + i + " contains a different item: " + stack.item.id);
        continue;
      }

      // Found the correct Curio item
      //player.tell("Debug: Found " + item + " in slot " + i + ". Checking NBT...");

      let nbt = stack.getNbt();
      if (!nbt) {
        //player.tell("Debug: No NBT found on this item.");
        break;
      }

      // Check for a ListTag named "skills"
      if (!nbt.contains("skills", 9)) {
        //player.tell("Debug: NBT does not contain a ListTag called 'skills'.");
        break;
      }

      // Read the ListTag (type 8 = strings)
      let rawList = nbt.getList("skills", 8);
      let listSize = rawList.size();
      //player.tell("Debug: NBT.skills ListTag has size " + listSize + ".");

      // Iterate through the ListTag to look for skillKey
      for (let idx = 0; idx < listSize; idx++) {
        let entry = rawList.getString(idx);
        //player.tell("Debug: NBT.skills[" + idx + "] = " + entry);
        if (entry === skillKey) {
          //player.tell("Debug: SkillKey '" + skillKey + "' found in NBT.skills!");
          foundMatch = true;
          break;
        }
      }
      // No need to check further slots once we've found—or at least checked—our Curio
      break;
    }
  });

  if (!foundMatch) {
    //player.tell("Debug: hasCurioWithNBT returning false for " + item + " / " + skillKey + ".");
  }
  return foundMatch;
}










function countCurios(player, item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional = CuriosApi.getCuriosHelper().getEquippedCurios(player);

  if (!optional.isPresent()) return 0;

  let count = 0;
  optional.ifPresent(handler => {
    for (let i = 0; i < handler.getSlots(); i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        count++;
      }
    }
  });
  return count;
}

let possible_affinities = [
  'blood',
  'fire',
  'ice',
  'nature',
  'ender',
  'lightning',
  'holy',
  'evocation',
  'aqua',
];

function handleAffinityGems(player) {
let affinity_gems = [];
possible_affinities.forEach(gem => {
  // Add one entry per lesser gem found
  let lesser_count = countCurios(player, `kubejs:${gem}_affinity_gem`);
  for (let i = 0; i < lesser_count; i++) {
    affinity_gems.push(gem);
  }

  // Add one entry per superior gem found
  let superior_count = countCurios(player, `kubejs:superior_${gem}_affinity_gem`);
  for (let i = 0; i < superior_count; i++) {
    affinity_gems.push(`superior_${gem}`);
  }

  // Add one entry per exalted gem found
  let exalted_count = countCurios(player, `kubejs:exalted_${gem}_affinity_gem`);
  for (let i = 0; i < exalted_count; i++) {
    affinity_gems.push(`exalted_${gem}`);
  }
});
return affinity_gems;
}

PlayerEvents.spellOnCast(event => {
  let player = event.player;
  let affinity_gems = handleAffinityGems(player);
 // Utils.server.tell(affinity_gems);
  if (affinity_gems.length == 0) return;

  let spell_school = event.schoolType.id.toString().split(':')[1];
  let affinity_school = null;
  let added_levels = 0;
  let mana_cost = event.manaCost;
  let refunded_mana = 0;

  affinity_gems.forEach(gem => {
    if (gem.includes('exalted')) {
      affinity_school = gem.split('_')[1];
      if (spell_school == affinity_school) {
        added_levels += 2;
        refunded_mana += (mana_cost * 0.25);
      }
    } else if (gem.includes('superior')) {
      affinity_school = gem.split('_')[1];
      if (spell_school == affinity_school) {
        added_levels += 1;
        refunded_mana += (mana_cost * 0.15);
      }
    } else {
      if (spell_school == gem) {
        refunded_mana += (mana_cost * 0.1);
      }
    }
  });
  //tell(event.spellLevel + added_levels)
  event.setSpellLevel(event.spellLevel + added_levels);
  event.setManaCost(event.manaCost - refunded_mana);
});


/**
 * Returns the first equipped Curio ItemStack matching `item` or null if none.
 *
 * @param player The player to inspect.
 * @param item   Item identifier or object usable with stack.is(item).
 * @returns      The matching ItemStack, or null if not found.
 */
function getCurioItem(player, item) {
  let CuriosApi = Java.loadClass("top.theillusivec4.curios.api.CuriosApi");
  let optional  = CuriosApi.getCuriosHelper().getEquippedCurios(player);

  if (!optional.isPresent()) return null;

  let result = null;
  optional.ifPresent(handler => {
    let slots = handler.getSlots();
    for (let i = 0; i < slots; i++) {
      let stack = handler.getStackInSlot(i);
      if (!stack.isEmpty() && stack.is(item)) {
        result = stack; // Return the first match
        break;
      }
    }
  });

  return result;
}
