// server_scripts/JSfunctions/applyEffect.js

global.addEffect = function (entity, effectId, duration, amplifier) {
  try {
    // Prefer KubeJS API (no reflection/class filter issues)
    let dur = Number(duration) | 0
    let amp = Number(amplifier) | 0
    if (!entity || !entity.potionEffects) {
      //console.log(`[addEffect] Invalid entity or missing potionEffects: ${entity}`)
      return false
    }
    entity.potionEffects.add(String(effectId), dur, amp, true, true)
    return true
  } catch (primaryErr) {
    // Fallback to server command as last resort (seconds-based)
    try {
      let secs = Math.max(1, Math.ceil((Number(duration) || 0) / 20))
      Utils.server.runCommandSilent(`/effect give ${entity.uuid} ${effectId} ${secs} ${amplifier | 0} true`)
      return true
    } catch (fallbackErr) {
      //console.log(`[addEffect] Failed to apply ${effectId}: ${primaryErr}; fallback: ${fallbackErr}`)
      return false
    }
  }
}


/**
 * 
 * @param {*} entity The entity to apply the effect to.
 * @param {*} effect The effect to apply (e.g., 'minecraft:slowness').
 * @param {*} duration Duration. Must be in Ticks or 'infinite'.
 * @param {*} amplifier OPTIONAL: (default: 0) Amplifier level of the effect (0 for level 1, 1 for level 2, etc.)
 * @param {*} logSource OPTIONAL: Generally the skill or item thats applying the effect, used for logging and debugging.
 * @param {*} sourcePlayer OPTIONAL: The player who is applying the effect, used for logging and debugging.
 * @returns 
 */

function applyEffect(entity, effect, duration, amplifier, logSource, sourcePlayer, hideParticles) {
    if (!entity || !effect || !duration) return;
    if (amplifier === undefined || amplifier === null) amplifier = 0;
    if (logSource === undefined || logSource === null) logSource = 'Unknown';
    if (sourcePlayer === undefined) sourcePlayer = null;
    // Apply the effect to the entity
    if (duration != 'infinite') {
        //global.addEffect(entity, effect, duration, amplifier)
        try {
          let potionEffects = entity.potionEffects
          if (!potionEffects) {
            //console.log(`No potion effect found, falling back to command version`)
            //throw new Error('potionEffects unavailable');
          } else {
            if (hideParticles && hideParticles == true) {
              entity.potionEffects.add(effect, duration, amplifier, false, false);
            } else {
              entity.potionEffects.add(effect, duration, amplifier, true, true);
            }
          }
          
        } catch (err) {
            console.warn(`[applyEffect] Direct add failed: ${err?.message || err}`);
            let secs = Math.max(1, Math.ceil((Number(duration) || 0) / 20))
            if (hideParticles && hideParticles == true) {
              Utils.server.runCommandSilent(`execute as ${entity} run effect give @s ${effect} ${secs} ${((amplifier|0))} true`)
            } else {
              Utils.server.runCommandSilent(`execute as ${entity} run effect give @s ${effect} ${secs} ${((amplifier|0))} false`)
            }
            
        }
        
    } else {
        Utils.server.runCommandSilent(`/effect give ${entity.uuid} ${effect} infinite ${amplifier} false`)
    }
    //console.log(`Applied effect ${effect} to entity ${entity.uuid} for ${duration} ticks with amplifier ${amplifier}. Source: ${logSource}, Player: ${sourcePlayer ? sourcePlayer.username : 'N/A'}`);
}




/**
 * 
 * @param {*} entity The entity to apply the effect to.
 * @param {*} effect The effect to apply (e.g., 'minecraft:slowness').
 * @param {*} duration Duration. Must be in Ticks or 'infinite'.
 * @param {*} amplifier OPTIONAL: (default: 0) Amplifier level of the effect (0 for level 1, 1 for level 2, etc.)
 * @param {*} logSource OPTIONAL: Generally the skill or item thats applying the effect, used for logging and debugging.
 * @param {*} sourcePlayer OPTIONAL: The player who is applying the effect, used for logging and debugging.
 * @param {*} ampCap OPTIONAL: Maximum amplifier level. If the new amplifier exceeds this, it will be capped. Default: no cap.
 * @param {*} durCap OPTIONAL: Maximum duration. If the new duration exceeds this, it will be capped. Default: no cap. MUST BE IN TICKS
 * @returns 
 */

function applyStackingEffect(entity, effect, duration, amplifier, logSource, sourcePlayer, ampCap, durCap) {
    if (!entity || !effect || !duration) return;
    if (amplifier === undefined || amplifier === null) amplifier = 0;
    if (logSource === undefined || logSource === null) logSource = 'Unknown';
    if (sourcePlayer === undefined) sourcePlayer = null;
    if (duration != 'infinite') {
        let currentDuration = 0
        let currentAmplifier = 0
        if (entity.potionEffects.isActive(effect)) {
            currentDuration = entity.potionEffects.getActive(effect).duration
            currentAmplifier = entity.potionEffects.getActive(effect).amplifier
            duration += currentDuration
            amplifier += currentAmplifier
        } else {
            amplifier = currentAmplifier
        }
        //tell(duration)
        //tell(amplifier)
        // Apply the effect to the entity
        if (ampCap !== undefined && ampCap !== null) {
            if (amplifier > ampCap) amplifier = ampCap
        }
        if (durCap !== undefined && durCap !== null) {
            if (duration > durCap) duration = durCap
        }
        Utils.server.runCommandSilent(`/effect clear ${entity.uuid} ${effect}`)
        entity.potionEffects.add(effect, duration, amplifier, true, true);
    } else {
        let newAmp = entity.potionEffects.getActive(effect).amplifier + 1
        //tell(newAmp)
        Utils.server.runCommandSilent(`/effect give ${entity.uuid} ${effect} infinite ${newAmp} false`)
    }
    //console.log(`Applied effect ${effect} to entity ${entity.uuid} for ${duration} ticks with amplifier ${amplifier}. Source: ${logSource}, Player: ${sourcePlayer ? sourcePlayer.username : 'N/A'}`);
}



/**
 * 
 * @param {*} player Player who is casting the AOE effect
 * @param {*} range Range (Radius) of the AOE effect
 * @param {*} effect Effect to apply to the entities
 * @param {*} duration Duration of the effect in Ticks
 * @param {*} amplifier  Amplifier of the effect
 * @param {*} totalEnemies Total number of enemies to apply the effect to
 */
function applyAoeEffect(player, range, effect, duration, amplifier, totalEnemies, glowing, glowColor) {
    Utils.server.scheduleInTicks(1, e => {
        let box = AABB.of(player.x+range, player.y+range, player.z+range, player.x-range, player.y-range, player.z-range)
        let dim = player.level
        let entitiesWithin = dim.getEntitiesWithin(box).filter(ent => ent.isMonster() && ent.health > 0 && !ent.potionEffects.isActive(effect) && !isAlly(player, ent))
        let counter = 0
        entitiesWithin.forEach(ent => {
            if (counter < totalEnemies) {
                counter += 1
                if (effect == 'burning') {
                    ent.remainingFireTicks = duration
                } else {
                    applyEffect(ent, effect, duration, amplifier)
                    //Utils.server.runCommandSilent(`/effect give ${ent.uuid} ${effect} ${duration} ${amplifier} false`)
                }
                if (glowing) {
                    global.applyGlow(ent, glowColor, duration)
                    //Utils.server.runCommandSilent(`/effect give ${ent.uuid} minecraft:glowing ${duration} 0 true`)
                }
            }
        })
    })
}

/**
 * 
 * @param {*} player Player who is casting the AOE effect
 * @param {*} range Range (Radius) of the AOE effect
 * @param {*} effect Effect to apply to the entities
 * @param {*} duration Duration of the effect in Ticks
 * @param {*} amplifier  Amplifier of the effect
 * @param {*} totalEnemies Total number of enemies to apply the effect to
 */
function applyAoeEffectOnEntity(player, entity, range, effect, duration, amplifier, totalEnemies, glowing) {
        Utils.server.scheduleInTicks(1, e => {
        let box = AABB.of(entity.x+range, entity.y+range, entity.z+range, entity.x-range, entity.y-range, entity.z-range)
        let dim = player.level
        let entitiesWithin = dim.getEntitiesWithin(box).filter(ent => ent.isMonster() && ent.isDeadOrDying() == false && !ent.potionEffects.isActive(effect) && !isAlly(player, ent))
        let counter = 0
        entitiesWithin.forEach(ent => {
            if (counter < totalEnemies) {
                counter += 1
                if (effect == 'burning') {
                    ent.remainingFireTicks = duration * 20
                } else {
                    applyEffect(ent, effect, duration, amplifier)
                    //Utils.server.runCommandSilent(`/effect give ${ent.uuid} ${effect} ${duration} ${amplifier} false`)
                }
                if (glowing) {
                    Utils.server.runCommandSilent(`/effect give ${ent.uuid} minecraft:glowing ${duration} 0 true`)
                }
            }
        })
    })
}