/**
 * Optimized & Refactored Code
 * All original functionality is preserved.
 *
 * Note: This script assumes helper functions like detectCurios, Utils.server.runCommandSilent,
 *       AABB.of, etc. are already defined.
 */

// ─── HELPER FUNCTIONS FOR ITEM EFFECTS ──────────────────────────────────────────

// ----- Cursed Necklace Effects -----
function handleCursedNecklaceAttack(player) {
    if (!detectCurios(player, 'kubejs:cursed_necklace')) return;
    if (Math.random() > 0.5) return;
    player.potionEffects.add('minecraft:strength', 40, 1);
    player.potionEffects.add('regeneration', 40, 1);
  }
  
  function handleCursedNecklaceHurt(player) {
    if (!detectCurios(player, 'kubejs:cursed_necklace')) return;
    if (Math.random() > 0.5) return;
    player.potionEffects.add('minecraft:blindness', 40, 1);
    player.potionEffects.add('minecraft:slowness', 40, 0);
  }
  
  // ----- Phantoms Twilight -----
  function handlePhantomsTwilight(event) {
    let player = event.entity;
    if (!detectCurios(player, 'kubejs:phantoms_twilight')) return;
    if (!event.source.actual || !event.source.actual.alive) return;
    let orderLevel = 3
    if (Math.random() > 0.05) return;
  
    // Apply buffs to the player
    player.potionEffects.add('irons_spellbooks:true_invisibility', 100, 0, true, true);
    player.potionEffects.add('minecraft:strength', 100, 1, true, true);
    player.potionEffects.add('irons_spellbooks:true_invisibility', 100, 0, true, true);
  
    // Find and affect nearby entities
    let box = AABB.of(
      player.x + 15 , player.y + 15 , player.z + 15 ,
      player.x - 15 , player.y - 15 , player.z - 15 
    );
    let entitiesWithin = event.level.getEntitiesWithin(box);
    entitiesWithin.forEach(ent => {
      if (!ent.alive || ent === player) return;
      applyEffect(ent, 'minecraft:glowing', 100 , 0);
      ent.knockback(3, (player.x - ent.x), (player.z - ent.z));
    });
  }
  
  // ----- Empyrean Scarab -----
  function handleEmpyreanScarabAttack(event, player) {
    // Only trigger when casting a spell from irons_spellbooks
    let s = isSpellDamageSource(event.source);
    if (!s) return;
    if (!detectCurios(player, 'kubejs:empyrean_scarab')) return;
    if (Math.random() * 100 > 50) return;
    player.magicData.addMana(player.persistentData.last_mana_cost);
  }
  
  // ----- Phalanx Ring -----
  function handlePhalanxRingTarget(event) {
    let player = event.entity;
    if (!detectCurios(player, 'kubejs:phalanx_ring')) return;
    if (!player.tags.contains('phalanx_ring_active')) return;
    if (Math.random() > 0.1) return;
    if (event.source.actual) {
      let uuid = event.source.actual.uuid;
      Utils.server.runCommandSilent(`/effect give ${uuid} alexscaves:stunned 40 0 true`);
    }
    player.tags.remove('phalanx_ring_active');
    Utils.server.scheduleInTicks(400, () => {
      player.tags.add('phalanx_ring_active');
    });
  }
  
  // ----- Specter's Boon -----
  // This function handles the target (player taking fatal damage)
  function handleSpectersBoonTarget(event) {
    let player = event.entity;
    if (!detectCurios(player, 'kubejs:specters_boon')) return;
    // If fatal damage and not already under boon effects…
    if (
      event.damage >= player.health &&
      event.source.actual && event.source.actual.alive &&
      !player.tags.contains('specters_boon')
    ) {
      Utils.server.runCommandSilent(`/sanity set ${player.username} 0`);
      player.persistentData.specters_boon = event.source.actual.uuid.toString();
      applyEffect(event.source.actual, 'minecraft:glowing', 200, 0);
      player.health += player.maxHealth;
      player.tags.add('specters_boon');
      player.potionEffects.add('minecraft:speed', 200, 4);
      Utils.server.scheduleInTicks(200, () => {
        player.tags.remove('specters_boon');
        player.persistentData.specters_boon = '';
        player.kill();
      });
    }
    // While under the boon effect, cancel further damage.
    if (player.tags.contains('specters_boon')) {
      event.cancel();
    }
  }
  
  // This function prevents a Specter's Boon–affected attacker from damaging any mob
  // other than the one that killed the player.
  function handleSpectersBoonAttacker(event) {
    let attacker = event.source.actual;
    if (!attacker.tags.contains('specters_boon')) return;
    if (event.entity.uuid.toString() !== attacker.persistentData.specters_boon) {
      event.cancel();
    }
  }
  
  // ----- Tears of the Aether -----
  function handleTearsOfTheAether(event) {
    let player = event.entity;
    if (!detectCurios(player, 'kubejs:tears_of_the_aether')) return;
    if (!event.source.actual || !event.source.actual.alive) return;
    if (Math.random() > 0.05) return;
    let box = AABB.of(
      player.x + 15, player.y + 15, player.z + 15,
      player.x - 15, player.y - 15, player.z - 15
    );
    let entitiesWithin = event.level.getEntitiesWithin(box);
    entitiesWithin.forEach(ent => {
      if (!ent.alive || ent === player) return;
      let uuid = ent.uuid;
      applyEffect(ent, 'minecells:stunned', 60, 0, 'Tears of the Aether');
      applyEffect(ent, 'cofh_core:sundered', 60, 0, 'Tears of the Aether');
      applyEffect(ent, 'minecraft:glowing', 60, 0, 'Tears of the Aether');
    });
  }
  
  // ----- Sufficiently Advanced Technology -----
  // When hit by a potato cannon, place a hollow cube of glass and spawn TNT.
  function handleSufficientlyAdvancedTechnology(event) {
    if (!event.source || !event.source.toString().includes('create.potato_cannon')) return;
    if (!event.source.immediate || !event.source.immediate.isPlayer()) return;
    let player = event.source.immediate.player;
    if (!detectCurios(player, 'kubejs:sufficiently_advanced_technology')) return;
    let target = event.entity;
    let target_x = Math.floor(target.x);
    let target_y = Math.floor(target.y);
    let target_z = Math.floor(target.z);
    Utils.server.runCommandSilent(
      `/fill ${target_x-2} ${target_y-2} ${target_z-2} ${target_x+2} ${target_y+2} ${target_z+2} minecraft:glass hollow`
    );
    Utils.server.scheduleInTicks(20, () => {
      [
        { x: target_x + 1, y: target_y, z: target_z },
        { x: target_x - 1, y: target_y, z: target_z },
        { x: target_x, y: target_y, z: target_z + 1 },
        { x: target_x, y: target_y, z: target_z - 1 }
      ].forEach(pos => {
        let tnt = event.level.createEntity('minecraft:tnt');
        tnt.x = pos.x;
        tnt.y = pos.y;
        tnt.z = pos.z;
        tnt.spawn();
      });
    });
    player.cooldowns.addCooldown('create:potato_cannon', 300);
  }
  
  // ─── ENTITY HURT EVENT (COMBINED) ───────────────────────────────────────────────
  EntityEvents.hurt(event => {
    if (!event.source.player) return;
      let attacker = event.source.player;
      handleCursedNecklaceAttack(attacker);
      handleEmpyreanScarabAttack(event, attacker);
      handleSpectersBoonAttacker(event);
      handleSufficientlyAdvancedTechnology(event);
  });


EntityEvents.hurt('player', event => {
    let player = event.entity;
    handleCursedNecklaceHurt(player);
    handlePhantomsTwilight(event);
    handlePhalanxRingTarget(event);
    handleSpectersBoonTarget(event);
    handleTearsOfTheAether(event);
})



  
  // ─── PLAYER & ITEM EVENTS ───────────────────────────────────────────────────────
  
  // Set default persistent data values on player login.
  PlayerEvents.loggedIn(event => {
    let player = event.player;
    if (!player.persistentData.get('last_mana_cost')) {
      player.persistentData.put('last_mana_cost', 0);
    }
    if (!player.persistentData.get('specters_boon')) {
      player.persistentData.putString('specters_boon', '');
    }
  });
  
  // Update last mana cost when a spell is cast.
  PlayerEvents.spellOnCast(event => {
    event.player.persistentData.put('last_mana_cost', event.manaCost);
  });
  
  // Phalanx Ring – on respawn, add (or remove) the active tag based on the player’s curios.
  PlayerEvents.respawned(event => {
    let hasPhalanxRing = event.player.nbt.toString().includes("kubejs:phalanx_ring");
    if (hasPhalanxRing) {
      event.player.tags.add('phalanx_ring_active');
    } else {
      event.player.tags.remove('phalanx_ring_active');
    }
  });
  
  // When the Phalanx Ring is picked up, add its active tag.
  ItemEvents.pickedUp('kubejs:phalanx_ring', event => {
    event.player.tags.add('phalanx_ring_active');
  });
  
  // ----- Twilight Codex -----
  // When right-clicked with shift, clear blocks, spawn a temporary black hole,
  // and then summon a nuclear bomb.
  ItemEvents.rightClicked('kubejs:twilight_codex', event => {
    if (!event.player.shiftKeyDown) return;
    if (event.player.cooldowns.isOnCooldown('kubejs:twilight_codex')) return;
    event.player.cooldowns.addCooldown('kubejs:twilight_codex', 12000);
    for (let counter = 0; counter < 30; counter++) {
      Utils.server.runCommandSilent(
        `/setblock ${event.player.x} ${event.player.y + counter} ${event.player.z} minecraft:air replace`
      );
    }
    let blackhole = event.level.createEntity('irons_spellbooks:black_hole');
    blackhole.x = event.player.x;
    blackhole.y = event.player.y - 5;
    blackhole.z = event.player.z;
    blackhole.spawn();
    Utils.server.scheduleInTicks(50, () => {
      blackhole.kill();
    });
    Utils.server.scheduleInTicks(60, () => {
      Utils.server.runCommandSilent(
        `/execute at ${event.player.username} run summon alexscaves:nuclear_bomb ${event.player.x} ${event.player.y + 30} ${event.player.z} {Motion:[0.0,-5.0,0.0]}`
      );
    });
  });
  
  // ----- Potato Cannon (Indistinguishable Magic) -----
  // When left-clicked, fire a TNT “projectile” and (if not in creative)
  // remove one TNT from the inventory.
  ItemEvents.firstLeftClicked('create:potato_cannon', event => {
    if (event.player.cooldowns.isOnCooldown('create:potato_cannon')) return;
    let player = event.player;
    if (!detectCurios(player, 'kubejs:indistinguishable_magic')) return;
    if (!player.creative) {
      let tntCount = player.inventory.countItem('minecraft:tnt');
      let slot = player.inventory.findSlotMatchingItem('minecraft:tnt');
      player.inventory.extractItem(slot, 1, false);
      if (tntCount < 1) return;
    }
    let crossbowOffset = 2.5;
    let heightOffset = -0.5;
    let tntEntity = event.level.createEntity('minecraft:tnt');
    tntEntity.tags.add(`Owner:${event.player.username}`);
    let x_rad = event.player.pitch * JavaMath.PI / 180;
    let y_rad = event.player.yaw * JavaMath.PI / 180;
    let dirV = {
      x: -Math.sin(y_rad) * Math.cos(x_rad),
      y: -Math.sin(x_rad),
      z: Math.cos(y_rad) * Math.cos(x_rad)
    };
    tntEntity.addMotion(dirV.x * 6, dirV.y * 6, dirV.z * 6);
    let pos = {
      x: player.x - Math.sin(y_rad) * crossbowOffset,
      y: player.y + player.getEyeHeight() + heightOffset,
      z: player.z + Math.cos(y_rad) * crossbowOffset
    };
    let fuse = player.crouching ? 2 : 6;
    tntEntity.mergeNbt({ Fuse: fuse });
    tntEntity.setPosition(pos.x, pos.y, pos.z);
    tntEntity.spawn();
    tntEntity.playSound('blocky_siege:cannon_fire');
    player.cooldowns.addCooldown('create:potato_cannon', 5);
  });
  