/*
 * Decompiled with CFR 0.152.
 */
package com.sts15.enderdrives.db;

import appeng.api.stacks.AEItemKey;
import com.sts15.enderdrives.config.serverConfig;
import com.sts15.enderdrives.db.AEKey;
import com.sts15.enderdrives.db.AEKeyCacheEntry;
import com.sts15.enderdrives.db.StoredEntry;
import com.sts15.enderdrives.inventory.EnderDiskInventory;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinTask;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.storage.LevelResource;
import net.neoforged.neoforge.server.ServerLifecycleHooks;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class EnderDBManager {
    private static final Logger LOGGER = LogManager.getLogger((String)"EnderDrives");
    public static final ConcurrentSkipListMap<AEKey, StoredEntry> dbMap = new ConcurrentSkipListMap();
    private static final BlockingQueue<byte[]> walQueue = new LinkedBlockingQueue<byte[]>();
    private static final ConcurrentHashMap<String, CachedCount> itemCountCache = new ConcurrentHashMap();
    private static File dbFile;
    private static File currentWAL;
    private static FileOutputStream walFileStream;
    private static DataOutputStream walWriter;
    private static final Object commitLock;
    public static volatile boolean running;
    public static volatile boolean dirty;
    private static Thread commitThread;
    private static final AtomicLong totalItemsWritten;
    private static final AtomicLong totalCommits;
    private static final ForkJoinPool SHARED_PARALLEL_POOL;
    private static final int MERGE_BUFFER_THRESHOLD;
    private static final long MIN_WAL_COMMIT_MS;
    private static final long MAX_WAL_COMMIT_MS;
    private static final long MIN_DB_COMMIT_MS;
    private static final long MAX_DB_COMMIT_MS;
    private static final boolean DEBUG_LOG;
    private static long lastWalCommitTime;
    private static long lastDbCommitTime;

    public static void init() {
        try {
            Path worldDir = ServerLifecycleHooks.getCurrentServer().getWorldPath(LevelResource.ROOT).resolve("data").resolve("enderdrives");
            Files.createDirectories(worldDir, new FileAttribute[0]);
            dbFile = worldDir.resolve("enderdrives.bin").toFile();
            currentWAL = worldDir.resolve("enderdrives.wal").toFile();
            EnderDBManager.migrateOldRecords();
            EnderDBManager.loadDatabase();
            EnderDBManager.replayWALs();
            EnderDBManager.openWALStream();
            EnderDBManager.startBackgroundCommit();
            Runtime.getRuntime().addShutdownHook(new Thread(EnderDBManager::shutdown));
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void clearRAMCaches() {
        Object object = commitLock;
        synchronized (object) {
            dbMap.clear();
        }
        itemCountCache.clear();
        EnderDBManager.log("[clearRAMCaches] RAM caches cleared successfully.", new Object[0]);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void shutdown() {
        running = false;
        Object object = commitLock;
        synchronized (object) {
            EnderDBManager.flushWALQueue();
            if (!dbMap.isEmpty()) {
                EnderDBManager.commitDatabase();
            }
            EnderDBManager.truncateCurrentWAL();
            try {
                EnderDBManager.closeWALStream();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            EnderDBManager.clearRAMCaches();
        }
        if (commitThread != null) {
            try {
                commitThread.join(500L);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            commitThread = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void saveItem(String scopePrefix, int freq, byte[] itemBytes, long deltaCount) {
        AEKey key = new AEKey(scopePrefix, freq, itemBytes);
        Object object = commitLock;
        synchronized (object) {
            dbMap.compute(key, (k, existing) -> {
                long newCount = (existing == null ? 0L : existing.count()) + deltaCount;
                if (newCount <= 0L) {
                    return null;
                }
                AEItemKey aeKey = null;
                try {
                    ItemStack s = EnderDiskInventory.deserializeItemStackFromBytes(itemBytes);
                    if (!s.isEmpty()) {
                        aeKey = AEItemKey.of((ItemStack)s);
                    }
                }
                catch (Exception exception) {
                    // empty catch block
                }
                return new StoredEntry(newCount, aeKey);
            });
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream();
                 DataOutputStream dos = new DataOutputStream(baos);){
                dos.writeUTF(scopePrefix);
                dos.writeInt(freq);
                dos.writeInt(itemBytes.length);
                dos.write(itemBytes);
                dos.writeLong(deltaCount);
                walQueue.add(baos.toByteArray());
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            dirty = true;
        }
    }

    public static long getItemCount(String scopePrefix, int freq, byte[] keyBytes) {
        AEKey key = new AEKey(scopePrefix, freq, keyBytes);
        return dbMap.getOrDefault(key, new StoredEntry(0L, null)).count();
    }

    public static int getTypeCountInclusive(String scope, int freq) {
        AEKey from = new AEKey(scope, freq, new byte[0]);
        AEKey to = new AEKey(scope, freq + 1, new byte[0]);
        return dbMap.subMap((Object)from, true, (Object)to, false).size();
    }

    public static long getTotalItemCountInclusive(String scope, int freq) {
        long total = 0L;
        AEKey from = new AEKey(scope, freq, new byte[0]);
        AEKey to = new AEKey(scope, freq + 1, new byte[0]);
        for (Map.Entry e : dbMap.subMap((Object)from, true, (Object)to, false).entrySet()) {
            total += ((StoredEntry)e.getValue()).count();
        }
        return total;
    }

    public static void clearFrequency(String scopePrefix, int frequency) {
        AEKey from = new AEKey(scopePrefix, frequency, new byte[0]);
        AEKey to = new AEKey(scopePrefix, frequency + 1, new byte[0]);
        NavigableMap sub = dbMap.subMap((Object)from, true, (Object)to, false);
        int removed = sub.size();
        sub.clear();
        EnderDBManager.log("[clearFrequency] Cleared frequency %d for scope %s (%d entries)", frequency, scopePrefix, removed);
    }

    public static int getTypeCount(String scopePrefix, int freq) {
        AEKey from = new AEKey(scopePrefix, freq, new byte[0]);
        AEKey to = new AEKey(scopePrefix, freq + 1, new byte[0]);
        return dbMap.subMap((Object)from, true, (Object)to, false).size();
    }

    public static List<AEKeyCacheEntry> queryItemsByFrequency(String scopePrefix, int freq) {
        AEKey lo = new AEKey(scopePrefix, freq, new byte[0]);
        AEKey hi = new AEKey(scopePrefix, freq + 1, new byte[0]);
        NavigableMap committed = dbMap.subMap((Object)lo, true, (Object)hi, false);
        ArrayList<AEKeyCacheEntry> result = new ArrayList<AEKeyCacheEntry>();
        for (Map.Entry e : committed.entrySet()) {
            long cnt = ((StoredEntry)e.getValue()).count();
            if (cnt <= 0L) continue;
            AEKey k = (AEKey)e.getKey();
            AEItemKey aek = ((StoredEntry)e.getValue()).aeKey();
            if (aek == null) continue;
            result.add(new AEKeyCacheEntry(k, aek, cnt));
        }
        return result;
    }

    public static long getTotalItemCount(String scopePrefix, int frequency) {
        String key = scopePrefix + "|" + frequency;
        CachedCount cached = itemCountCache.get(key);
        long now = System.currentTimeMillis();
        if (cached == null || now - cached.timestamp >= 1000L) {
            long newCount = EnderDBManager.calculateTotalItemCount(scopePrefix, frequency);
            itemCountCache.put(key, new CachedCount(newCount, now));
            EnderDBManager.log("[getTotalItemCount] Recalculated item count: scope=%s freq=%d total=%d", scopePrefix, frequency, newCount);
            return newCount;
        }
        EnderDBManager.log("[getTotalItemCount] Using cached item count: scope=%s freq=%d total=%d", scopePrefix, frequency, cached.count);
        return cached.count;
    }

    private static long calculateTotalItemCount(String scopePrefix, int frequency) {
        List<AEKeyCacheEntry> entries = EnderDBManager.queryItemsByFrequency(scopePrefix, frequency);
        try {
            return (Long)((ForkJoinTask)SHARED_PARALLEL_POOL.submit(() -> entries.parallelStream().mapToLong(AEKeyCacheEntry::count).sum())).get();
        }
        catch (Exception e) {
            e.printStackTrace();
            return 0L;
        }
    }

    public static List<ItemStack> getTopStacks(String scopePrefix, int frequency, int max) {
        List<AEKeyCacheEntry> entries = EnderDBManager.queryItemsByFrequency(scopePrefix, frequency);
        try {
            return (List)((ForkJoinTask)SHARED_PARALLEL_POOL.submit(() -> ((Stream)entries.stream().sorted(Comparator.comparingLong(AEKeyCacheEntry::count).reversed()).limit(max).parallel()).map(e -> e.aeKey().toStack((int)Math.min(e.count(), Integer.MAX_VALUE))).toList())).get();
        }
        catch (Exception e) {
            e.printStackTrace();
            return Collections.emptyList();
        }
    }

    public static void commitDatabase() {
        try {
            File temp = new File(dbFile.getAbsolutePath() + ".tmp");
            try (DataOutputStream dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(temp), 524288));){
                Properties props = new Properties();
                String ver = "undefined";
                try (InputStream in = EnderDBManager.class.getResourceAsStream("/mod_version.properties");){
                    if (in != null) {
                        props.load(in);
                        ver = props.getProperty("mod.version");
                    }
                }
                dos.writeUTF("EDB1");
                dos.writeUTF(ver);
                dos.writeInt(1);
                dos.writeLong(System.currentTimeMillis());
                List<byte[]> recs = EnderDBManager.parallelCall(() -> dbMap.entrySet().parallelStream().map(e -> {
                    try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
                        byte[] byArray;
                        try (DataOutputStream tmp = new DataOutputStream(baos);){
                            AEKey k = (AEKey)e.getKey();
                            tmp.writeUTF(k.scope());
                            tmp.writeInt(k.freq());
                            tmp.writeInt(k.itemBytes().length);
                            tmp.write(k.itemBytes());
                            tmp.writeLong(((StoredEntry)e.getValue()).count());
                            byArray = baos.toByteArray();
                        }
                        return byArray;
                    }
                    catch (IOException ex) {
                        ex.printStackTrace();
                        return null;
                    }
                }).filter(Objects::nonNull).toList(), List.of());
                for (byte[] r : recs) {
                    dos.write(r);
                }
            }
            Files.move(temp.toPath(), dbFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            dirty = false;
            EnderDBManager.log("[commitDatabase] Database committed successfully.", new Object[0]);
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static AtomicLong getTotalItemsWritten() {
        return totalItemsWritten;
    }

    public static AtomicLong getTotalCommits() {
        return totalCommits;
    }

    public static int getDatabaseSize() {
        return dbMap.size();
    }

    public static long getDatabaseFileSizeBytes() {
        return dbFile.exists() ? dbFile.length() : 0L;
    }

    private static void startBackgroundCommit() {
        if (commitThread != null && commitThread.isAlive()) {
            EnderDBManager.log("[startBackgroundCommit] Commit thread already running; skipping new launch.", new Object[0]);
            return;
        }
        running = true;
        commitThread = new Thread(() -> {
            EnderDBManager.log("[startBackgroundCommit] Background WAL commit thread starting...", new Object[0]);
            try {
                int WAL_BATCH_SIZE = 100;
                long nextWalTime = System.currentTimeMillis() + MIN_WAL_COMMIT_MS;
                long nextDbTime = System.currentTimeMillis() + MIN_DB_COMMIT_MS;
                while (running) {
                    try {
                        Object object;
                        boolean queueThresholdMet;
                        long now = System.currentTimeMillis();
                        int queueSize = walQueue.size();
                        boolean timeToFlushWAL = now >= nextWalTime;
                        boolean bl = queueThresholdMet = queueSize >= 100;
                        if ((timeToFlushWAL || queueThresholdMet) && queueSize > 0) {
                            object = commitLock;
                            synchronized (object) {
                                ArrayList batch = new ArrayList();
                                walQueue.drainTo(batch, 100);
                                if (!batch.isEmpty()) {
                                    EnderDBManager.log("[startBackgroundCommit] WAL flush: entries={}, (time={}, threshold={})", batch.size(), timeToFlushWAL, queueThresholdMet);
                                    for (byte[] rec : batch) {
                                        walWriter.writeInt(rec.length);
                                        walWriter.write(rec);
                                        walWriter.writeLong(EnderDBManager.checksum(rec));
                                        totalItemsWritten.incrementAndGet();
                                    }
                                    walWriter.flush();
                                    lastWalCommitTime = now;
                                    EnderDBManager.log("[startBackgroundCommit] WAL flushed, totalItemsWritten={}", totalItemsWritten.get());
                                }
                            }
                            nextWalTime = now + MIN_WAL_COMMIT_MS;
                        }
                        if (dirty && now >= nextDbTime) {
                            object = commitLock;
                            synchronized (object) {
                                EnderDBManager.log("[startBackgroundCommit] DB checkpoint: entries={} dirty={}", dbMap.size(), dirty);
                                EnderDBManager.commitDatabase();
                                EnderDBManager.truncateCurrentWAL();
                                lastDbCommitTime = now;
                                totalCommits.incrementAndGet();
                                dirty = false;
                                EnderDBManager.log("[startBackgroundCommit] DB committed, totalCommits={}", totalCommits.get());
                            }
                            nextDbTime = now + MIN_DB_COMMIT_MS;
                        }
                        Thread.sleep(100L);
                    }
                    catch (Exception e) {
                        LOGGER.error("Background commit error", (Throwable)e);
                    }
                }
            }
            catch (Exception e) {
                LOGGER.error("[EnderDB] WAL Commit thread crashed", (Throwable)e);
            }
        }, "EnderDB-CommitThread");
        commitThread.setDaemon(true);
        commitThread.start();
    }

    private static void applyBinaryOperation(byte[] data) {
        try (DataInputStream dis = new DataInputStream(new ByteArrayInputStream(data));){
            String scopePrefix = dis.readUTF();
            int freq = dis.readInt();
            int keyLen = dis.readInt();
            byte[] keyBytes = new byte[keyLen];
            dis.readFully(keyBytes);
            long delta = dis.readLong();
            AEKey key = new AEKey(scopePrefix, freq, keyBytes);
            long oldVal = dbMap.getOrDefault(key, new StoredEntry(0L, null)).count();
            long newVal = oldVal + delta;
            if (newVal <= 0L) {
                dbMap.remove(key);
            } else {
                AEItemKey aeKey = null;
                try {
                    ItemStack stack = EnderDiskInventory.deserializeItemStackFromBytes(keyBytes);
                    if (!stack.isEmpty()) {
                        aeKey = AEItemKey.of((ItemStack)stack);
                    }
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
                if (aeKey != null) {
                    dbMap.put(key, new StoredEntry(newVal, aeKey));
                } else {
                    dbMap.put(key, new StoredEntry(newVal, null));
                }
            }
            EnderDBManager.log("Applying WAL: key=%s delta=%d old=%d new=%d", key, delta, oldVal, newVal);
            dirty = true;
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void truncateCurrentWAL() {
        try {
            EnderDBManager.closeWALStream();
            new FileOutputStream(currentWAL).close();
            EnderDBManager.openWALStream();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static void flushWALQueue() {
        if (walQueue.isEmpty()) {
            return;
        }
        try {
            EnderDBManager.log("[FlushWALQueue] Flushing WAL queue with {} entries on shutdown", walQueue.size());
            ArrayList batch = new ArrayList();
            walQueue.drainTo(batch);
            for (byte[] rec : batch) {
                walWriter.writeInt(rec.length);
                walWriter.write(rec);
                walWriter.writeLong(EnderDBManager.checksum(rec));
            }
            walWriter.flush();
        }
        catch (IOException e) {
            LOGGER.error("Error flushing WAL queue during shutdown", (Throwable)e);
        }
    }

    private static void replayWALs() throws IOException {
        File dir;
        Object[] rotatedWALs;
        if (currentWAL.exists()) {
            EnderDBManager.replayAndDeleteWAL(currentWAL);
        }
        if ((rotatedWALs = (dir = currentWAL.getParentFile()).listFiles((d, name) -> name.startsWith("enderdrives.wal.") && name.matches(".*\\.\\d+$"))) != null) {
            Arrays.sort(rotatedWALs);
            for (Object rotated : rotatedWALs) {
                EnderDBManager.replayAndDeleteWAL((File)rotated);
            }
        }
        EnderDBManager.truncateCurrentWAL();
    }

    private static void replayAndDeleteWAL(File walFile) {
        try {
            DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(walFile)));
            try {
                try {
                    while (true) {
                        int length = dis.readInt();
                        byte[] data = new byte[length];
                        dis.readFully(data);
                        long storedChecksum = dis.readLong();
                        if (EnderDBManager.checksum(data) != storedChecksum) {
                            EnderDBManager.log("Checksum mismatch for record in %s", walFile.getName());
                            continue;
                        }
                        EnderDBManager.log("Replaying record from %s: data length=%d", walFile.getName(), length);
                        EnderDBManager.applyBinaryOperation(data);
                    }
                }
                catch (EOFException eof) {
                    if (walFile.delete()) {
                        EnderDBManager.log("Deleted WAL file %s", walFile.getName());
                    } else {
                        EnderDBManager.log("Failed to delete WAL file %s", walFile.getName());
                    }
                    dis.close();
                }
            }
            catch (Throwable throwable) {
                try {
                    dis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void openWALStream() throws IOException {
        if (currentWAL == null) {
            throw new IllegalStateException("currentWAL file is not set!");
        }
        walFileStream = new FileOutputStream(currentWAL, true);
        walWriter = new DataOutputStream(new BufferedOutputStream(walFileStream));
    }

    private static void closeWALStream() throws IOException {
        if (walWriter != null) {
            walWriter.close();
            walWriter = null;
        }
        if (walFileStream != null) {
            walFileStream.close();
            walFileStream = null;
        }
    }

    private static void loadDatabase() throws IOException {
        if (!dbFile.exists() || dbFile.length() == 0L) {
            return;
        }
        Properties props = new Properties();
        String curVer = "undefined";
        try (InputStream in = EnderDBManager.class.getResourceAsStream("/mod_version.properties");){
            if (in != null) {
                props.load(in);
                curVer = props.getProperty("mod.version");
            }
        }
        try {
            DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(dbFile)));
            try {
                dis.mark(128);
                boolean hasHeader = false;
                String header = dis.readUTF();
                if ("EDB1".equals(header)) {
                    hasHeader = true;
                    String fileVer = dis.readUTF();
                    int fmt = dis.readInt();
                    long ts = dis.readLong();
                    EnderDBManager.log("Loaded EDB1 header ver={} fmt={} ts={}", fileVer, fmt, new Date(ts));
                    if (!fileVer.equals(curVer)) {
                        EnderDBManager.backupDatabaseFile(fileVer);
                    }
                } else {
                    dis.reset();
                    EnderDBManager.backupDatabaseFile("0.0.0");
                }
                while (true) {
                    String scope = dis.readUTF();
                    int freq = dis.readInt();
                    int len = dis.readInt();
                    byte[] key = new byte[len];
                    dis.readFully(key);
                    long count = dis.readLong();
                    AEItemKey aek = null;
                    try {
                        ItemStack s = EnderDiskInventory.deserializeItemStackFromBytes(key);
                        if (!s.isEmpty()) {
                            aek = AEItemKey.of((ItemStack)s);
                        }
                    }
                    catch (Exception x) {
                        x.printStackTrace();
                    }
                    dbMap.put(new AEKey(scope, freq, key), new StoredEntry(count, aek));
                }
            }
            catch (Throwable throwable) {
                try {
                    dis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (EOFException eOFException) {
            return;
        }
    }

    private static long checksum(byte[] data) {
        CRC32 crc = new CRC32();
        crc.update(data);
        return crc.getValue();
    }

    private static void log(String format, Object ... args) {
        if (DEBUG_LOG) {
            LOGGER.info("[EnderDBManager] " + format, args);
        }
    }

    private static void migrateOldRecords() {
        List<Map.Entry> toMigrate = EnderDBManager.parallelCall(() -> dbMap.entrySet().parallelStream().filter(entry -> {
            String scope = ((AEKey)entry.getKey()).scope();
            return scope == null || scope.isEmpty() || !scope.matches("^[a-z]+_[a-z0-9\\-]+$") && !scope.equals("global");
        }).toList(), List.of());
        if (toMigrate.isEmpty()) {
            return;
        }
        EnderDBManager.log("[migrateOldRecords] Detected {} old-format records. Migrating to global scope...", toMigrate.size());
        for (Map.Entry entry : toMigrate) {
            AEKey oldKey = (AEKey)entry.getKey();
            StoredEntry value = (StoredEntry)entry.getValue();
            AEKey newKey = new AEKey("global", oldKey.freq(), oldKey.itemBytes());
            long existing = dbMap.getOrDefault(newKey, new StoredEntry(0L, null)).count();
            dbMap.put(newKey, new StoredEntry(existing + value.count(), null));
            dbMap.remove(oldKey);
        }
        dirty = true;
    }

    private static <T> T parallelCall(Callable<T> task, T fallback) {
        try {
            return (T)((ForkJoinTask)SHARED_PARALLEL_POOL.submit((Callable)task)).get();
        }
        catch (Exception e) {
            e.printStackTrace();
            return fallback;
        }
    }

    private static void backupDatabaseFile(String version) {
        String timestamp = LocalDateTime.now().toString().replace(":", "-");
        String backupName = String.format("enderdrives_%s_%s.zip", version, timestamp);
        File backupZip = new File(dbFile.getParent(), backupName);
        try (ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(backupZip));){
            EnderDBManager.zipFile(dbFile, zos);
            if (currentWAL != null && currentWAL.exists()) {
                EnderDBManager.zipFile(currentWAL, zos);
            }
            LOGGER.info("Backed up existing database to {} due to mod version change.", (Object)backupZip.getName());
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private static void zipFile(File file, ZipOutputStream zos) throws IOException {
        try (FileInputStream fis = new FileInputStream(file);){
            ZipEntry entry = new ZipEntry(file.getName());
            zos.putNextEntry(entry);
            fis.transferTo(zos);
            zos.closeEntry();
        }
    }

    static {
        commitLock = new Object();
        running = true;
        dirty = false;
        commitThread = null;
        totalItemsWritten = new AtomicLong(0L);
        totalCommits = new AtomicLong(0L);
        SHARED_PARALLEL_POOL = new ForkJoinPool(Math.min(4, Runtime.getRuntime().availableProcessors()));
        MERGE_BUFFER_THRESHOLD = (Integer)serverConfig.END_DB_MERGE_BUFFER_THRESHOLD.get();
        MIN_WAL_COMMIT_MS = ((Integer)serverConfig.END_DB_MIN_COMMIT_INTERVAL_MS.get()).intValue();
        MAX_WAL_COMMIT_MS = ((Integer)serverConfig.END_DB_MAX_COMMIT_INTERVAL_MS.get()).intValue();
        MIN_DB_COMMIT_MS = ((Integer)serverConfig.END_DB_MIN_DB_COMMIT_INTERVAL_MS.get()).intValue();
        MAX_DB_COMMIT_MS = ((Integer)serverConfig.END_DB_MAX_DB_COMMIT_INTERVAL_MS.get()).intValue();
        DEBUG_LOG = (Boolean)serverConfig.END_DB_DEBUG_LOG.get();
        lastWalCommitTime = System.currentTimeMillis();
        lastDbCommitTime = System.currentTimeMillis();
    }

    private record CachedCount(long count, long timestamp) {
    }
}

