/*
 * Decompiled with CFR 0.152.
 */
package net.poe.entitylootdrops.lootdrops.config;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonSyntaxException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import net.poe.entitylootdrops.lootdrops.config.EventConfig;
import net.poe.entitylootdrops.lootdrops.config.ExampleFileGenerator;
import net.poe.entitylootdrops.lootdrops.config.LootConfigManager;
import net.poe.entitylootdrops.lootdrops.config.WorkingJsonGenerator;
import net.poe.entitylootdrops.lootdrops.events.LootEventManager;
import net.poe.entitylootdrops.lootdrops.model.CustomDropEntry;
import net.poe.entitylootdrops.lootdrops.model.EntityDropEntry;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class LootConfigLoader {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String CONFIG_DIR = "config/EntityLootDrops";
    private static final String LOOT_DROPS_DIR = "Loot Drops";
    private static final String NORMAL_DROPS_DIR = "Normal Drops";
    private static final String EVENT_DROPS_DIR = "Event Drops";
    private static final String MOBS_DIR = "Mobs";
    private static final String MESSAGES_FILE = "messages.json";
    private static final String[] EVENT_TYPES = new String[]{"Winter", "Summer", "Easter", "Halloween"};
    private final LootConfigManager configManager;
    private final LootEventManager eventManager;

    public LootConfigLoader(LootConfigManager configManager, LootEventManager eventManager) {
        this.configManager = configManager;
        this.eventManager = eventManager;
    }

    public void loadConfig() {
        LOGGER.info("Starting loot configuration loading...");
        HashSet<String> previousActiveEvents = new HashSet<String>(this.eventManager.getActiveEvents());
        boolean previousDropChanceState = this.eventManager.isDropChanceEventActive();
        boolean previousDoubleDropsState = this.eventManager.isDoubleDropsActive();
        boolean previousDebugState = this.eventManager.isDebugLoggingEnabled();
        this.configManager.clearConfigurations();
        this.createConfigDirectories();
        this.createExampleFiles();
        try {
            EventConfig.loadConfig();
            LOGGER.info("EventConfig loaded successfully");
        }
        catch (Exception e) {
            LOGGER.error("Failed to load EventConfig", (Throwable)e);
        }
        try {
            this.eventManager.createActiveEventsFile();
            LOGGER.info("Active_Events.json created/verified");
        }
        catch (Exception e) {
            LOGGER.error("Failed to create Active_Events.json", (Throwable)e);
        }
        boolean activeEventsLoaded = false;
        try {
            this.eventManager.loadActiveEventsState();
            activeEventsLoaded = true;
            LOGGER.info("Active events loaded from Active_Events.json");
        }
        catch (Exception e) {
            LOGGER.error("Failed to load active events from file", (Throwable)e);
        }
        this.loadAllDrops();
        this.loadMessages();
        if (!activeEventsLoaded) {
            this.eventManager.restorePreviousState(previousActiveEvents, previousDropChanceState, previousDoubleDropsState, previousDebugState);
            LOGGER.info("Used previous state as fallback since no active events file was loaded");
        }
    }

    private void createConfigDirectories() {
        LOGGER.info("Creating configuration directories...");
        try {
            Path lootDropsDir = Paths.get(CONFIG_DIR, LOOT_DROPS_DIR);
            Files.createDirectories(lootDropsDir, new FileAttribute[0]);
            Path normalDropsDir = lootDropsDir.resolve(NORMAL_DROPS_DIR);
            Files.createDirectories(normalDropsDir, new FileAttribute[0]);
            Files.createDirectories(normalDropsDir.resolve(MOBS_DIR), new FileAttribute[0]);
            Path eventDropsDir = lootDropsDir.resolve(EVENT_DROPS_DIR);
            Files.createDirectories(eventDropsDir, new FileAttribute[0]);
            for (String eventType : EVENT_TYPES) {
                Path eventDir = eventDropsDir.resolve(eventType);
                Files.createDirectories(eventDir, new FileAttribute[0]);
                Files.createDirectories(eventDir.resolve(MOBS_DIR), new FileAttribute[0]);
            }
            LOGGER.info("Configuration directories created successfully");
        }
        catch (Exception e) {
            LOGGER.error("Failed to create configuration directories", (Throwable)e);
        }
    }

    private void createExampleFiles() {
        LOGGER.info("Checking and creating example files...");
        try {
            Path normalDropsDir = Paths.get(CONFIG_DIR, LOOT_DROPS_DIR, NORMAL_DROPS_DIR);
            if (this.isDirectoryEmptyOfJsonFiles(normalDropsDir)) {
                ExampleFileGenerator.createAllExamples();
                WorkingJsonGenerator.createAllWorkingFiles();
            }
            Path eventDropsDir = Paths.get(CONFIG_DIR, LOOT_DROPS_DIR, EVENT_DROPS_DIR);
            for (String eventType : EVENT_TYPES) {
                Path eventDir = eventDropsDir.resolve(eventType);
                Path eventMobsDir = eventDir.resolve(MOBS_DIR);
                if (!this.isDirectoryEmptyOfAnyFiles(eventMobsDir)) continue;
                WorkingJsonGenerator.createEventDropsExample(eventType);
                WorkingJsonGenerator.createEventMobsExample(eventType);
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to create example files", (Throwable)e);
        }
    }

    private boolean isDirectoryEmptyOfJsonFiles(Path directory) {
        try {
            if (!Files.exists(directory, new LinkOption[0])) {
                return true;
            }
            return Files.walk(directory, 1, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".json")).findFirst().isEmpty();
        }
        catch (Exception e) {
            LOGGER.error("Failed to check if directory is empty of JSON files: {}", (Object)directory, (Object)e);
            return false;
        }
    }

    private boolean isDirectoryEmptyOfAnyFiles(Path directory) {
        try {
            if (!Files.exists(directory, new LinkOption[0])) {
                return true;
            }
            return Files.walk(directory, 1, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).findFirst().isEmpty();
        }
        catch (Exception e) {
            LOGGER.error("Failed to check if directory is empty of any files: {}", (Object)directory, (Object)e);
            return false;
        }
    }

    private void loadAllDrops() {
        try {
            Path eventDropsDir;
            Path lootDropsDir = Paths.get(CONFIG_DIR, LOOT_DROPS_DIR);
            if (!Files.exists(lootDropsDir, new LinkOption[0])) {
                LOGGER.warn("Loot drops directory does not exist: {}", (Object)lootDropsDir);
                return;
            }
            Path normalDropsDir = lootDropsDir.resolve(NORMAL_DROPS_DIR);
            if (Files.exists(normalDropsDir, new LinkOption[0])) {
                LOGGER.info("Loading normal drops from: {}", (Object)normalDropsDir);
                this.loadDropsFromDirectory(normalDropsDir, NORMAL_DROPS_DIR);
            }
            if (Files.exists(eventDropsDir = lootDropsDir.resolve(EVENT_DROPS_DIR), new LinkOption[0])) {
                LOGGER.info("Loading event drops from: {}", (Object)eventDropsDir);
                Files.walk(eventDropsDir, 1, new FileVisitOption[0]).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).filter(path -> !path.equals(eventDropsDir)).forEach(eventDir -> {
                    String eventName = eventDir.getFileName().toString();
                    LOGGER.info("Loading event drops for: {}", (Object)eventName);
                    this.loadDropsFromDirectory((Path)eventDir, eventName);
                });
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to load drop configurations", (Throwable)e);
        }
    }

    private void loadDropsFromDirectory(Path directory, String dirKey) {
        try {
            List<EntityDropEntry> entityDrops;
            List<CustomDropEntry> generalDrops = this.loadGeneralDropsFromDirectory(directory);
            if (!generalDrops.isEmpty()) {
                this.configManager.setHostileDrops(dirKey, generalDrops);
                LOGGER.info("Loaded {} general drops for {}", (Object)generalDrops.size(), (Object)dirKey);
            }
            if (!(entityDrops = this.loadEntityDropsFromMobsDirectories(directory)).isEmpty()) {
                this.configManager.setEntityDrops(dirKey, entityDrops);
                LOGGER.info("Loaded {} entity drops for {}", (Object)entityDrops.size(), (Object)dirKey);
            }
        }
        catch (Exception e) {
            LOGGER.error("Failed to load drops from directory: {}", (Object)directory, (Object)e);
        }
    }

    private List<CustomDropEntry> loadGeneralDropsFromDirectory(Path directory) {
        ArrayList<CustomDropEntry> allDrops = new ArrayList<CustomDropEntry>();
        try {
            Files.walk(directory, 1, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".json")).filter(path -> !this.isInMobsDirectory((Path)path)).forEach(jsonFile -> {
                try {
                    String json = new String(Files.readAllBytes(jsonFile));
                    if (json.trim().isEmpty()) {
                        LOGGER.warn("Empty JSON file: {}", jsonFile);
                        return;
                    }
                    Gson gson = new Gson();
                    CustomDropEntry[] drops = (CustomDropEntry[])gson.fromJson(json, CustomDropEntry[].class);
                    if (drops != null) {
                        for (CustomDropEntry drop : drops) {
                            if (drop == null) continue;
                            allDrops.add(drop);
                        }
                    }
                }
                catch (Exception e) {
                    LOGGER.error("Failed to load general drops from file: {}", jsonFile, (Object)e);
                }
            });
        }
        catch (Exception e) {
            LOGGER.error("Failed to walk directory for general drops: {}", (Object)directory, (Object)e);
        }
        return allDrops;
    }

    private List<EntityDropEntry> loadEntityDropsFromMobsDirectories(Path categoryDir) {
        ArrayList<EntityDropEntry> allEntityDrops = new ArrayList<EntityDropEntry>();
        try {
            LOGGER.info("Looking for Mobs directories in: {}", (Object)categoryDir);
            Files.walk(categoryDir, new FileVisitOption[0]).filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).filter(path -> path.getFileName().toString().equals(MOBS_DIR)).forEach(mobsDir -> {
                LOGGER.info("Found Mobs directory: {}", mobsDir);
                try {
                    Files.walk(mobsDir, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".json")).forEach(jsonFile -> {
                        LOGGER.info("Processing entity drops file: {}", jsonFile);
                        try {
                            String json = new String(Files.readAllBytes(jsonFile));
                            if (json.trim().isEmpty()) {
                                LOGGER.warn("Empty JSON file: {}", jsonFile);
                                return;
                            }
                            Gson gson = new Gson();
                            EntityDropEntry[] drops = (EntityDropEntry[])gson.fromJson(json, EntityDropEntry[].class);
                            if (drops != null) {
                                for (EntityDropEntry drop : drops) {
                                    if (drop == null || drop.getEntityId() == null) continue;
                                    allEntityDrops.add(drop);
                                    LOGGER.info("Loaded entity drop for {} from file: {}", (Object)drop.getEntityId(), jsonFile);
                                }
                            } else {
                                LOGGER.warn("Failed to parse JSON or got null from file: {}", jsonFile);
                            }
                        }
                        catch (Exception e) {
                            LOGGER.error("Failed to load entity drops from file: {}", jsonFile, (Object)e);
                        }
                    });
                }
                catch (Exception e) {
                    LOGGER.error("Failed to walk Mobs directory: {}", mobsDir, (Object)e);
                }
            });
        }
        catch (Exception e) {
            LOGGER.error("Failed to find Mobs directories in: {}", (Object)categoryDir, (Object)e);
        }
        LOGGER.info("Total entity drops loaded: {}", (Object)allEntityDrops.size());
        return allEntityDrops;
    }

    private boolean isInMobsDirectory(Path path) {
        return path.toString().contains(MOBS_DIR);
    }

    private void loadMessages() {
        try {
            String json;
            Path messagesFile = Paths.get(CONFIG_DIR, LOOT_DROPS_DIR, MESSAGES_FILE);
            if (!Files.exists(messagesFile, new LinkOption[0])) {
                LOGGER.info("Messages file does not exist, creating default: {}", (Object)messagesFile);
                this.createDefaultMessagesFile(messagesFile);
            }
            if ((json = Files.readString(messagesFile, StandardCharsets.UTF_8)).trim().isEmpty()) {
                LOGGER.warn("Messages file is empty: {}", (Object)messagesFile);
                return;
            }
            LOGGER.debug("Loading messages from: {}", (Object)messagesFile);
            LOGGER.debug("JSON content: {}", (Object)json);
            try {
                String message;
                String message2;
                JsonElement value;
                String eventName;
                JsonElement jsonElement = JsonParser.parseString((String)json);
                if (!jsonElement.isJsonObject()) {
                    LOGGER.error("Messages file must contain a JSON object, but found: {}", (Object)jsonElement.getClass().getSimpleName());
                    return;
                }
                JsonObject rootObject = jsonElement.getAsJsonObject();
                if (rootObject.has("enable") && rootObject.get("enable").isJsonObject()) {
                    JsonObject enableMessages = rootObject.getAsJsonObject("enable");
                    for (Map.Entry entry : enableMessages.entrySet()) {
                        eventName = (String)entry.getKey();
                        value = (JsonElement)entry.getValue();
                        if (!value.isJsonPrimitive() || !value.getAsJsonPrimitive().isString()) continue;
                        message2 = value.getAsString();
                        this.eventManager.setEventEnableMessage(eventName, message2);
                        LOGGER.info("Loaded enable message for {}: {}", (Object)eventName, (Object)message2);
                    }
                }
                if (rootObject.has("disable") && rootObject.get("disable").isJsonObject()) {
                    JsonObject disableMessages = rootObject.getAsJsonObject("disable");
                    for (Map.Entry entry : disableMessages.entrySet()) {
                        eventName = (String)entry.getKey();
                        value = (JsonElement)entry.getValue();
                        if (value.isJsonPrimitive() && value.getAsJsonPrimitive().isString()) {
                            message2 = value.getAsString();
                            this.eventManager.setEventDisableMessage(eventName, message2);
                            LOGGER.info("Loaded disable message for {}: {}", (Object)eventName, (Object)message2);
                            continue;
                        }
                        LOGGER.warn("Skipping invalid disable message for '{}': expected string but found {}", (Object)eventName, (Object)value.getClass().getSimpleName());
                    }
                }
                if (rootObject.has("drop_chance_enable") && rootObject.get("drop_chance_enable").isJsonPrimitive()) {
                    message = rootObject.get("drop_chance_enable").getAsString();
                    this.eventManager.setDropChanceEnableMessage(message);
                    LOGGER.info("Loaded drop chance enable message: {}", (Object)message);
                }
                if (rootObject.has("drop_chance_disable") && rootObject.get("drop_chance_disable").isJsonPrimitive()) {
                    message = rootObject.get("drop_chance_disable").getAsString();
                    this.eventManager.setDropChanceDisableMessage(message);
                    LOGGER.info("Loaded drop chance disable message: {}", (Object)message);
                }
                if (rootObject.has("double_drops_enable") && rootObject.get("double_drops_enable").isJsonPrimitive()) {
                    message = rootObject.get("double_drops_enable").getAsString();
                    this.eventManager.setDoubleDropsEnableMessage(message);
                    LOGGER.info("Loaded double drops enable message: {}", (Object)message);
                }
                if (rootObject.has("double_drops_disable") && rootObject.get("double_drops_disable").isJsonPrimitive()) {
                    message = rootObject.get("double_drops_disable").getAsString();
                    this.eventManager.setDoubleDropsDisableMessage(message);
                    LOGGER.info("Loaded double drops disable message: {}", (Object)message);
                }
            }
            catch (JsonSyntaxException e) {
                LOGGER.error("Invalid JSON syntax in messages file: {}", (Object)messagesFile, (Object)e);
            }
            catch (Exception e) {
                LOGGER.error("Failed to parse messages from file: {}", (Object)messagesFile, (Object)e);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to read messages file: {}", (Object)e.getMessage());
        }
        catch (Exception e) {
            LOGGER.error("Unexpected error while loading messages", (Throwable)e);
        }
    }

    private void createDefaultMessagesFile(Path messagesFile) throws IOException {
        String defaultContent = "{\n  \"enable\": {\n    \"Easter\": \"\u00a7a[Events] \u00a7aEaster Event\u00a7f has been \u00a7aenabled!\",\n    \"Winter\": \"\u00a7a[Events] \u00a7aWinter Event\u00a7f has been \u00a7aenabled!\",\n    \"Summer\": \"\u00a7a[Events] \u00a7aSummer Event\u00a7f has been \u00a7aenabled!\",\n    \"Halloween\": \"\u00a7a[Events] \u00a7aHalloween Event\u00a7f has been \u00a7aenabled!\"\n  },\n  \"disable\": {\n    \"Easter\": \"\u00a7c[Events] \u00a7aEaster Event\u00a7f has been \u00a7cdisabled!\",\n    \"Winter\": \"\u00a7c[Events] \u00a7aWinter Event\u00a7f has been \u00a7cdisabled!\",\n    \"Summer\": \"\u00a7c[Events] \u00a7aSummer Event\u00a7f has been \u00a7cdisabled!\",\n    \"Halloween\": \"\u00a7c[Events] \u00a7aHalloween Event\u00a7f has been \u00a7cdisabled!\"\n  },\n  \"drop_chance_enable\": \"\u00a7a[Events] \u00a7aDouble Drop Chance Event\u00a7f has been \u00a7aenabled! \u00a76x2x drop rates\",\n  \"drop_chance_disable\": \"\u00a7c[Events] \u00a7aDouble Drop Chance Event\u00a7f has been \u00a7cdisabled!\",\n  \"double_drops_enable\": \"\u00a7a[Events] \u00a7aDouble Drops Event\u00a7f has been \u00a7aenabled! \u00a76x2x drop amounts\",\n  \"double_drops_disable\": \"\u00a7c[Events] \u00a7aDouble Drops Event\u00a7f has been \u00a7cdisabled!\"\n}";
        Files.writeString(messagesFile, (CharSequence)defaultContent, StandardCharsets.UTF_8, new OpenOption[0]);
    }
}

