/*
 * Decompiled with CFR 0.152.
 */
package net.poe.entityprotector.config;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import net.poe.entityprotector.config.EntityConfig;
import net.poe.entityprotector.config.GlobalNeutralConfig;
import net.poe.entityprotector.config.TamedEntityConfig;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class ConfigurationManager {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final String ENTITIES_CONFIG_DIR = "config/EntityProtector/Entities/";
    private static final String TAMED_CONFIG_DIR = "config/EntityProtector/Tamed_Entities/";
    private static final String GLOBAL_CONFIG_FILE = "config/EntityProtector/Global_Mobs.json";
    private static final Map<String, EntityConfig> entityConfigs = new HashMap<String, EntityConfig>();
    private static final Map<String, TamedEntityConfig> tamedEntityConfigs = new HashMap<String, TamedEntityConfig>();
    private static GlobalNeutralConfig globalNeutralConfig = new GlobalNeutralConfig();

    public static void initialize() {
        try {
            Files.createDirectories(Paths.get(ENTITIES_CONFIG_DIR, new String[0]), new FileAttribute[0]);
            Files.createDirectories(Paths.get(TAMED_CONFIG_DIR, new String[0]), new FileAttribute[0]);
            Files.createDirectories(Paths.get("config/EntityProtector/", new String[0]), new FileAttribute[0]);
            ConfigurationManager.loadAllConfigs();
            ConfigurationManager.loadGlobalNeutralConfig();
            if (entityConfigs.isEmpty()) {
                ConfigurationManager.createExampleEntityConfig();
            }
            if (tamedEntityConfigs.isEmpty()) {
                ConfigurationManager.createExampleTamedConfig();
            }
            if (!Files.exists(Paths.get(GLOBAL_CONFIG_FILE, new String[0]), new LinkOption[0])) {
                ConfigurationManager.createExampleGlobalConfig();
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to initialize config directories", (Throwable)e);
        }
    }

    private static void createExampleEntityConfig() {
        EntityConfig protectedExample = new EntityConfig("minecraft:zombie");
        protectedExample.setTakeNormalDamage(false);
        protectedExample.setTakeMobDamage(false);
        protectedExample.setTakeGenericSpellDamage(false);
        protectedExample.setTakeMineSlashSpellDamage(false);
        protectedExample.setTakeIronSpellDamage(false);
        protectedExample.setCanDamagePlayer(true);
        protectedExample.setCancelAggroTime(5);
        protectedExample.setNeutralUntilAttacked(false);
        ConfigurationManager.saveEntityConfig(protectedExample);
        EntityConfig neutralExample = new EntityConfig("minecraft:skeleton");
        neutralExample.setTakeNormalDamage(true);
        neutralExample.setTakeMobDamage(true);
        neutralExample.setTakeGenericSpellDamage(true);
        neutralExample.setTakeMineSlashSpellDamage(true);
        neutralExample.setTakeIronSpellDamage(true);
        neutralExample.setCanDamagePlayer(true);
        neutralExample.setCancelAggroTime(0);
        neutralExample.setNeutralUntilAttacked(true);
        ConfigurationManager.saveEntityConfig(neutralExample);
    }

    private static void createExampleTamedConfig() {
        TamedEntityConfig enabledExample = new TamedEntityConfig("minecraft:wolf");
        enabledExample.setUseWolfLikeAI(true);
        ConfigurationManager.saveTamedEntityConfig(enabledExample);
        TamedEntityConfig disabledExample = new TamedEntityConfig("irons_spellbooks:summoned_frog");
        disabledExample.setUseWolfLikeAI(false);
        ConfigurationManager.saveTamedEntityConfig(disabledExample);
    }

    private static void createExampleGlobalConfig() {
        GlobalNeutralConfig globalConfig = new GlobalNeutralConfig();
        globalConfig.setEnabled(false);
        globalConfig.setTakeNormalDamage(true);
        globalConfig.setTakeMobDamage(true);
        globalConfig.setTakeGenericSpellDamage(true);
        globalConfig.setTakeMineSlashSpellDamage(true);
        globalConfig.setTakeIronSpellDamage(true);
        globalConfig.setCanDamagePlayer(true);
        globalConfig.setCancelAggroTime(0);
        globalConfig.setNeutralUntilAttacked(true);
        globalConfig.setExcludeEntitys(Arrays.asList("minecraft:zombie", "modid:entityid"));
        ConfigurationManager.saveGlobalNeutralConfig(globalConfig);
        LOGGER.info("Created example global config at: {}", (Object)GLOBAL_CONFIG_FILE);
    }

    public static void loadAllConfigs() {
        entityConfigs.clear();
        tamedEntityConfigs.clear();
        try {
            Files.walk(Paths.get(ENTITIES_CONFIG_DIR, new String[0]), new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".json")).forEach(ConfigurationManager::loadEntityConfig);
        }
        catch (IOException e) {
            LOGGER.error("Failed to load entity configs", (Throwable)e);
        }
        try {
            Files.walk(Paths.get(TAMED_CONFIG_DIR, new String[0]), new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(path -> path.toString().endsWith(".json")).forEach(ConfigurationManager::loadTamedConfig);
        }
        catch (IOException e) {
            LOGGER.error("Failed to load tamed entity configs", (Throwable)e);
        }
    }

    private static void loadGlobalNeutralConfig() {
        try {
            Path configPath = Paths.get(GLOBAL_CONFIG_FILE, new String[0]);
            if (Files.exists(configPath, new LinkOption[0])) {
                String content = new String(Files.readAllBytes(configPath));
                globalNeutralConfig = (GlobalNeutralConfig)GSON.fromJson(content, GlobalNeutralConfig.class);
                LOGGER.info("Loaded global config - Enabled: {}, Excluded entities: {}", (Object)globalNeutralConfig.isEnabled(), (Object)globalNeutralConfig.getExcludeEntitys().size());
            } else {
                LOGGER.info("Global config not found, using defaults");
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to load global config", (Throwable)e);
            globalNeutralConfig = new GlobalNeutralConfig();
        }
        catch (Exception e) {
            LOGGER.error("Failed to parse global config JSON", (Throwable)e);
            globalNeutralConfig = new GlobalNeutralConfig();
        }
    }

    public static void saveGlobalNeutralConfig(GlobalNeutralConfig config) {
        try {
            Path configPath = Paths.get(GLOBAL_CONFIG_FILE, new String[0]);
            String json = GSON.toJson((Object)config);
            Files.write(configPath, json.getBytes(), new OpenOption[0]);
            globalNeutralConfig = config;
            LOGGER.info("Saved global config");
        }
        catch (IOException e) {
            LOGGER.error("Failed to save global config", (Throwable)e);
        }
    }

    private static void loadEntityConfig(Path path) {
        try {
            String content = new String(Files.readAllBytes(path));
            EntityConfig config = (EntityConfig)GSON.fromJson(content, EntityConfig.class);
            if (config != null && config.getEntityId() != null) {
                entityConfigs.put(config.getEntityId(), config);
                LOGGER.debug("Loaded entity config for: {}", (Object)config.getEntityId());
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to load entity config: " + String.valueOf(path), (Throwable)e);
        }
        catch (Exception e) {
            LOGGER.error("Failed to parse entity config JSON: " + String.valueOf(path), (Throwable)e);
        }
    }

    private static void loadTamedConfig(Path path) {
        try {
            String content = new String(Files.readAllBytes(path));
            TamedEntityConfig config = (TamedEntityConfig)GSON.fromJson(content, TamedEntityConfig.class);
            if (config != null && config.getEntityId() != null) {
                tamedEntityConfigs.put(config.getEntityId(), config);
                LOGGER.debug("Loaded tamed entity config for: {}", (Object)config.getEntityId());
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to load tamed entity config: " + String.valueOf(path), (Throwable)e);
        }
        catch (Exception e) {
            LOGGER.error("Failed to parse tamed entity config JSON: " + String.valueOf(path), (Throwable)e);
        }
    }

    public static void saveEntityConfig(EntityConfig config) {
        try {
            String fileName = config.getEntityId().replace(':', '_') + ".json";
            Path configPath = Paths.get(ENTITIES_CONFIG_DIR, fileName);
            String json = GSON.toJson((Object)config);
            Files.write(configPath, json.getBytes(), new OpenOption[0]);
            entityConfigs.put(config.getEntityId(), config);
            LOGGER.info("Saved entity config for: {}", (Object)config.getEntityId());
        }
        catch (IOException e) {
            LOGGER.error("Failed to save entity config for: " + config.getEntityId(), (Throwable)e);
        }
    }

    public static void saveTamedEntityConfig(TamedEntityConfig config) {
        try {
            String fileName = config.getEntityId().replace(':', '_') + ".json";
            Path configPath = Paths.get(TAMED_CONFIG_DIR, fileName);
            String json = GSON.toJson((Object)config);
            Files.write(configPath, json.getBytes(), new OpenOption[0]);
            tamedEntityConfigs.put(config.getEntityId(), config);
            LOGGER.info("Saved tamed entity config for: {}", (Object)config.getEntityId());
        }
        catch (IOException e) {
            LOGGER.error("Failed to save tamed entity config for: " + config.getEntityId(), (Throwable)e);
        }
    }

    public static EntityConfig getEntityConfig(String entityId) {
        return entityConfigs.get(entityId);
    }

    public static TamedEntityConfig getTamedEntityConfig(String entityId) {
        return tamedEntityConfigs.get(entityId);
    }

    public static GlobalNeutralConfig getGlobalNeutralConfig() {
        return globalNeutralConfig;
    }

    public static boolean hasConfig(String entityId) {
        return entityConfigs.containsKey(entityId);
    }

    public static boolean hasTamedConfig(String entityId) {
        return tamedEntityConfigs.containsKey(entityId);
    }

    public static boolean shouldUseWolfLikeAI(String entityId) {
        TamedEntityConfig config = tamedEntityConfigs.get(entityId);
        return config == null || config.isUseWolfLikeAI();
    }

    public static boolean shouldBeNeutralUntilAttacked(String entityId) {
        EntityConfig config = entityConfigs.get(entityId);
        if (config != null) {
            return config.isNeutralUntilAttacked();
        }
        if (globalNeutralConfig.isEnabled()) {
            if (globalNeutralConfig.getExcludeEntitys().contains(entityId)) {
                return false;
            }
            return globalNeutralConfig.isNeutralUntilAttacked();
        }
        return false;
    }

    public static boolean shouldTakeDamage(String entityId, String damageType) {
        EntityConfig config = entityConfigs.get(entityId);
        if (config != null) {
            return ConfigurationManager.getDamageSettingFromConfig(config, damageType);
        }
        if (globalNeutralConfig.isEnabled()) {
            if (globalNeutralConfig.getExcludeEntitys().contains(entityId)) {
                return true;
            }
            return ConfigurationManager.getDamageSettingFromGlobalConfig(globalNeutralConfig, damageType);
        }
        return true;
    }

    public static int getCancelAggroTime(String entityId) {
        EntityConfig config = entityConfigs.get(entityId);
        if (config != null) {
            return config.getCancelAggroTime();
        }
        if (globalNeutralConfig.isEnabled()) {
            if (globalNeutralConfig.getExcludeEntitys().contains(entityId)) {
                return 0;
            }
            return globalNeutralConfig.getCancelAggroTime();
        }
        return 0;
    }

    public static boolean canDamagePlayer(String entityId) {
        EntityConfig config = entityConfigs.get(entityId);
        if (config != null) {
            return config.isCanDamagePlayer();
        }
        if (globalNeutralConfig.isEnabled()) {
            if (globalNeutralConfig.getExcludeEntitys().contains(entityId)) {
                return true;
            }
            return globalNeutralConfig.isCanDamagePlayer();
        }
        return true;
    }

    private static boolean getDamageSettingFromConfig(EntityConfig config, String damageType) {
        switch (damageType.toLowerCase()) {
            case "normal": {
                return config.isTakeNormalDamage();
            }
            case "mob": {
                return config.isTakeMobDamage();
            }
            case "generic_spell": {
                return config.isTakeGenericSpellDamage();
            }
            case "mine_slash_spell": {
                return config.isTakeMineSlashSpellDamage();
            }
            case "iron_spell": {
                return config.isTakeIronSpellDamage();
            }
        }
        return true;
    }

    private static boolean getDamageSettingFromGlobalConfig(GlobalNeutralConfig config, String damageType) {
        switch (damageType.toLowerCase()) {
            case "normal": {
                return config.isTakeNormalDamage();
            }
            case "mob": {
                return config.isTakeMobDamage();
            }
            case "generic_spell": {
                return config.isTakeGenericSpellDamage();
            }
            case "mine_slash_spell": {
                return config.isTakeMineSlashSpellDamage();
            }
            case "iron_spell": {
                return config.isTakeIronSpellDamage();
            }
        }
        return true;
    }

    public static void reloadConfigs() {
        LOGGER.info("Reloading all configurations...");
        ConfigurationManager.loadAllConfigs();
        ConfigurationManager.loadGlobalNeutralConfig();
        LOGGER.info("Configuration reload complete. Loaded {} entity configs, {} tamed configs, global config enabled: {}", (Object)entityConfigs.size(), (Object)tamedEntityConfigs.size(), (Object)globalNeutralConfig.isEnabled());
    }

    public static boolean isEntityExcludedFromGlobal(String entityId) {
        return globalNeutralConfig.getExcludeEntitys().contains(entityId);
    }

    public static boolean isGlobalConfigEnabled() {
        return globalNeutralConfig.isEnabled();
    }

    public static void logConfigStatus() {
        LOGGER.info("=== EntityProtector Configuration Status ===");
        LOGGER.info("Entity Configs: {}", (Object)entityConfigs.size());
        LOGGER.info("Tamed Entity Configs: {}", (Object)tamedEntityConfigs.size());
        LOGGER.info("Global Config Enabled: {}", (Object)globalNeutralConfig.isEnabled());
        LOGGER.info("Global Excluded Entities: {}", globalNeutralConfig.getExcludeEntitys());
        LOGGER.info("==========================================");
    }
}

